#ifndef FREEHDL_KERNEL_ACL_H
#define FREEHDL_KERNEL_ACL_H

#include <freehdl/kernel-error.hh>
#include <stdlib.h>


#define MAX_ACL_DEPTH 100 // Max depth of acl objects
#define ACL_MARKER 0x80000000 // This const is used to store an range 
// value within an acl list and to mark the end of a list
#define ACL_RANGE ACL_MARKER


// The following macro is used to store a range value into a alc list.
// Note, are stord as 4 int values into the list. ACL_RANGE_MARKER 
// marks the start of a range value. Thereafter the left, the direction
// and the right value are stored into the list. Note further, this
// macro is based on the "<<" operator of the acl class.
#define acl_range(left, direction ,right) ACL_RANGE << left << direction << right 


class acl;
typedef acl *pacl;
extern acl *free_acl[MAX_ACL_DEPTH + 1]; // List of acl objects that are currently not in use


// acl are used to identify elements or set of elements of
// a composite object. It is simply a list of int values. The
// first element stores the maximum length and the number of items
// currently on the list. Note, an acl instance can be created 
// with "new(x)" only, where "x" is the maximum length (number of
// items) of the list.
class acl {
  struct _head { short index; short size; };
  union _data { _head head; int value; acl *next; };
  _data data;
  _head &get_header() { return (&this->data)[-1].head; }
  int get_index() const { return (&this->data)[-1].head.index; }
public:
  // Returns the number of items of an acl instance
  int get_size() const { return (&this->data)[-1].head.size; }

  // Allocate a new acl instance from the list of free acl
  // objects. 
  void *operator new(size_t, int s) {
#ifdef DEBUG
    string deb;
#endif;
    // Test parameter s
    v_assert(s >= MAX_ACL_DEPTH, "MAX_ACL_DEPTH too small");
#ifdef DEBUG
    deb=s;
    v_assert(s < 1, "Illegal acl size (" + deb + ")");
#endif
    acl *new_acl;
    if (free_acl[s] != NULL) {
      // There is an unused object in free_acl with the
      // correct length
      new_acl = free_acl[s];
      free_acl[s] = new_acl->data.next;
    } else
      // Otherwise create a new acl list. Two elementes are
      // used to mark the end of the list and one element
      // stores the length of the list. Note, the length
      // is stored in the -1'st element!
      new_acl = ((acl*)malloc((s + 3) * sizeof(_data)) + 1);

    // Mark the current as well as the absolute end of the list.
    // Note, contrary to the current end the absolute end will
    // be never overwritten!
    (&new_acl->data)[0].value = ACL_MARKER;
    (&new_acl->data)[1].value = ACL_MARKER;
    (&new_acl->data)[s].value = ACL_MARKER;
    (&new_acl->data)[s+1].value = ACL_MARKER;
    // Store the current size of the list
    new_acl->get_header().index = 0;
    new_acl->get_header().size = s;
    return new_acl;
  }

  // Removes a acl item. The unused item is inserted into
  // the free_acl array.
  void operator delete(void *a) {
    // The size of the current acl list is stored in the -1'nd
    // element
    register int s = ((acl*)a)->get_header().size;
    ((acl*)a)->data.next = free_acl[s];
    free_acl[s] = (acl*)a;
  }

  acl() { };

  // Clear the current acl list, i.e. remove all entries.
  acl &clear() {
    // Mark the current end of the list. Note that the it is not
    // necessary to reset the absolute end of the list as it is not
    // overwritten!
    (&data)[0].value = ACL_MARKER;
    (&data)[1].value = ACL_MARKER;
    // Set the size of the current size to 0
    get_header().index = 0;

    return *this;
  }


//   // Appends two acls lists. The result is stored into
//   // the current acl list. Note, no range checking is done!
//   acl &operator<<(const acl &a) {
//     const int s1 = get_index(), s2 = a.get_index();
//     // Don't forget to copy the "end of list" marker
//     memcpy(&data.value, &a.data.value, sizeof(_data) * (s2 + 2));
//     get_header().index = s1 + s2;
//     return *this;
//   }

  // Adds a single int value to the current acl list. The result 
  // is stored into the current acl list. Note, no range checking 
  // is done!
  acl &operator<<(const int i) {
    ((int*)&data.value)[get_header().index++] = i;
    // Mark end of list
    ((int*)&data.value)[get_header().index + 1] = ACL_MARKER;
    return *this;
  }

//   // Returns a new acl instance which is intialized by appending 
//   // the current acl list with "a".
//   acl &operator+(const acl &a) {
//     acl *new_acl = new(get_index() + a.get_index()) acl;
//     memcpy(&new_acl->data.value, &data.value, sizeof(int) * get_header().index);
//     memcpy(&((int*)&new_acl->data.value)[get_index()], 
// 	   &a.data.value, sizeof(int) * (a.get_index() + 2));
//     new_acl->get_header().index = get_index() + a.get_index();
//     return *new_acl;
//   }

  // Returns true if there are no more values
  bool end() const { 
    return (this == NULL)? true : (&data.value)[0] == ACL_MARKER && (&data.value)[1] == ACL_MARKER; 
  }

  // Compare operator
  bool operator==(const acl &a);

  // Assigment operator
  acl &operator=(const acl &a) {
    memcpy(&data.value, &a.data.value, sizeof(int) * (a.get_index() + 2));
    get_header().index = a.get_index();
    return *this;
  }

  // Creates a copy of the acl
  acl *clone() {
    if (this == NULL) return (acl*)NULL;
    acl *new_acl = new(get_size()) acl;
    *new_acl = *this;
    return new_acl;
  }

  // Returns item number i of an acl instance
  int get(int i) const { return ((int*)&data.value)[i]; }

  // Returns the current value the acl instance points to
  int get() const { return ((int*)&data.value)[0]; }

  // Set index i to value j
  acl *set(int i, int j) { 
    ((int*)&data.value)[i] = j;
    return (acl*)&data.value;
  }

  // Returns the next acl instance
  acl *next() { (acl*)((int*)&data.value)[1]; }

};

extern pacl tmpacl;
extern pacl tmpacl2;

#endif
