#ifndef FREEHDL_KERNEL_KERNEL_CLASS_H
#define FREEHDL_KERNEL_KERNEL_CLASS_H

#include <freehdl/kernel-flags.hh>
#include <freehdl/kernel-classes.hh>
#include <freehdl/kernel-list.hh>
#include <freehdl/std-standard.hh>
#include <freehdl/kernel-process-base.hh>
#include <freehdl/kernel-wait-info.hh>
#include <freehdl/kernel-global-event-queue.hh>


// Arguments that are passed in form the command line
extern int main_argc;
extern char **main_argv;


class sig_info_base;

class kernel_class {
  // Number of processes
  static int process_counter; 
  // Array of pointers to the processes the model consist of
  static process_base **process_array;
  // Each simulation cycle is assigned a unique cycle id. The cycle id
  // may be used to easily distinguish between different simulation
  // cycles. The id is incremented before a new simulation cycle
  // starts.
  static int cycle_id;
  // Array to store the processes which will be executed
  // next. Actually, this pointer is the head of a linked list. The
  // last link pointer of the list stores -1. Usually, -1 is not a
  // valid process_base address. Hence, it can be used similar to
  // NULL.
  static process_base *processes_to_execute;
  static g_trans_queue global_transaction_queue;
  // wait_id counter and current process
  int automatic_wait_id_counter;
  process_base *current_process;

public:

  // Number of processes which were executed during the simulation. Is
  // only enabled if corresponding macro is defined. See "flags.hh"
  STATISTICS(static int executed_processes_counter;);
  // Number of transactions which were created during simulation. Is
  // only enabled if corresponding macro is defined. See "flags.hh"
  STATISTICS(static int created_transactions_counter;);

  // Basic constuctor
  kernel_class(); 

  // ******************************************
  // Methods executed during elaboration
  // ******************************************

  // Elaborates the model
  void elaborate_model();
  // Adds a new process to the kernel
  void add_process(process_base *proc);
  // Calls constructor of an component specified by library, entity, and
  // architecture. An empty string "" can be used as wildcard for library,
  // entity, or architecture.
  void elaborate_component(const char *library, const char *entity,
			   const char *architecture, 
			   name_stack &iname, const char *name, 
			   map_list *mlist);
  // Adds a new signal to the kernel
  void add_signal(sig_info_base *signal);
  // Returns a pointer to a driver_info instance. This method is 
  // called by processes during elaboration to get a driver. All
  // signal assignments done by a process are executed via a
  // corresponding driver instance.
  driver_info *get_driver(process_base *proc, sig_info_base *signal, acl *a = NULL);
  // Rebuilds driver array. This method is called by processes during
  // elaboration e.g. to get an additional driver for another scalar
  // instance of an composite signal. All signal assignments done by a
  // process are executed via a corresponding driver instance.
  void get_driver(driver_info *driver, process_base *proc, acl *a);
  // Create and setup wait_info instances. The wait id is created
  // automatically. Note, all automatically wait ids less then 0.
  short setup_wait_info(const sigacl_list &salist, process_base *proc);
  // Create and setup wait_info instance using wait_id as wait id.
  short setup_wait_info(const short wait_id, const sigacl_list &salist, process_base *proc);

  // ******************************************
  // Methods executed during simulation
  // ******************************************

  // Returns the current simulation time
  const time &get_sim_time(void) { return global_transaction_queue.get_sim_time(); }
  // Returns the current delta count
  const int get_delta(void) { return global_transaction_queue.get_delta(); }
  // Get the current cycle id
  const int get_cycle_id(void) { return cycle_id; }
  // Execute all processes present on the processes_to_execute array
  void execute_processes();
  // Adds a process to be executed within the current simulation cycle
  void add_process_to_execute(process_base *process) {
    process_base &proc = *process;
    // Note, we check for NULL here (and not for -1). If the pointer
    // is equal to -1 then this element is at the end of the list. If
    // it is NULL then it is NOT on the list.
    if (proc.next_process == NULL) {
      proc.next_process = processes_to_execute;
      processes_to_execute = &proc;
    }
  }
  // Execute a simulation cycle. Returns whether there is still
  // an transaction on the event list
  bool next_cycle();
  // Assign all next transactions with the current simulation time to
  // the drivers. Returns wheter really some assginments have been done.
  // Note, if a transaction is removed from the driver, then it is not
  // removed from the global transaction list. Hence, we must first check
  // whether the first transaction of a driver list has the correct time
  // stamp.
  bool assign_next_transactions();
  // Add a transaction to the global transaction queue.
  void add_to_global_transaction_queue(driver_info *driver, const time &time_value) {
    global_transaction_queue.add_to_queue(driver, time_value);
  }
#ifdef EVENT_PROFILE
  int event_report(char *filename) {
    global_transaction_queue.event_report(filename);
  }
#endif

};



// The simulation kernel
extern kernel_class kernel; 

extern list<sig_info_base*> signal_list; // List of signals

#endif
