#ifndef FREEHDL_KERNEL_MAP_LIST_H
#define FREEHDL_KERNEL_MAP_LIST_H

#include <freehdl/std-vhdl-types.hh>
#include <freehdl/kernel-list.hh>
#include <freehdl/kernel-classes.hh>
#include <freehdl/kernel-acl.hh>

// A signal_link instance describes the connection
// between a formal and actual SIGNAL
struct signal_link {
  // The following two members determine the part of 
  // the formal signal which is addressed by this 
  // link element
  acl *formal_aclp;
  string formal_name;
  // To represent f1(formal_sig) => actual_sig
  // If a conversion function is used in an association list, then
  // this conversion function must be executed on control of the kernel.
  // However, the kernel does not know this function at kernel compile time
  // Moreover, it does not even know it parameter and return value types
  void (*wrapper_fn_formal)(void *dest, void *data);

  // The next four members determine the part of the
  // actual signal which is addressed by the map
  // operation. Note, if the formal is of mode IN then
  // the actual can be a constant expression. In this
  // case value points to this expression and type
  // describes its type.
  acl *actual_aclp;
  sig_info_base *actual_signal;	// equal to NULL is that 
								// actual is an expression!
  void *value;	// Only used if actual is an expression
				// Otherwies value is set to NULL.
  type_info_interface *type;	// Only used if actual is 
								// an expression. Otherwise equal to NULL.
  // To represent formal_sig => f2(actual_sig)
  void (*wrapper_fn_actual)(void *dest, void *data);
    
  // Constructor. Initializes formal_aclp, actual_aclp,
  // actual_signal, value, and type.
  signal_link(); 
  // Destructor. Removes formal_aclp, actual_aclp, and
  // value. 
    ~signal_link();
};


// A generic_link instance describes the connection
// between a formal generic parameter and actual
struct generic_link {
  // The following two members determine the part of 
  // the formal generic parameter which is addressed by 
  // this generic_link instance
  acl *formal_aclp;
  string formal_name;
  // value points to value the generic parameter is 
  // associated with and type points to the type info
  // instance.
  void *value;	// The value will be NULL if generic map is mapped with open
  type_info_interface *type;	// So is the type...
  
  // Constructor. Initializes formal_aclp, actual_aclp,
  // value, and type.
  generic_link(); 
  // Destructor. Removes formal_aclp, actual_aclp, and
  // value. 
  ~generic_link();
};


class map_list {
 public:

  //////////////////////////////////////////////////////
  // Members of map_list
  //////////////////////////////////////////////////////

  list<signal_link*> signal_maplist;
  list<generic_link*> generic_maplist;

  //////////////////////////////////////////////////////
  // Methods of map_list
  // Note, all signal_map resp. generic_map methods 
  // create a copy of the acls!!! 
  //////////////////////////////////////////////////////

  // Adds a generic link to the generic_map list
  // It has only one connection with expression.
  void generic_map(const char *formal, acl *formal_acl,
		   void *value, type_info_interface *type);

  // the mapping with open
  void generic_map(const char *formal, acl *formal_acl);

  // Adds a signal link to the signal_map list
  void signal_map(const char *formal, acl *formal_acl,
		  sig_info_base *actual, acl *actual_acl);

  // the mapping with an expression
  void signal_map(const char *formal, acl *formal_acl,
		  void *value, type_info_interface *type);

  // the mapping with open
  void signal_map(const char *formal, acl *formal_acl);

  // the mapping with formal convert function
  void signal_map(const char *formal, acl *formal_acl,
		  void (*wrapper_fn_formal)(void *,void *),
		  sig_info_base *actual, acl *actual_acl);

  // the mapping with actual convert function
  void signal_map(const char *formal, acl *formal_acl,
		  sig_info_base *actual, acl *actual_acl,
		  void (*wrapper_fn_actual)(void *,void *));

  // the mapping with formal/actual convert function
  void signal_map(const char *formal, acl *formal_acl,
		  void (*wrapper_fn_formal)(void *,void *),
		  sig_info_base *actual, acl *actual_acl,
		  void (*wrapper_fn_actual)(void *,void *));

  //////////////////////////////////////////////////////
  // The query methods are used by the kernel to search
  // for a specific generic parameter respectively 
  // signal formal. Note, severals signal_link respectively
  // generic_link instances may exist for formals of 
  // composite type. Hence, the query functions store
  // in "result" a LIST of appropriate pointers.
  //////////////////////////////////////////////////////

  // Returns in result a list of pointers to the generic_link
  // instances with formal_name = name. Returns true
  // if at least one entry was found.
  bool query_generic(list<generic_link*> &result, const string &name);
  
  // Returns in result a list of pointers to the signal_link
  // instances with formal_name = name. Returns true
  // if at least actual was found.
  bool query_signal(list<signal_link*> &result, const string &name);


  // Calculates the index range for an unconstrained formal array port
  // or generic parameter. The bounds are determined from the index type
  // of the array respectively the index values which were used in the
  // association list. Returns -1 if an error occured, 0 otherwise.
  // Returns the bound in "left" and "right". "name" is the name of the
  // formal, "ainfo" the info instance of the formal and "mlist" a list
  // of elements associated with the formal.
  int get_port_array_bounds(list<signal_link*> &mlist, const string &name, 
			    array_info *ainfo, int &left, int &right);

  // Constructor
  map_list();

  // Destructor
  ~map_list();

  // Reset map_list instance so that it may be re-used
  void reset();
};


// global temproary map list object 
extern map_list tmpml;

#endif
