#ifndef FREEHDL_KERNEL_SIG_INFO_H
#define FREEHDL_KERNEL_SIG_INFO_H

#include <string>
#include <freehdl/kernel-classes.hh>
#include <freehdl/kernel-list.hh>
#include <freehdl/kernel-name-stack.hh>
#include <freehdl/kernel-map-list.hh>
#include <freehdl/std-standard.hh>
#include <freehdl/kernel-resolver.hh>


/*
 * Information flow of signals
 */
#define vIN  1
#define vOUT 2
#define vINOUT 3
#define vBUFFER 4
#define vLINKAGE 5

/*
 * Signal typeof a signal
 */
#define vREGISTER 1
#define vBUS 2

/*
 * Signal attribute types
 */
#define vDELAYED 1
#define vTRANSACTION 2
#define vSTABLE 3
#define vQUIET 4

/*
 * Delay type
 */
#define vTRANSPORT 1
#define vINERTIAL 2


/******************************************************
 * Some definitions which are used by the kernel only 
 ******************************************************/
#ifdef KERNEL

#include <string>
#include <map>

typedef map<string, sig_info_base*, less<string> > signal_map;

// signal_table is a hash tables (map) to associate signal names with
// the corresponding sign_info_base pointers.
extern signal_map signal_table;

#endif
/******************************************************
 * End of internal kernel definitions    
 ******************************************************/


class reader_info;

/*
 * Base sig info class
 */
struct sig_info_base {
  // Points to the reader object of the signal
  void *reader_pointer; 
  // Type of the signal
  type_info_interface *type;
  
  // Scalar items the signal reader consists of
  int scalar_count;
  // array of reader_info pointers. There is a separate pointer
  // for each scalar element of the signal
  reader_info **readers;
  // List of drivers for the signal. Note there is a entry in the
  // list for each scalar element which is driven by a process
  list<driver_info*> driver_list;

  union { char direction; char sig_type; char attribute; };

  // To support multiple fathers of a signal (a composite signal)
  // which represents the partial/direct/indirect signal interconnection
  // direct mapping : formal => actual
  // partial mapping : formal(1 downto 0) => actual(0 downto 1) or actual.field
  // indirect mapping : f1(formal) => f2(actual)
  struct link {
    // Determines which part of the formal signal is addressed
    acl *formal_aclp;
    // Determines which part of the actual signal is addressed
    acl *actual_aclp;
    // wrapper function to support conversion function of formal/actual signal
    void (*wrapper_fn_formal)(void *dest, void *data);
    void (*wrapper_fn_actual)(void *dest, void *data);
    // Points to the actual signal
    sig_info_base *actual;
  };
  // Stores the father signal(s) of the current signal
  list<link*> signal_links;

  resolver_base *resolver;

  // Instance name and name of the signal
  string instance_name;
  string name;

  /* Constructors */
  sig_info_base(name_stack &iname, map_list *mlist,
		type_info_interface *ty, char dir,
		resolver_base *res);
  sig_info_base(name_stack &iname, type_info_interface *ty, 
		char sty, resolver_base *res);
  sig_info_base(name_stack &iname, type_info_interface *ty, 
		char attr, sig_info_base *base_sig, acl *aclp,
		time delay);
  sig_info_base(name_stack &iname);  

  // Inits the reader of a signal
  void init_reader(const void *src);
};



template<class T>class sig_info : public sig_info_base {
 public:
  /* Casts the reader pointer to the corresponding type */
  T &reader() { return *(T*)reader_pointer; }
  /* Constructor to instantiate a port signal */
  sig_info(name_stack &iname, map_list *mlist,
	   type_info_interface *type, char direction,
	   resolver_base *res) : 
    sig_info_base(iname, mlist, type, direction, res) {};
  /* Constructor to instantiate "normal" signal */
  sig_info(name_stack &iname, type_info_interface *type, 
	   char sigtype, resolver_base *res) :
    sig_info_base(iname, type, sigtype, res) {};
  /* Constructor to instantiate a signal kind attribute */
  sig_info(name_stack &iname, type_info_interface *type, 
	   char attr, sig_info_base *base_sig, acl *aclp,
	   time delay) :
    sig_info_base(iname, type, attr, base_sig, aclp, delay) {};
  /* Constructor to instantiate a guard signal */
  sig_info(name_stack &iname) : sig_info_base(iname) {};
  /* Init the reader of the signal */
  void init(const T &value) { init_reader(&value); }
};



#endif
