#ifndef FREEHDL_STD_VHDL_TYPES_H
#define FREEHDL_STD_VHDL_TYPES_H

#include <strstream.h>
#include <limits.h>
#include <freehdl/kernel-error.hh>
#include <freehdl/kernel-acl.hh>


/* *************************************************************
 *  Some routines of global interest
 * ************************************************************* */

/* Returns true if value is in between left and right */
template<class T>
inline bool check_bounds(const T &left, const T &right, const T &value) {
  if (left <= right) 
    return left <= value && value <= right;
  else 
    return left >= value && value >= right;
}

template<class T>
inline T min(const T old_value, const T new_value) { 
  return new_value < old_value? new_value : old_value; 
}

template<class T>
inline T max(const T old_value, const T new_value) { 
  return new_value > old_value? new_value : old_value; 
}

/* *************************************************************
 *  Type-Ids for the different VHDL type groups 
 * ************************************************************* */
enum type_id {INTEGER, ENUM, FLOAT, PHYSICAL, RECORD, ARRAY, ACCESS, VHDLFILE};


/* *************************************************************
 *  Typedefs
 * ************************************************************* */
typedef long long int lint;


/* *************************************************************
 *  Logartihmus dualis of the sizes of VHDL types (in Bytes)
 * ************************************************************* */
#define INTEGER_SIZE sizeof(integer) 
#define ENUM_SIZE sizeof(enumeration)
#define FLOAT_SIZE sizeof(floatingpoint)
#define PHYSICAL_SIZE sizeof(physical)
#define RECORD_SIZE -1 /* invalid, has not const size */
#define ARRAY_SIZE sizeof(array_base)  /* Not including memory for info and data instances */
#define ACCESS_SIZE sizeof(access_base)
#define VHDLFILE_SIZE -1 /* not defined yet */


/* *************************************************************
 *  Range directions
 * ************************************************************* */
enum range_direction {to, downto};


/* *************************************************************
 *  All VHDL integer types are actually C ints
 * ************************************************************* */
typedef int integer;

/* Always retuns true. Note that that the parameter is actually never
 * used. It is required by the compiler to select the appropriate
 * function. */
inline bool scalar(integer *d) {  return true; }

/* Actually does nothing */
inline void cleanup(integer *d) { };


/* *************************************************************
 *  All VHDL enumeration types are actually C chars
 * ************************************************************* */
typedef char enumeration;

/* Always retuns true. Note that that the parameter is actually never
 * used. It is required by the compiler to select the appropriate
 * function. */
inline bool scalar(enumeration *d) {  return true; }

/* Actually does nothing */
inline void cleanup(enumeration *d) { };


/* *************************************************************
 *  All VHDL floating point types are actually C doubles
 * ************************************************************* */
typedef double floatingpoint;

/* Always retuns true. Note that that the parameter is actually never
 * used. It is required by the compiler to select the appropriate
 * function. */
inline bool scalar(floatingpoint *d) {  return true; }

/* Actually does nothing */
inline void cleanup(floatingpoint *d) { };


/* *************************************************************
 *  All VHDL physical types are actually long long ints (64 bit)
 * ************************************************************* */
typedef lint physical;

/* Always retuns true. Note that that the parameter is actually never
 * used. It is required by the compiler to select the appropriate
 * function. */
inline bool scalar(physical *d) {  return true; }

/* Actually does nothing */
inline void cleanup(physical *d) { };


/* *************************************************************
 *  Some functions for composite types
 * ************************************************************* */

/* Function to call the cleanup operator of the appropriate array
   instance */
template<class A>
void cleanup(A *p) { p->cleanup_instance(); }


/* *************************************************************
 *  Some operators
 * ************************************************************* */

/* not operator */
inline enumeration op_not(const enumeration a) { 
  return a ? 0 : 1; 
}

/* nand operator */
inline enumeration op_nand(const enumeration a, const enumeration b) {
  return op_not(a & b);
}

/* xor operator */
inline enumeration op_xor(const enumeration a, const enumeration b) {
  return a ^ b;
}

/* xnor operator */
inline enumeration op_xnor(const enumeration a, const enumeration b) {
  return op_not(op_xor(a, b));
}

/* nor operator */
inline enumeration op_nor(const enumeration a, const enumeration b) {
  return a | b ? 0 : 1;
}



/* *************************************************************
 *  The type info interface class is te base class for all 
 *  derived type info classes 
 * ************************************************************* */
class type_info_interface {
public:
  type_id id;
  short size; /* size of a corresponding type object */
  type_info_interface(const type_id i, const short s);

  // Returns true if the current info instance describes a scalar type
  inline bool scalar();

  // Prints value of instance (src is pointing to) into a char
  // buffer and returns a pointer to the buffer. Note, the buffer
  // is static! Hence, each call to str uses the SAME buffer!
  char *str(const void *src);

  // Copy the content of src to dest. Note can be used
  // for scalar VHDL objects only!
  inline void *fast_copy(void *dest, const void *src);
  // Compare the content of src and dest. Note: can be used
  // for scalar VHDL objects only!
  bool fast_compare(const void *dest, const void *src);
  // Assigns the content of src and dest. Note: can be used
  // for scalar VHDL objects only! Returns true if the new
  // value differs from the old one
  bool fast_assign(void *dest, const void *src);

  // Gets left and right bounds of an enumeration or integer type and
  // returns 0. Returns -1 if type is no enumeration or integer type.
  int get_bounds(int &left, int &right);

  // Create a new object. Memory is allocated by malloc
  virtual void *create() = 0;
  // Clone a object. Memory is allocated by malloc
  virtual void *clone(const void *src) = 0;
  // Copy the content of src to dest
  virtual void *copy(void *dest, const void *src) = 0;
  // Initiliazes a instance
  virtual void init(void *src) = 0;
  // Clears additional memory occupied by an object.
  virtual void clear(void *src) {};
  // Compare src1 with src2. Returns true if they are equal
  virtual bool compare(const void *src1, const void *src2) = 0;
  // Copy the content of src to dest. Returns true if the new
  // value differs from the old one
  virtual bool assign(void *dest, const void *src) = 0;
  // Removes an object (frees memory)
  virtual void remove(void *src) = 0;
  // Returns the address of an appropriate subelement of
  // the current object. acl is the "path" to the subelement
  virtual void *element(void *src, acl *a) { return src; }
  // Returns the address of an appropriate subelement of
  // the current object. i is the index number of the subelment.
  void *element(void *src, int i);
  // Convert acl to index start and end. Returns start value
  int acl_to_index(acl *a, int &start, int &end) const;
  // Convert acl to index start
  int acl_to_index(acl *a) const;
  // Get info instance if scalar element with index i
  type_info_interface *get_info(int i) const;
  // Returns the number of scalar elements of the current object
  virtual int element_count() { return 1; }
  // Prints the content of src into an string stream
  virtual void print(ostrstream &str, const void *src) = 0;
  // Usualy increments/decrements reference counter of type_info
  // instances. However, as these instances are static for all
  // but composite types both function actually do noting. Note
  // that the corresponding methods are overloaded by the array_info
  // and record_info classes. 
  void add_ref() { };
  void remove_ref() { };
};


// Copy the content of src to dest. Note can be used
// for scalar VHDL objects only!
inline void *
type_info_interface::fast_copy(void *dest, const void *src) 
{
  switch (id) {
  case ENUM:
    *(char*)dest = *(char*)src;
    break;
  case INTEGER:
    *(int*)dest = *(int*)src;
    break;
  case FLOAT:
  case PHYSICAL:
    *(lint*)dest = *(lint*)src;
    break;
  }
  return dest;
}

// Compare the content of src and dest. Note: can be used
// for scalar VHDL objects only!
inline bool 
type_info_interface::fast_compare(const void *dest, const void *src) 
{
  switch (id) {
  case ENUM:
    return *(char*)dest == *(char*)src;
    break;
  case INTEGER:
    return *(int*)dest == *(int*)src;
    break;
  case FLOAT:
  case PHYSICAL:
    return *(lint*)dest == *(lint*)src;
    break;
  }
}


// Assigns the content of src and dest. Note: can be used
// for scalar VHDL objects only! Returns true if the new
// value differs from the old one
inline bool 
type_info_interface::fast_assign(void *dest, const void *src) 
{
  switch (id) {
  case ENUM:
    {
      char src_val = *(char*)src, dest_val = *(char*)dest;
      if (src_val != dest_val) {
	*(char*)dest = src_val;
	return true;
      } else return false;
    }
  case INTEGER:
    {
      int src_val = *(int*)src, dest_val = *(int*)dest;
      if (src_val != dest_val) {
	*(int*)dest = src_val;
	return true;
      } else return false;
    }
  case FLOAT:
  case PHYSICAL: 
    {
      lint src_val = *(lint*)src, dest_val = *(lint*)dest;
      if (src_val != dest_val) {
	*(lint*)dest = src_val;
	return true;
      } else return false;
    }
  }
}


inline bool
type_info_interface::scalar()
{
  switch (id) {
  case RECORD:
  case ARRAY:
    return false;
  default:
    return true;
  }
}



/* *************************************************************
 *  Integer info base class
 * ************************************************************* */
class integer_info_base : public type_info_interface {
public:
  int left_bound, right_bound, low_bound, high_bound;

  integer_info_base(const int le, const int ri, 
		    const int lo, const int hi);

  void *create();
  void *clone(const void *src);
  void *copy(void *dest, const void *src);
  void init(void *src);
  bool compare(const void *src1, const void *src2);
  bool assign(void *dest, const void *src);
  void remove(void *src);
  void print(ostrstream &str, const void *src);
};




/* *************************************************************
 *  Float  info base class
 * ************************************************************* */
class float_info_base : public type_info_interface {
public:
  double left_bound, right_bound, low_bound, high_bound;

  float_info_base(const double le, const double ri, 
		  const double lo, const double hi);

  void *create();
  void *clone(const void *src);
  void *copy(void *dest, const void *src);
  void init(void *src);
  bool compare(const void *src1, const void *src2);
  bool assign(void *dest, const void *src);
  void remove(void *src);
  void print(ostrstream &str, const void *src);
};


/* *************************************************************
 *  Enum info base class
 * ************************************************************* */
class enum_info_base : public type_info_interface {
public:
  int left_bound, right_bound, length;
  const char **values;

  enum_info_base(const int le, const int ri, const char **val);

  void *create();
  void *clone(const void *src);
  void *copy(void *dest, const void *src);
  void init(void *src);
  bool compare(const void *src1, const void *src2);
  bool assign(void *dest, const void *src);
  void remove(void *src);
  void print(ostrstream &str, const void *src);
};



/* *************************************************************
 *  Physical info base class
 * ************************************************************* */
class physical_info_base : public type_info_interface {
public:
  lint left_bound, right_bound, low_bound, high_bound;
  const char **units;
  const lint *scale;
  int unit_count;

  physical_info_base(const lint le, const lint ri, 
		     const lint lo, const lint hi,
		     const char **un, const lint *sc, int uc);

  void *create();
  void *clone(const void *src);
  void *copy(void *dest, const void *src);
  void init(void *src);
  bool compare(const void *src1, const void *src2);
  bool assign(void *dest, const void *src);
  void remove(void *src);
  void print(ostrstream &str, const void *src);
};


/* *************************************************************
 *  Array info base class
 * ************************************************************* */

class array_info : public type_info_interface {
public:
  /* Dimension info */
  range_direction index_direction; /* = "to" if index is of enumeration type */
  int left_bound, right_bound;
  int length; /* = -1 if bounds are actually not set (unconstrained array) */

  /* Index type info */
  type_info_interface *index_type;
  
  /* Element info */
  type_info_interface *element_type;


  /* Counts the number of array instances which are currently using  
   * this array_info_base object to define their bounds. Hence, several 
   * array instances may share a single array_info_base instance! */
  int ref_counter;
  
  /* Methods */

  /* Constructor. If rc is set to -1 then the array_info instance 
   * will not be removed */
  array_info(type_info_interface *et = NULL, type_info_interface *it = NULL, int rc = 0);
  /* Constructor to create an info instance for an array subtype */
  array_info(array_info *base, int le, range_direction r, int ri, int rc=0);
  /* Constructor to create an info instance where the left bound is
   * determined by base and the right bound is derived from len */
  array_info(array_info *base, int len, int rc=0);
  /* Destructor */
  ~array_info() { element_type->remove_ref(); }
  /* Set the element info object */
  void set(type_info_interface *et) { element_type = et; element_type->add_ref(); }
  /* Set the element info object and the range of the array */
  void set(type_info_interface *et, type_info_interface *it, int le, range_direction r, int ri) { 
    index_direction = r;
    left_bound = le; 
    right_bound = ri;
    length = (ri - le) * (r == to? 1 : -1);
    length = length < 0? 0 : length + 1;
    index_type = it;
    element_type = et;
    element_type->add_ref();
  }
  /* This method must be called by each array instance which references
   * the current info instance */
  void add_ref() { 
    if (ref_counter >= 0) ref_counter++;
  }
  /* If rec_counter == 0 then no object uses this info instance and
   * it is removed. Note, there are also static info instances which
   * are not removed even when they are not in use. To create a
   * static info instance, array_info is called with parameter -1. */
  void remove_ref() {
    if (ref_counter > 0) {
      ref_counter--;
      if (ref_counter == 0) 
	delete this;
    }
  }
  /* Returns the number of scalar elements of the current type */
  int element_count() { return length * element_type->element_count(); }
  /* Returns the address of an appropriate subelement of
   * the current object. acl is the "path" to the subelement */
  void *element(void *src, acl *a);
  /* Note, the kernel does not handel object of composite directly.
   * Hence create, clone, ... defined here are dummy functions only.
   * A exception from this is the method print. */
  void *create();
  void *clone(const void *src) {};
  void *copy(void *dest, const void *src);
  void init(void *src);
  void clear(void *src);
  bool compare(const void *src1, const void *src2) { };
  bool assign(void *dest, const void *src) { };
  void remove(void *src);
  void print(ostrstream &str, const void *src);
};


// Function to cast an type_info_interface pointer to an array_info
// pointer
inline array_info *parray_info(type_info_interface *t) { return (array_info*)t; }


/* *************************************************************
 * The type classes
 * ************************************************************* */


/* *************************************************************
 * Array type class
 * ************************************************************* */
class array_base {
public:
  array_info *info; /* Points to the object which stores information
		     * about the current array */
  char *data; /* Pointer to the actual data of the array */

  array_base() { };
  array_base(const array_base &a);
  array_base(array_info *i) { info = i; i->add_ref(); }
  void set_info(array_info *i) {
    if (info) info->remove_ref();
    info = i;
    i->add_ref();
  }
  array_base &operator=(const array_base &a);
  static int size() { return ARRAY_SIZE; }
};


/* Always retuns false. Note that that the parameter is actually never
 * used. It is required by the compiler to select the appropriate
 * function. */
inline bool scalar(array_base *a) { return false; }


/* E is the array element type and R is the corresponding
 * info class of the _unconstrained_ array range type. */
template<class E, class R> 
class array_type : public array_base {
 public:
  typedef E E_type; /* declare a local type name which is
  equivalent to the element type */
  void cleanup_instance();
  array_type() {};
  /* init initiliazes an array from another array or, if 
   * p == NULL the elements of the array are initialized
   * with their default values. */
  array_type &init(type_info_interface *ainfo, void *p = NULL);
  array_type(const array_type &a) : array_base() { init(a.info, (void*)&a); }
  array_type(array_info *ainfo, const E *iarray);
  array_type(array_info *ainfo, const E &val);
  ~array_type() { cleanup_instance(); }

  /* Methods used to build array aggregats */
  array_type &aggregate_set(int left, range_direction direction, int right, const E &value);
  array_type &aggregate_copy(int left, range_direction direction, int right, int copy_index);

  E &operator[](const int i) {
    int index = (info->index_direction == to? i - info->left_bound : info->left_bound - i);
    if (index < 0 || index > info->length)
      error(ERROR_ARRAY_INDEX);
    return ((E*)data)[index];
  }

  array_type &operator=(const array_type &array);

  bool operator!=(const array_type &array);
  bool operator==(const array_type &array);
  bool operator<(const array_type &array);
  bool operator<=(const array_type &array);
  bool operator>(const array_type &array);
  bool operator>=(const array_type &array);
};


/* Method to cleanup an array instance, i.e. memory allocated by the
   instance is freed */
template<class E, class R>
void array_type<E,R>::cleanup_instance()
{
  /* Note, scalar((E*)NULL) can be evaluated at compile time. The
   * parameter is actually only used to select a appropriate
   * scalar(...) function */
  if (!scalar((E*)NULL))
    /* If the element type is not scalar then execute
     * cleanup() for each element in the array */
    for (int i = 0; i < info->length; i++)
      cleanup(&((E*)data)[i]);

  if (data != NULL)
    delete[] data; /* Remove data memory */

  info->remove_ref(); /* Unlink from the info instance */
}




/* The concat operator */
template<class A, class E>
A concat(array_info *ainfo, const A &a1, const A &a2)
{
  int length = ainfo->length;
  // If both arrays are null arrays then return the result is the
  // right operand
  if (ainfo->length == 0)
    return A(a2);

  // Create new array
  A new_array;
  new_array.info = ainfo;
  new_array.info->add_ref();

  /* Allocate memory for the data */
  const int mem_size = length * ainfo->element_type->size();
  new_array.data = new char[mem_size];

  /* Note, scalar((E*)NULL) can be evaluated at compile time. The
   * parameter is actually only used to select a appropriate
   * scalar(...) function */
  type_info_interface *etype = ainfo->element_type;
  if (!scalar((E*)NULL)) {
    /* If the element type is not scalar then init the memory and
     * execute init(...) for each element in the array */
    memset(new_array.data, 0, mem_size);
    for (int i = 0; i < length; i++)
      etype->init(&((E*)new_array.data)[i]);
  }

  /* Copy data */
  int length1 = a1.info->length1;
  int length2 = a2.info->length2;
  int i;
  for (i = 0; i < length1; i++)
    ((E*)new_array.data)[i] = ((E*)a1.data)[i];
  for (; i < length; i++)
    ((E*)new_array.data)[i] = ((E*)a2.data)[i];
  
  return new_array;
}


/* Another version of the concat operator */
template<class A, class E>
A concat(const A &a1, const A &a2)
{
  int length = a1.info->length + a2.info->length;
  // If both arrays are null arrays then return the result is the
  // right operand
  if (length == 0)
    return A(a2);

  // Create new array
  A new_array;
  array_info &ref_ainfo = a1.info->length != 0? *a1.info : *a2.info;

  new_array.info = new array_info(ref_ainfo.element_type, ref_ainfo.index_type, 0);
  array_info &new_info = *new_array.info;
  new_array.info->add_ref();
  int right;
  ref_ainfo.index_type->get_bounds(new_info.left_bound, right);
  if (new_info.left_bound < right) {
    new_info.right_bound = new_info.left_bound + length - 1;
    new_info.index_direction = to;
    if (new_info.right_bound > right) error(ERROR_ARRAY_INDEX_OUT_OF_BOUNDS);
  } else {
    new_info.right_bound = new_info.left_bound - length + 1;
    new_info.index_direction = downto;
    if (new_info.right_bound < right) error(ERROR_ARRAY_INDEX_OUT_OF_BOUNDS);
  }
  new_info.length = length;

  /* Allocate memory for the data */
  const int mem_size = length * sizeof(E);
  new_array.data = new char[mem_size];

  /* Note, scalar((E*)NULL) can be evaluated at compile time. The
   * parameter is actually only used to select a appropriate
   * scalar(...) function */
  type_info_interface *etype = new_array.info->element_type;
  if (!scalar((E*)NULL)) {
    /* If the element type is not scalar then execute init(...) for
     * each element in the array after setting the memory to 0 */
    memset(new_array.data, 0, mem_size);
    for (int i = 0; i < length; i++)
      etype->init(&((E*)new_array.data)[i]);
  }

  /* Copy data */
  int length1 = a1.info->length;
  int i;
  for (i = 0; i < length1; i++)
    ((E*)new_array.data)[i] = ((E*)a1.data)[i];
  for (int j = 0; i < length; i++, j++)
    ((E*)new_array.data)[i] = ((E*)a2.data)[j];
  
  return new_array;
}



template<class E, class R>
array_type<E,R>::array_type<E,R>(array_info *ainfo, const E *iarray)
{
  /* Create a new array_info instance */
  info = ainfo;
  info->add_ref();

  /* Allocate memory for the data */
  int length = info->length;
  const int mem_size = length * sizeof(E);
  data = new char[mem_size];

  /* Note, scalar((E*)NULL) can be evaluated at compile time. The
   * parameter is actually only used to select a appropriate
   * scalar(...) function */
  type_info_interface *etype = info->element_type;
  if (!scalar((E*)NULL)) {
    /* If the element type is not scalar then execute init(...) for
     * each element in the array after initialing the memory to 0 */
    memset(data, 0, mem_size);
    for (int i = 0; i < length; i++)
      etype->init(&((E*)data)[i]);
  }

  /* Copy data */
  for (int i = 0; i < length; i++)
    ((E*)data)[i] = iarray[i];
}


template<class E, class R>
array_type<E,R>::array_type<E,R>(array_info *ainfo, const E &val)
{
  /* Create a new array_info instance */
  info = ainfo;
  info->add_ref();
  int length = info->length;

  /* Allocate memory for the data */
  const int mem_size = length * sizeof(E);
  data = new char[mem_size];

  if (!scalar((E*)NULL))
    /* If the element type is not scalar then init memory to 0 */
    memset(data, 0, mem_size);

  /* Note, scalar((E*)NULL) can be evaluated at compile time. The
   * parameter is actually only used to select a appropriate
   * scalar(...) function */
  type_info_interface *etype = info->element_type;
  if (!scalar((E*)NULL)) {
    /* If the element type is not scalar then execute init(...) for
     * each element in the array after initialing the memory to 0 */
    memset(data, 0, mem_size);
    for (int i = 0; i < length; i++)
      etype->init(&((E*)data)[i]);
  }

  /* Each element of the array is set to val */
  for (int i = 0; i < length; i++)
    ((E*)data)[i] = val;
}



template<class E, class R>
array_type<E,R> &array_type<E,R>::operator=(const array_type<E,R> &array)
{
  int length = info->length;

  /* Check whether array bounds are compatible */
  if (info != array.info && length != array.info->length)
    error(ERROR_INCOMPATIBLE_ARRAYS);

  /* Copy the data part of the arrays */
  for (int i = 0; i < length; i++)
    ((E*)data)[i] = ((E*)array.data)[i];

  return *this;
}


template<class E, class R>
array_type<E,R> &array_type<E,R>::init(type_info_interface *ainfo, void *p = NULL)
{
  info = (array_info*)ainfo;
  info->add_ref();
  int length = info->length;
  type_info_interface *etype = info->element_type;

  /* Allocate memory for the data */
  const int mem_size = length * sizeof(E);
  data = new char[mem_size];

  if (!scalar((E*)NULL))
    /* If the element type is not scalar then init memory to 0 */
    memset(data, 0, mem_size);

  if (p) {
    /* If p != NULL then p points to an array instance */
    E *src_data = (E*)((array_base*)p)->data;
    /* Copy each element of the array */
    for (int i = 0; i < length; i++)
      etype->copy(&((E*)data)[i], &src_data[i]);
  } else {
    /* Initialize each element of the array */
    for (int i = 0; i < length; i++)
      etype->init(&((E*)data)[i]);
  }

  return *this;
}


/* Method to set array elements left to/downto right to value */
template<class E, class R>
array_type<E,R> &array_type<E,R>::aggregate_set(int left, range_direction direction, int right, const E &value)
{
  if (direction == downto) {
    int tmp = left;
    left = right;
    right = tmp;
  }
  for (int i = left; i <= right; i++)
    (*this)[i] = value;

  return *this;
}


/* Method to copys array element with index copy_index to array
 * elements left to/downto right to value */
template<class E, class R>
array_type<E,R> &array_type<E,R>::aggregate_copy(int left, range_direction direction, int right, int copy_index)
{
  if (direction == downto) {
    int tmp = left;
    left = right;
    right = tmp;
  }
  E &value = ((E*)data)[copy_index];
  for (int i = left; i <= right; i++)
    (*this)[i] = value;

  return *this;
}



/* *************************************************************
 * Array alias type class
 * ************************************************************* */

/* array_alias class is a array class derived from an array_type class
 * (T). This special class is used to build array alias instances of
 * an array object. Basically, the alias class adds some new
 * constructors and a destructor. The constructors do not create a new
 * data array but reuse an existing array. Hence, the constructor does
 * not deallocate the array. 
 * 
 * Note that T must be an array_type class! */
template<class T> 
class array_alias : public T {
public:
  array_alias(array_info *base, int le, range_direction dir, int ri, int rc, void *iarray) : T() {
    /* Create a new array_info instance */
    info = new array_info(base, le, dir, ri, rc);
    data = (char*)iarray;
  }
  array_alias(array_info *ainfo, void *iarray) : T() { 
    info = ainfo; 
    info->add_ref(); 
    data = (char*)iarray; 
  }
  ~array_alias() { 
    /* Note that the data array is NOT deallocated as the memory has
     * been allocated by another array_type instance */
    data = NULL; /* Set pointer to NULL so that the destructor of
		  * class T will NOT remove the memory! */
  }
};






#endif
