#include <stdlib.h>
#include <freehdl/kernel-flags.hh>
#include <freehdl/std-standard.hh>
#include <freehdl/kernel-reader-info.hh>
#include <freehdl/kernel-driver-info.hh>
#include <freehdl/kernel-sig-info.hh>
#include <freehdl/kernel-kernel-class.hh>


const time zero_time = 0;

// free_items points to a list of free transaction items. Note, all
// transaction lists share the SAME free_item list.
void *fqueue<long long int,time>::free_items = NULL;


driver_info::driver_info(process_base *proc, sig_info_base *sig, int i) 
{
  process = proc;
  signal = sig;
  type = signal->type->get_info(i);
  index_start = i;
  rinfo = signal->readers[i];
  size = 0;
}



driver_info::driver_info(process_base *proc, sig_info_base *sig, int i, driver_info **drvs, int sz) 
{
  process = proc;
  signal = sig;
  type = signal->type;
  index_start = i;
  rinfo = NULL;
  drivers = drvs;
  size = sz;
}


/*************************************************************************
 *
 * Templates to create transactions for scalar signals
 *
 *************************************************************************/



// Creates a transaction with transport delay
template<class T>
inline void
do_transport_assignment(driver_info &driver, const T &value, const time &time_value) 
{
  fqueue<lint, time> &transactions = driver.transactions;
  void *pos = transactions.start();
  time tr_time = kernel.get_sim_time() + time_value;
  
  while (transactions.next(pos)) {
    void *npos = transactions.next(pos);
    if (transactions.key(npos) >= tr_time) {
      transactions.cut_remove(npos);
      break;
    }
    pos = npos;
  }

  (T&)transactions.content(transactions.append(pos, tr_time)) = value;
  kernel.add_to_global_transaction_queue(&driver, tr_time);
  STATISTICS(kernel.created_transactions_counter++;);
  REPORT(cout << "Transaction (transport delay) for signal " << signal->instance_name <<
	 " (value=" << driver.type->str(value) << ", time=";
	 cout << L3std_Q8standard_I4time_INFO->str(&tr_time) << ") created." << endl;);
}




// Creates a transaction with inertial delay
template<class T>
inline void
do_inertial_assignment(driver_info &driver, const T &value, 
		       const time &time_value, const time &start_time) 
{
  fqueue<lint, time> &transactions = driver.transactions;
  void *pos = transactions.start();
  time rm_time = kernel.get_sim_time() + start_time;

  while (transactions.next(pos) && transactions.key(transactions.next(pos)) < rm_time)
    pos = transactions.next(pos);
  void *start_pos = pos;

  void *rpos = NULL;
  time tr_time = kernel.get_sim_time() + time_value;
  while (transactions.next(pos)) {
    void *npos = transactions.next(pos);
    if (transactions.key(npos) >= tr_time) {
      transactions.cut_remove(npos);
      break;
    }
    if (value != ((T&)transactions.content(npos))) {
      if (rpos)
	while (rpos != npos)
	  rpos = transactions.remove(npos);
      transactions.remove(npos);
      rpos = NULL;
      pos = start_pos;
    } else {
      rpos = rpos? rpos : npos;
      pos = npos;
    }
  }
    
  (T&)transactions.content(transactions.append(pos, tr_time)) = value;
  kernel.add_to_global_transaction_queue(&driver, tr_time);
  STATISTICS(kernel.created_transactions_counter++;);
  REPORT(cout << "Transaction (inertial delay) for signal " << signal->instance_name <<
	 " (value=" << type->str(value) << ", time=";
	 cout << L3std_Q8standard_I4time_INFO->str(&tr_time) << ") created." << endl;)
}




// Creates a transaction with inertial delay where the plus reject
// interval is equal to the delay value.
template<class T>
inline void
do_inertial_assignment(driver_info &driver, const T &value, const time &time_value) 
{
  // Check if the new value and the current value of the signal are
  // equal. If they are equal then the assignment cannot produce an
  // *event*. Hence, the transaction can be skipped. Note, this
  // technique can be used only if the signal is *not resolved* and
  // none of the attributes TRANSACTION or QUIET or ... are applied on
  // this signal!
  if (value == *((T*)driver.rinfo->reader))
    return;
  
  fqueue<lint, time> &transactions = driver.transactions;
  void *new_pos = transactions.new_item();
  void *pos = transactions.start();
  time tr_time = kernel.get_sim_time() + time_value;

  (T&)transactions.content(new_pos) = value;
  transactions.key(new_pos) = tr_time;
  
  void *rpos = NULL;
  while (transactions.next(pos)) {
    void *npos = transactions.next(pos);
    if (transactions.key(npos) >= tr_time) {
      transactions.cut_remove(npos);
      break;
    }
    if (value != ((T&)transactions.content(npos))) {
      if (rpos)
	while (rpos != npos)
	  rpos = transactions.remove(npos);
      transactions.remove(npos);
      rpos = NULL;
      pos = transactions.start();
    } else {
      rpos = rpos? rpos : npos;
      pos = npos;
    }
  }
    
  transactions.append(pos, new_pos);
  kernel.add_to_global_transaction_queue(&driver, tr_time);
  STATISTICS(kernel.created_transactions_counter++;);
  REPORT(cout << "Transaction (inertial delay) for signal " << signal->instance_name <<
	 " (value=" << driver.type->str(value) << ", time=";
	 cout << L3std_Q8standard_I4time_INFO->str(&tr_time) << ") created." << endl;)
}



/*************************************************************************
 *
 * Methods to create transctions for composite signals 
 *
 *************************************************************************/


// Creates transaction composite signals
void
driver_info::transport_assign(const array_base &value, int first, const time &time_value) 
{
  type_info_interface *etype = ((array_info*)type)->element_type;
  // First calculate transaction time and remove time
  time tr_time = kernel.get_sim_time() + time_value;

  if (etype->scalar()) {
    //******************************************************************************
    // If the elements of the array are scalar then ....  First get
    // index range of scalar elements that are target of the assigment
    //******************************************************************************
    int length = value.info->length; // Determine length of assignment
    int element_size = etype->size; // Get size of an array element
    first -= index_start;

    if (length != value.info->length) // check whether length of target and length of source match
      error("Array out of bounds!", __FILE__, __LINE__);

    // Now create a separate transaction for each scalar element of
    // the target
    for (int i = 0, j = first - index_start, k = 0; i < length; i++, j++, k += element_size) {
      // Note, the current driver_info instance does not store the
      // transaction lists but an array of pointers to scalar
      // driver_info instances!
      fqueue<long long int, time> &tr_list = drivers[j]->transactions;
      void *pos = tr_list.start();
      
      // Remove transactions from the list
      while (tr_list.next(pos)) {
	void *npos = tr_list.next(pos);
	if (tr_list.key(npos) >= tr_time) {
	  tr_list.cut_remove(npos);
	  break;
	}
	pos = npos;
      }

      // Get the scalar value from the source array
      void *element_value =  &value.data[k]; //value.info->type_info_interface::element((void*)&value, i);

      // Create a new transactions
      etype->fast_copy(&tr_list.content(tr_list.insert(pos, tr_time)), element_value);
      kernel.add_to_global_transaction_queue(drivers[j], tr_time);
      STATISTICS(kernel.created_transactions_counter++;);
      REPORT(cout << "Transaction (inertial delay) for signal " << signal->instance_name << 
	     "(index=" << i << ") (value=" << type->str(value) << ", time=";
	     cout << L3std_Q8standard_I4time_INFO->str(&tr_time) << ") created." << endl;);
    }

  } else {
    //******************************************************************************
    // If the elements are of a composite type ....
    //******************************************************************************
    // Get number of scalar elements each array element consist of
    int length = value.info->length; // Determine length of assignment
    int step = etype->element_count(); // Number of scalar elements
    // each array element consists of
    int element_size = etype->size; 
    for (int i = first, j = 0, k = 0; k < length; k++, j += element_size, i += step) {
      switch (etype->id) {
      case ARRAY:
	transport_assign((array_base&)value.data[j], i, time_value);
	break;
      case RECORD:
	//scalar_driver(i)->transport_assign((record_base&)value.data[j],
	//i, i + step - 1, time_value, start_time);
	break;
      }
    }
  }
}



// Creates transaction for composite signals
void
driver_info::inertial_assign(const array_base &value, int first, const time &time_value, 
			     const time &start_time) 
{
  type_info_interface *etype = value.info->element_type;
  // First calculate transaction time and remove time
  time tr_time = kernel.get_sim_time() + time_value;
  time rm_time = kernel.get_sim_time() + start_time;

  if (etype->scalar()) {
    //******************************************************************************
    // If the elements of the array are scalar then ....  First get
    // index range of scalar elements that are target of the assigment
    //******************************************************************************
    int length = value.info->length; // Determine length of assignment
    int element_size = etype->size; // Get size of an array element
    first -= index_start;

    // Now create a separate transaction for each scalar element of
    // the target
    for (int i = 0, j = first - index_start, k = 0; i < length; i++, j++, k += element_size) {
      // Note, the current driver_info instance does not store the
      // transaction lists but an array of pointers to scalar
      // driver_info instances!
      fqueue<long long int, time> &tr_list = drivers[j]->transactions;
      void *pos = tr_list.start();
      
      while (tr_list.next(pos) && tr_list.key(tr_list.next(pos)) < rm_time)
	pos = tr_list.next(pos);
      void *start_pos = pos;

      // Get the scalar value from the source array
      void *element_value =  &value.data[k];

      // Perfrom inertial delay mechanism
      void *rpos = NULL;
      time tr_time = kernel.get_sim_time() + time_value;
      while (tr_list.next(pos)) {
	void *npos = tr_list.next(pos);
	if (tr_list.key(npos) >= tr_time) {
	  tr_list.cut_remove(npos);
	  break;
	}
	if (!etype->fast_compare(element_value, &tr_list.content(npos))) {
	  if (rpos)
	    while (rpos != npos)
	      rpos = tr_list.remove(npos);
	  tr_list.remove(npos);
	  rpos = NULL;
	  pos = start_pos;
	} else {
	  rpos = rpos? rpos : npos;
	  pos = npos;
	}
      }

      // Create a new transactions
      etype->fast_copy(&tr_list.content(tr_list.insert(pos, tr_time)), element_value);
      kernel.add_to_global_transaction_queue(drivers[j], tr_time);
      STATISTICS(kernel.created_transactions_counter++;);
      REPORT(cout << "Transaction (inertial delay) for signal " << signal->instance_name << 
	     "(index=" << i << ") (value=" << type->str(value) << ", time=";
	     cout << L3std_Q8standard_I4time_INFO->str(&tr_time) << ") created." << endl;);
    }

  } else {
    //******************************************************************************
    // If the elements are of a composite type ....
    //******************************************************************************
    // Get number of scalar elements each array element consist of
    int length = value.info->length; // Determine length of assignment
    int step = etype->element_count(); // Number of scalar elements
    // each array element consists of
    int element_size = etype->size; 
    for (int i = first, j = 0, k = 0; k < length; k++, j += element_size, i += step) {
      switch (etype->id) {
      case ARRAY:
	inertial_assign((array_base&)value.data[j], i, time_value, start_time);
	break;
      case RECORD:
	//scalar_driver(i)->inertial_assign((record_base&)value.data[j],
	//i, i + step - 1, time_value, start_time);
	break;
      }
    }
  }
}


/*************************************************************************
 *
 * Methods to create transactions for scalar signals 
 *
 *************************************************************************/


// Creates transaction for enumeration signals
void
driver_info::transport_assign(const enumeration &value, const time &time_value) 
{
  do_transport_assignment<enumeration>(*this, value, time_value);
}



// Creates transaction for integer signals
void
driver_info::transport_assign(const integer &value, const time &time_value) 
{
  do_transport_assignment<integer>(*this, value, time_value);
}



// Creates transaction for enumeration signals
void
driver_info::inertial_assign(const enumeration &value, const time &time_value, 
			     const time &start_time) 
{
  do_inertial_assignment<enumeration>(*this, value, time_value, start_time);
}



// Creates transaction for integer signals
void
driver_info::inertial_assign(const integer &value, const time &time_value, 
			     const time &start_time) 
{
  do_inertial_assignment<integer>(*this, value, time_value, start_time);
}



// Creates transaction for enumeration signals
void
driver_info::inertial_assign(const enumeration &value, const time &time_value) 
{
  do_inertial_assignment<enumeration>(*this, value, time_value);
}


// Creates transaction for integer signals
void
driver_info::inertial_assign(const integer &value, const time &time_value) 
{
  do_inertial_assignment<integer>(*this, value, time_value);
}














