#define KERNEL

#include <freehdl/kernel-flags.hh>
#include <freehdl/kernel-process-base.hh>
#include <freehdl/kernel-sig-info.hh>
#include <freehdl/kernel-wait-info.hh>
#include <freehdl/kernel-kernel-class.hh>
#include <freehdl/kernel-handle-info.hh>
#include <freehdl/kernel-driver-info.hh>
#include <freehdl/kernel-reader-info.hh>

// Arguments that are passed in form the command line
int main_argc;
char **main_argv;

// Number of processes
int kernel_class::process_counter; 
// Array of pointers to the processes the model consist of
process_base **kernel_class::process_array;
// Array to store the processes which will be executed next
process_base *kernel_class::processes_to_execute;
// Each simulation cycle is assigned a unique cycle id. The cycle id
// may be used to easily distinguish between different simulation
// cycles. The id is incremented before a new simulation cycle
// starts.
int kernel_class::cycle_id;

// Number of processes which were executed during the simulation. Is
// only enabled if corresponding macro is defined. See "flags.hh"
STATISTICS(int kernel_class::executed_processes_counter;);
// Number of transactions which were created during simulation. Is
// only enabled if corresponding macro is defined. See "flags.hh"
STATISTICS(int kernel_class::created_transactions_counter;);



g_trans_queue kernel_class::global_transaction_queue;


// List of pointers to all process of the model. This list
// is used only at simulation startup and removed afterwards.
list<process_base*> *plist; // List of processes

// List of signals
list<sig_info_base*> signal_list; 

// Basic constructor for the kernel
kernel_class::kernel_class() 
{
  process_counter = 0;
  plist = new list<process_base*>;
  process_array = NULL;
  processes_to_execute = (process_base*)-1;
  automatic_wait_id_counter = 0;
  current_process = NULL;
  cycle_id = 0;
  STATISTICS(executed_processes_counter = 0;);
}


// Elaborates the VHDL mode. During elaboration all components,
// signals and processes are instantiated and linked to the kernel
void
kernel_class::elaborate_model() 
{
  instance_name.push("");

  // Elaborate first component found in component list
  elaborate_component("", "", "", instance_name, "", NULL);
  
  // All processes are instantiated cause we now know the number of processes
  // which are present in the model.
  process_array = new process_base*[process_counter];
  void *pos = plist == NULL ? NULL : plist->first();
  int i = 0;
  while (pos) {
    process_array[i] = plist->content(pos);
    // All processes must be executed at least once at simulation startup.
    // So add each process to the process_to_execute here.
    plist->content(pos)->next_process =  processes_to_execute;
    processes_to_execute = plist->content(pos);
    pos = plist->next(pos);
    i++;
  }
  // Now remove the plist because we do not need the list any more
  delete plist;
  plist = NULL;
  
#ifdef READER_INFO_STATISTICS
  // Determine how many processes are sensitive on one, two, three,
  // ... signals. First create an array of counters and initialize
  // them with 0.
  int *reader_info_counters = (int*)alloca(process_counter * sizeof(int));
  for (int j = 0; j < process_counter; j++)
    reader_info_counters[j] = 0;
  
  // Visit all reader_info instances and get number of processes which
  // are sensitive on the corresponding scalar reader element. Note,
  // only signals which do not have any father are taken into account.
  signal_map::iterator it = signal_table.begin();
  while (it != signal_table.end()) { // Visit all signals
    sig_info_base *signal = (*it).second; // Get sig_info_base pointer
    if (!signal->signal_links.first()) // Does signal has no father?
      for (int j = 0; j < signal->scalar_count; j++)
	// Get number of sensitive processes for each scalar element
	// of the signal
	reader_info_counters[signal->readers[j]->wait_elements.size()]++;
    it++;
  }

  // Plot results.
  cout << "reader_info statistics:" << endl;
  for (int j = 0; j < process_counter; j++)
    if (reader_info_counters[j])
      cout << reader_info_counters[j] << " of length " << j << endl;
#endif

  instance_name.pop();
}


// Calls constructor of an component specified by library, entity, and
// architecture. An empty string "" can be used as wildcard for library,
// entity, or architecture.
void
kernel_class::elaborate_component(const char *library, const char *entity,
				  const char *architecture,
				  name_stack &iname, const char *name,
				  map_list *mlist)
{
  iname.set(name);
  handle function = get_handle(library, entity, architecture);

  if (function == NULL) // Bail out if component not found
    error("Component " + string(entity) + "(" + string(architecture) + ")" + " from library " + 
	  string(library) + " not found in executable", __FILE__, __LINE__);

  // Execute component constructor
  (*function)(iname, mlist);
}


// Create and setup wait_info instances
short 
kernel_class::setup_wait_info(const short wait_id, const sigacl_list &salist, process_base *proc)
{
  wait_info wait_elem(wait_id, proc);
  
  // Declare the current wait_info instance as active. Hence, if only
  // a single wait_info instance is associated with the process, it
  // does not have to be actived in the process code explicitly.
  proc->active_wait_id = wait_id;

  for (int i = 0; i < salist.count; i++) {
    if (salist.list[i].signal->type->scalar())
      // If the signal is scalar then add wait element to reader
      salist.list[i].signal->readers[0]->wait_elements.append(wait_elem);
    else {
      // The signal is a composite signal. Hence, the process is
      // sensitive on each scalar element of the signal which is
      // addressed by the corresponding acl instance. First, get
      // index bounds.
      int start = 0, end;
      salist.list[i].signal->type->acl_to_index(salist.list[i].aclp, start, end);
      reader_info **readers = salist.list[i].signal->readers;
      for (int j = start; j <= end; j++)
	if (readers[j] == NULL)
	  error("Internal error!", __FILE__, __LINE__);
	else
	  readers[j]->wait_elements.append(wait_elem);
    }
  }
  return wait_id;
}  


// Get wait id, create and setup wait_info instances
short
kernel_class::setup_wait_info(const sigacl_list &salist, process_base *proc)
{
  // Test whether a automatic wait id were created for the current
  // process. If not then initialize automatic_wait_id_counter. Note,
  // all automatically created wait ids are less than 0.
  if (proc != current_process) {
    current_process = proc;
    automatic_wait_id_counter = 0;
  }

  // Create and setup wait_info instance.
  return setup_wait_info(--automatic_wait_id_counter, salist, proc);
}



// Executes all processes which are on the processes_to_execute array
void
kernel_class::execute_processes()
{
  process_base *proc = processes_to_execute;
  STATISTICS(int processes_executed = 0;);

  while (proc != (process_base*)-1) {
    REPORT(cout << "Executing process " <<  processes_to_execute[i]->instance_name << 
	   "." << endl;)
    process_base *next_proc = proc->next_process;
    proc->next_process = NULL;
    proc->execute();
    proc = next_proc;
    STATISTICS(processes_executed++;);
  }
  
  STATISTICS(executed_processes_counter += processes_executed;);
  processes_to_execute = (process_base*)-1;
}


bool
kernel_class::assign_next_transactions()
{
  return global_transaction_queue.assign_next_transactions();
}

bool
kernel_class::next_cycle() 
{
  // Increment cycle id counter to separate the current simulation
  // cycle from the previous ones
  cycle_id++; 

  if (global_transaction_queue.next_cycle()) {
    execute_processes();
    return global_transaction_queue.empty(); 
  } else {
    return false;
  }
}


void 
kernel_class::add_process(process_base *proc) 
{
  void *nproc = plist->append(proc);
  process_counter++;
}

void
kernel_class::add_signal(sig_info_base *signal) 
{
  signal_list.append(signal);
  REPORT(cout << "New signal " << signal->instance_name << endl);
}


void *
find(list<driver_info*> &driver_list, void *pos, int start)
{
  if (pos)
    pos = driver_list.next(pos);
  else
    pos = driver_list.first();

  while (pos)
    if (driver_list.content(pos)->index_start == start)
      return pos;
    else
      pos = driver_list.next(pos);

  return NULL;
}


// Returns a pointer to a driver_info instance. This method is 
// called by processes during elaboration to get a driver. All
// signal assignments done by a process are executed via a
// corresponding driver instance.
driver_info *
kernel_class::get_driver(process_base *proc, sig_info_base *signal, acl *a)
{
  int start = 0, end;

  // Get start and end index value of signal
  signal->type->acl_to_index(a, start, end);

  if (start == end) {
    // Only a scalar signal element is target of this driver. Check
    // whether the process has already a driver for this scalar
    // signal.
    void *pos = NULL;
    while ((pos = find(signal->driver_list, pos, start)) != NULL) {
      if (signal->driver_list.content(pos)->process == proc)
	return signal->driver_list.content(pos);
      else
	error("Sorry, resolved signals are not supported yet!", __FILE__, __LINE__);
    }
    if (signal->type->scalar()) {
      // Create a new driver for the signal
      driver_info *driver = new driver_info(proc, signal, start);
      // Add driver to driver list of the signal
      signal->driver_list.append(driver);

      return driver;
    } else {
      driver_info **drivers = new driver_info*[1];
      drivers[0] = new driver_info(proc, signal, start);
      // Add driver to driver list of the signal
      signal->driver_list.append(drivers[0]);

      return new driver_info(proc, signal, start, drivers, 1);
    }

  } else {
    // Create drivers for several scalar elements of the signal.
    int array_size = end - start + 1;
    driver_info **drivers = new driver_info*[array_size];
    for (int i = start, j = 0; i <= end; i++, j++) {
      // Only a scalar signal element is target of this driver. Check
      // whether the process has already a driver for this scalar
      // signal.
      void *pos = NULL;
      while ((pos = find(signal->driver_list, pos, i)) != NULL) {
	if (signal->driver_list.content(pos)->process == proc)
	  break;
	else
	  error("Sorry, resolved signals are not supported yet!", __FILE__, __LINE__);
      }
      if (pos == NULL) {
	drivers[j] = new driver_info(proc, signal, i);
	signal->driver_list.append(drivers[j]);
      }
    }

    return new driver_info(proc, signal, start, drivers, array_size);
  }
}





// Rebuilds driver array. This method is called by processes during
// elaboration e.g. to get an additional driver for another scalar
// instance of an composite signal. All signal assignments done by a
// process are executed via a corresponding driver instance.
void
kernel_class::get_driver(driver_info *driver, process_base *proc, acl *a)
{
  int start = 0, end;
  sig_info_base *signal = driver->signal;

  // Get start and end index value of signal
  signal->type->acl_to_index(a, start, end);

  if (signal->type->scalar()) {
    // if this signal is scalar just check whether it has no multiple
    // sources.
    void *pos = NULL;
    while ((pos = find(signal->driver_list, pos, start)) != NULL) {
      if (signal->driver_list.content(pos)->process != proc)
	error("Sorry, resolved signals are not supported yet!", __FILE__, __LINE__);
    }
    
    return;

  } else {
    int old_start = driver->index_start;
    int old_end = old_start + driver->size - 1;
    int new_start = old_start < start? old_start : start;
    int new_end = old_end > end? old_end : end;
    bool size_changed = old_start != new_start || old_end != new_end;

    driver_info **drivers;
    if (size_changed) {
      // create a new drivers array and copy old pointer to the new array
      drivers = new driver_info*[new_end - new_start + 1];
      for (int j = 0; j <= old_end - old_start; j++)
	drivers[j + old_start - new_start] = driver->drivers[j];
      // delete old drivers array and set the drivers pointer to the
      // newly allocated array
      delete[] driver->drivers; 
      driver->drivers = drivers;
    } else
      drivers = driver->drivers;

    for (int i = start, j = start - new_start; i <= end; i++, j++) {
      // Check whether the process has already a driver for this
      // scalar signal.
      void *pos = NULL;
      while ((pos = find(signal->driver_list, pos, i)) != NULL) {
	if (signal->driver_list.content(pos)->process == proc)
	  break;
	else
	  error("Sorry, resolved signals are not supported yet!", __FILE__, __LINE__);
      }
      if (pos == NULL) {
	drivers[j] = new driver_info(proc, signal, i);
	signal->driver_list.append(drivers[j]);
      }
    }

    driver->index_start = new_start;
    driver->size = new_end - new_start + 1;
  }
}
