#include <iostream.h>
#include <freehdl/kernel-flags.hh>
#include <freehdl/kernel-kernel-class.hh>
#include <freehdl/kernel-name-stack.hh>
#include <freehdl/kernel-map-list.hh>
#include <freehdl/kernel-sig-info.hh>

#ifdef PERFMON_STATISTICS
#include "pcounter.hh"
#endif



// The kernel. Note, there is only a single instantiation of
// the kernel_class
kernel_class kernel;



void print_sim_time(ostream &outp)
{
  lint tval = kernel.get_sim_time();
  int i = 0;

  if (tval != 0) {
    for (i=1; i < L3std_Q8standard_I4time::unit_count; i++)
      if (tval % L3std_Q8standard_I4time::scale[i] != 0) { i--; break; }
    tval = tval / L3std_Q8standard_I4time::scale[i];
  }

  outp << "Simulation time = " << tval << " " << L3std_Q8standard_I4time::units[i] <<
    " + " << kernel.get_delta() << "d" << endl;
  //outp << "Simulation time = " << L3std_Q8standard_I4time_INFO->str(&kernel.get_sim_time()) <<
  //" + " << kernel.get_delta() << "d" << endl;
}


void print_help(ostream &outp)
{
  outp << "Available commands:" << endl;
  outp << "  c <number> : execute cycles = execute <number> simulation cycles" << endl;
  outp << "  n          : next = execute next simulation cycle" << endl;
  outp << "  q          : quit = quit simulation" << endl;
  outp << "  r <time>   : run = execute simulation for <time>" << endl;
  outp << "  s          : show = show signal values" << endl;
#ifdef EVENT_PROFILE
  outp << "  p <file>   : event profile information write to <file>" << endl;
#endif
}


// Simulation control center (SCC)
void scc(istream &inp, ostream &outp) {
  string cmd;

  print_help(outp);

  // Print the current simulation time
  print_sim_time(outp);

  while (1) {
    // Output prompt
    outp << "> "; outp.flush();
    inp >> cmd;

    // Quit simulation
    if (cmd == "q" || cmd == "quit")
      return;

    // Simulate the next cycle
    else if (cmd == "n" || cmd == "next") {
      // Execute next simulation cycle
      kernel.next_cycle();
      // Print simulation time
      print_sim_time(outp);

    // Cycle
    } else if (cmd == "c" || cmd == "cycle") {
      STATISTICS(int counter_start = kernel.executed_processes_counter;);
      STATISTICS(int transactions_counter = kernel.created_transactions_counter;);

      outp << "Number of cycles to execute? "; outp.flush();
      int cycles;
      inp >> cycles;
      outp << "Executing " << cycles << " simulation cycles." << endl;
      while (cycles--) {
	// Execute next simulation cycle
	kernel.next_cycle();
      }
      // Print simulation time
      print_sim_time(outp);
      outp << endl;
      STATISTICS(outp << kernel.executed_processes_counter - counter_start 
		 << " processes were executed.\n";);
      STATISTICS(outp << kernel.created_transactions_counter -  transactions_counter
		 << " transaction were created.\n";);

    // Run
    } else if (cmd == "r" || cmd == "run") {
      outp << "Run simulation for which time span? "; outp.flush();
      int val, i;
      inp >> val;
      char time_unit[10];
      inp >> time_unit;
      outp << endl;

      for (i=0; i < L3std_Q8standard_I4time::unit_count; i++)
	if (!strcmp(time_unit, L3std_Q8standard_I4time::units[i])) break;
      if (i == L3std_Q8standard_I4time::unit_count) {
	outp << "Unknown time unit " << time_unit << endl;
	continue;
      }

      STATISTICS(int counter_start = kernel.executed_processes_counter;);
      STATISTICS(int transactions_counter = kernel.created_transactions_counter;);

      lint time_value = kernel.get_sim_time();
      time_value = time_value + L3std_Q8standard_I4time::scale[i] * (lint)val;
      outp << "Simulating model to time " << val << " " << time_unit << endl;
      
#ifdef PERFMON_STATISTICS
      start_pcounter(main_argc, main_argv);
#endif

      while (kernel.get_sim_time() < time_value) {
	// Execute next simulation cycle
	kernel.next_cycle();
      }

#ifdef PERFMON_STATISTICS
      end_pcounter();
#endif

      // Print simulation time
      print_sim_time(outp);
      outp << endl;
      STATISTICS(outp << kernel.executed_processes_counter - counter_start << 
		 " processes were executed.\n";);
      STATISTICS(outp << kernel.created_transactions_counter - transactions_counter 
		 << " transaction were created.\n";);


    // Print all signals
    } else if (cmd == "s" || cmd == "show") {
      void *pos = signal_list.first();
      while (pos) {
	ostrstream s;
	outp << signal_list.content(pos)->instance_name << " = ";
	signal_list.content(pos)->type->print(s, signal_list.content(pos)->reader_pointer);
	s << '\0';
	outp << s.str() << endl;
	pos = signal_list.next(pos);
      }

#ifdef EVENT_PROFILE
    } else if (cmd == "p" || cmd == "profile") {
      outp << "Print into file: "; outp.flush();
      char filename[80];
      inp >> filename;
      outp << endl;

      kernel.event_report(filename);
#endif
    } else {
      outp << "Unkown command " << cmd << endl;
      print_help(outp);
    }
  }
}


int
main(int argc, char *argv[]) {
  // Store command line arguments
  main_argc = argc;
  main_argv = argv;
  // Elaborate the VHDL model: instantiate all components, signals
  // and processes
  kernel.elaborate_model();
  // Execute processe the first time.
  kernel.execute_processes();
  // Read in command and execute them
  scc(cin, cout);
  
  return 0;
}


int
ttt(int j)
{
  for (int i = 0; i < j; i++)
    delete[] new char[i];
  return 0;
}
