#define KERNEL // Include internal kernel definitions

#include <freehdl/kernel-error.hh>
#include <freehdl/kernel-sig-info.hh>
#include <freehdl/kernel-kernel-class.hh>
#include <freehdl/kernel-reader-info.hh>
#include <freehdl/kernel-driver-info.hh>


// signal_table is a hash tables (map) to associate signal names with
// the corresponding sig_info_base pointers.
signal_map signal_table;


sig_info_base::sig_info_base(name_stack &iname, map_list *mlist,
			     type_info_interface *ty, char dir,
			     resolver_base *res)
{
  instance_name = iname.get_name();
  if (signal_table.find(instance_name) != signal_table.end())
    error(ERROR_DUBLICATE_INSTANCE_NAME, string("Dublicate instance name '") + 
	  instance_name + string("'"));
  signal_table[instance_name] = this; // Store sig_info_base pointer into global hash table
  name = iname.get_top();
  type = ty;
  direction = dir;
  resolver = res;

  scalar_count = type->element_count(); // Get number of scalar elements of the current formal

  list<signal_link*> link_list;
  if (mlist != NULL &&
      mlist->query_signal(link_list, name)) { // Test whether the signal has a father
    if (link_list.size() > 1) // Currently only an entire formal can be mapped! 
      error(ERROR_NOT_SUPPORTED, "Sorry, currently only an entire formal can be mapped!");
    link *connection = new link;
    connection->formal_aclp = NULL;
    connection->actual_aclp = link_list.content(link_list.first())->actual_aclp->clone();
    connection->wrapper_fn_formal = NULL; // Currently not supported
    connection->wrapper_fn_actual = NULL; // Currently not supported
    connection->actual = link_list.content(link_list.first())->actual_signal;
    signal_links.append(connection);

    sig_info_base &actual = *connection->actual;

    // If scalar count < 0 then formal is an unconstrained array type
    // and we must determine the array bounds from the actual
    if (scalar_count < 0) {
      int left, right;
      // Calculate the array bounds from the association list and create a new
      // array_info instance. Note, this instance will be not removed until end
      // of simulation.
      array_info *ainfo_pointer = (array_info*)type;
      mlist->get_port_array_bounds(link_list, name, ainfo_pointer, left, right);
      if (type != NULL) // Free type info instance
	type->remove_ref();
      type = new array_info(ainfo_pointer, left, ainfo_pointer->index_direction, right, -1);
      // Recalculate number of scalar elements 
      scalar_count = type->element_count();
      // Get reader of the signal 
      reader_pointer = actual.type->element(actual.reader_pointer, connection->actual_aclp);

    } else if (type->id == ARRAY) {
      // Create a new array instance with its info an data pointer set
      // to NULL
      array_base *new_array = (array_base*)type->create();
      new_array->set_info((array_info*)type); // Set info instance of array
      
      if (connection->actual_aclp->end())
	// If the entire actual array is mapped then extract the data
	// pointer from the actual reader and assign it to the data
	// pointer of the formal
	new_array->data = ((array_base*)actual.reader_pointer)->data;
      else
	// If a slice of the actual array is mapped then get the
	// corresponding left element of the actual array and assign
	// it to the formal data pointer
	new_array->data = (char*)actual.type->element(actual.reader_pointer, connection->actual_aclp);

      reader_pointer = (void*)new_array;

    } else if (type->id == RECORD) {
      // Not implemented yet!
      error(ERROR_NOT_SUPPORTED, "Sorry, no record ports are supported yet!");
      
    } else
      // Get scalar actual and assign actual pointer to current
      // reader_pointer
      reader_pointer = actual.type->element(actual.reader_pointer, connection->actual_aclp);
  

    // Get reader_info pointers
    readers = new reader_info*[scalar_count];
    int start = 0, end;
    actual.type->acl_to_index(connection->actual_aclp, start, end);
    for (int i = 0, j = start; i < scalar_count; i++, j++)
      readers[i] = actual.readers[j];

  } else { // The signal has no father, hence a new signal is created
    reader_pointer = type->create();
    readers = new reader_info*[scalar_count];
    for (int i = 0; i < scalar_count; i++)
      readers[i] = new reader_info(type->element(reader_pointer, i), type->get_info(i));
  }

  kernel.add_signal(this);
}


sig_info_base::sig_info_base(name_stack &iname, type_info_interface *ty, 
			     char sty, resolver_base *res)
{
  instance_name = iname.get_name();
  if (signal_table.find(instance_name) != signal_table.end())
    error(ERROR_DUBLICATE_INSTANCE_NAME, string("Dublicate instance name '") + 
	  instance_name + string("'"));
  signal_table[instance_name] = this; // Store sig_info_base pointer into global hash table
  name = iname.get_top();
  type = ty;
  sig_type = sty;
  reader_pointer = type->create();
  resolver = res;

  scalar_count = type->element_count(); // Get number of scalar objects
  readers = new reader_info*[scalar_count];
  for (int i = 0; i < scalar_count; i++)
    readers[i] = new reader_info(type->element(reader_pointer,i), type->get_info(i));

  kernel.add_signal(this);
}


sig_info_base::sig_info_base(name_stack &iname, type_info_interface *ty, 
			     char attr, sig_info_base *base_sig, acl *aclp,
			     time delay) 
{
  // Sorry, not supported yet
}


sig_info_base::sig_info_base(name_stack &iname)
{
  // Sorry, not supported yet
}


void
sig_info_base::init_reader(const void *src)
{
  // Set reader value of signal
  type->copy(reader_pointer, src);
  
  // Set value of event and last_ value transaction
  for (int i = 0; i < scalar_count; i++) {
    type_info_interface *etype = type->get_info(i);
    void *element = type->element((void*)src, i);
    etype->fast_copy(&fqueue<long long int, time>::content(readers[i]->last_event_tr_item), element);
    etype->fast_copy(&fqueue<long long int, time>::content(readers[i]->last_value_tr_item), element);
  }
}



