#include <strstream.h>
#include <stdlib.h>
#include <string.h>
#include <freehdl/std-vhdl-types.hh>



/* *************************************************************
 *  Memory management for scalar objects
 * ************************************************************* */

long long int *free_items = NULL;

inline void 
internal_remove(void *src)
{
  *(int*)src = (int)free_items;
  free_items = (long long int*)src;
}


inline void *
internal_alloc()
{
  if (free_items) {
    void *result = free_items;
    free_items = (long long int*)*free_items;
    return result;
  } else
    return malloc(sizeof(long long int));
}



/* *************************************************************
 *  type_info_interface
 * ************************************************************* */

char buffer[1000];

// Not clean!
char *
type_info_interface::str(const void *src)
{
  ostrstream s;
  print(s, src);
  s << '\0';
  strcpy(buffer, s.str());
  return buffer;
}


type_info_interface::type_info_interface(const type_id i, const short s)
{
  id = i;
  size = s;
}


void *
type_info_interface::element(void *src, int i)
{
  switch (id) {
  case RECORD: // Not defined yet
    return src;
    break;
  case ARRAY: 
    {
      // the current object is actually an array_info instance
      array_info &ainfo = *(array_info*)this;
      array_base &abase = *(array_base*)src;
      int subelement_count = ainfo.element_type->element_count();
      int element_index = i / subelement_count;
      if (subelement_count == 1)
	// if the element of the array is a scalar type then
	return &abase.data[element_index * ainfo.element_type->size];
      else
	// the element type is a composite type
	return ainfo.element_type->element(&abase.data[element_index * ainfo.element_type->size], 
					   i - element_index * subelement_count);
      break;
    }
  default: 
    return src;
    break;
  }
}



int 
type_info_interface::acl_to_index(acl *a, int &start, int &end) const
{
  switch (id) {
  case RECORD: // Not defined yet
    return start;
    break;
  case ARRAY: 
    {
      // the current object is actually an array_info instance
      array_info &ainfo = *(array_info*)this;
      int subelement_count = ainfo.element_type->element_count();
      if (a->end()) {
	// If acl ends then return all elements
	end = start + ainfo.length * subelement_count - 1;
	return start;

      } else if (*(int*)a == ACL_RANGE) {
	// Acl determines a range 
	int sindex, eindex;
	if (ainfo.index_direction == to) {
	  sindex = ((int*)a)[1] - ainfo.left_bound;
	  eindex = ((int*)a)[3] - ainfo.left_bound;
	} else {
	  sindex = ainfo.left_bound - ainfo.right_bound - ((int*)a)[1];
	  eindex = ainfo.left_bound - ainfo.right_bound - ((int*)a)[3];
	}
	end = start + (eindex + 1) * subelement_count - 1;
	start += sindex * subelement_count;
	return start;

      } else {
	// A single element is referenced by the acl
	int index = ainfo.index_direction==to? 
	  (*(int*)a - ainfo.left_bound) : 
	  (ainfo.left_bound - ainfo.right_bound - *(int*)a);
	if (subelement_count == 1) {
	  // if an element of the array is a scalar type then
	  start += index;
	  end = start;
	  return start;
	} else {
	  start += index * subelement_count;
	  // the element type is a composite type
	  return ainfo.element_type->acl_to_index(++a, start, end);
	}
      }

      break;
    }
  default:
    end = start;
    return start;
    break;
  }
}


int 
type_info_interface::acl_to_index(acl *a) const
{
  switch (id) {
  case RECORD: // Not defined yet
    return 0;
    break;
  case ARRAY: 
    {
      // the current object is actually an array_info instance
      array_info &ainfo = *(array_info*)this;
      int subelement_count = ainfo.element_type->element_count();

      if (a->end()) {
	return 0;

      } else if (*(int*)a == ACL_RANGE) {
	// Acl determines a range 
	int sindex;
	if (ainfo.index_direction == to)
	  sindex = ((int*)a)[1] - ainfo.left_bound;
	else 
	  sindex = ainfo.left_bound - ainfo.right_bound - ((int*)a)[1];
	
	return sindex * subelement_count;

      } else {
	// A single element is referenced by the acl
	int index = ainfo.index_direction==to? 
	  (*(int*)a - ainfo.left_bound) : 
	  (ainfo.left_bound - ainfo.right_bound - *(int*)a);

	if (subelement_count == 1)
	  // if an element of the array is a scalar type then return
	  return index;
	else
	  // the element type is a composite type
	  return index * subelement_count + ainfo.element_type->acl_to_index(++a);
      }

      break;
    }
  default:
    return 0;
    break;
  }
}


type_info_interface *
type_info_interface::get_info(int i) const {
  switch (id) {
  case RECORD:
    error("Sorry, records are currently not implemented!", __FILE__, __LINE__);
    break;
  case ARRAY: {
    array_info &ainfo = *(array_info*)this;
    if (ainfo.element_type->scalar()) return ainfo.element_type;
    i = i % ainfo.element_type->element_count();
    return ainfo.element_type->get_info(i);
    break;
  }
  default:
    return (type_info_interface*)this;
    break;
  }
}



int 
type_info_interface::get_bounds(int &left, int &right)
{
  switch (id) {
  case INTEGER:
  case ARRAY: {
    integer_info_base &iinfo = *(integer_info_base*)this;
    left = iinfo.left_bound;
    right = iinfo.right_bound;
    return 0;
  }
  case ENUM: {
    enum_info_base &einfo = *(enum_info_base*)this;
    left = einfo.left_bound;
    right = einfo.right_bound;
    return 0;
  }
  default:
    return -1;
  }
}



/* *************************************************************
 *  Integer info base class
 * ************************************************************* */

integer_info_base::
integer_info_base(const int le, const int ri, 
		  const int lo, const int hi) : 
  type_info_interface(INTEGER, INTEGER_SIZE) {
    left_bound = le; 
    right_bound = ri;
    low_bound = lo;
    high_bound = hi;
}


void *
integer_info_base::create() {
  integer *p = (integer*)internal_alloc();
  *p = left_bound;
  return p;
}


void *
integer_info_base::clone(const void *src) {
  integer *p = (integer*)internal_alloc();
  *p = *((integer*)src);
  return p;
}


void 
integer_info_base::init(void *src) {
  *((integer*)src) = left_bound;
}


void *
integer_info_base::copy(void *dest, const void *src) {
  *((integer*)dest) = *((integer*)src);
  return dest;
}


bool
integer_info_base::compare(const void *src1, const void *src2) {
  return *((integer*)src1) == *((integer*)src2);
}


bool 
integer_info_base::assign(void *dest, const void *src) {
  int new_value = *((integer*)src);
  bool event = new_value != *((integer*)dest);
  *((integer*)dest) = new_value;
  return event;
}


void
integer_info_base::remove(void *src)
{
  internal_remove(src);
}


void
integer_info_base::print(ostrstream &str, const void *src) {
  str << *((integer*)src);
}


/* *************************************************************
 *  Float  info base class
 * ************************************************************* */

float_info_base::
float_info_base(const double le, const double ri, 
		const double lo, const double hi) : 
  type_info_interface(FLOAT, FLOAT_SIZE) {
    left_bound = le;
    right_bound = ri;
    low_bound = lo;
    high_bound = hi;
}


void *
float_info_base::create() {
  floatingpoint *p = (floatingpoint*)internal_alloc();
  *p = left_bound;
  return p;
}


void *
float_info_base::clone(const void *src) {
  floatingpoint *p = (floatingpoint*)internal_alloc();
  *p = *((floatingpoint*)src);
  return p;
}


void *
float_info_base::copy(void *dest, const void *src) {
  *((floatingpoint*)dest) = *((floatingpoint*)src);
  return dest;
}


void 
float_info_base::init(void *src) {
  *((floatingpoint*)src) = left_bound;
}


bool
float_info_base::compare(const void *src1, const void *src2) {
  return *((floatingpoint*)src1) == *((floatingpoint*)src2);
}


bool 
float_info_base::assign(void *dest, const void *src) {
  double new_value = *((floatingpoint*)src);
  bool event = new_value != *((floatingpoint*)dest);
  *((floatingpoint*)dest) = new_value;
  return event;
}


void
float_info_base::remove(void *src)
{
  internal_remove(src);
}


void
float_info_base::print(ostrstream &str, const void *src) {
  str << *((floatingpoint*)src);
}


/* *************************************************************
 *  Enum info base class
 * ************************************************************* */

enum_info_base::
enum_info_base(const int le, const int ri, const char **val) :
  type_info_interface(ENUM, ENUM_SIZE) {
    left_bound = le;
    right_bound = ri;
    length = ri - le;
    values = val;
}


void *
enum_info_base::create() {
  enumeration *p = (enumeration*)internal_alloc();
  *p = left_bound;
  return p;
}


void *
enum_info_base::clone(const void *src) {
  enumeration *p = (enumeration*)internal_alloc();
  *p = *((enumeration*)src);
  return p;
}


bool
enum_info_base::compare(const void *src1, const void *src2) {
  return *((enumeration*)src1) == *((enumeration*)src2);
}


void *
enum_info_base::copy(void *dest, const void *src) {
  *((enumeration*)dest) = *((enumeration*)src);
  return dest;
}


void 
enum_info_base::init(void *src) {
  *((enumeration*)src) = left_bound;
}


bool 
enum_info_base::assign(void *dest, const void *src) {
  enumeration new_value = *((enumeration*)src);
  bool event = new_value != *((enumeration*)dest);
  *((enumeration*)dest) = new_value;
  return event;
}


void
enum_info_base::remove(void *src)
{
  internal_remove(src);
}


void
enum_info_base::print(ostrstream &str, const void *src) {
  str << values[*((enumeration*)src)];
}



/* *************************************************************
 *  Physical info base class
 * ************************************************************* */

physical_info_base::
physical_info_base(const long long int le, const long long int ri, 
		   const long long int lo, const long long int hi,
		   const char **un, const long long int *sc, 
		   int uc) : 
  type_info_interface(PHYSICAL, PHYSICAL_SIZE) {
    left_bound = le;
    right_bound = ri;
    low_bound = lo;
    high_bound = hi;
    units = un;
    scale = sc;
    unit_count = uc;
}



void *
physical_info_base::create() {
  physical *p = (physical*)internal_alloc();
  *p = left_bound;
  return p;
}


void *
physical_info_base::clone(const void *src) {
  physical *p = (physical*)internal_alloc();
  *p = *((physical*)src);
  return p;
}


void *
physical_info_base::copy(void *dest, const void *src) {
  *((physical*)dest) = *((physical*)src);
  return dest;
}


void 
physical_info_base::init(void *src) {
  *((physical*)src) = left_bound;
}


bool
physical_info_base::compare(const void *src1, const void *src2) {
  return *((physical*)src1) == *((physical*)src2);
}


bool 
physical_info_base::assign(void *dest, const void *src) {
  long new_value = *((physical*)src);
  bool event = new_value != *((physical*)dest);
  *((physical*)dest) = new_value;
  return event;
}


void
physical_info_base::remove(void *src)
{
  internal_remove(src);
}


void
physical_info_base::print(ostrstream &str, const void *src) {
  str << *((physical*)src) << " " << units[0];
}




/* *************************************************************
 *  Array info class
 * ************************************************************* */

array_info::array_info(type_info_interface *et, type_info_interface *it, int rc) : 
  type_info_interface(ARRAY, array_base::size())
{
  element_type = et;
  element_type->add_ref();
  index_type = it;
  length = -1; /* indicate unconstrained array */
  it->get_bounds(left_bound, right_bound); /* Preload bounds */
  index_direction = left_bound <= right_bound ? to : downto;
  ref_counter = rc;
}


array_info::array_info(array_info *base, int le, range_direction r, int ri, int rc) :
  type_info_interface(ARRAY, array_base::size())
{
  ref_counter = rc;
  set(base->element_type, base->index_type, le, r, ri);
}


/* Constructor to create an info instance where the left bound is
 * determined by base and the right bound is derived from len */
array_info::array_info(array_info *base, int len, int rc=0) :
  type_info_interface(ARRAY, array_base::size())
{
  ref_counter = rc;
  int right;
  int le = base->index_type->get_bounds(left_bound, right);
  if (left_bound < right) {
    right_bound = left_bound + len - 1;
    index_direction = to;
    if (right_bound > right) error(ERROR_ARRAY_INDEX_OUT_OF_BOUNDS);
  } else {
    right_bound = left_bound - len + 1;
    index_direction = downto;
    if (right_bound < right) error(ERROR_ARRAY_INDEX_OUT_OF_BOUNDS);
  }
  length = len;
  element_type = base->element_type;
  index_type = base->index_type;
}


void array_info::print(ostrstream &str, const void *src)
{
  int length = ((array_base*)src)->info->length;
  type_info_interface *einfo = ((array_base*)src)->info->element_type;
  char *data = ((array_base*)src)->data;

  str << "(";
  for (int i = 0; i < length; i++) {
    if (i) str << ",";
    einfo->print(str, &data[i * einfo->size]);
  }
  str << ")";
}



void *
array_info::element(void *src, acl *a)
{
  if (a == NULL || a->end()) return src;
  array_info &ainfo = *(array_info*)this;
  array_base &abase = *(array_base*)src;
  return ainfo.element_type->element(&abase.data[a->get() * ainfo.element_type->size], 
				    a->next());
}


void 
array_info::init(void *src)
{
  array_base &abase = *(array_base*)src;
  abase.set_info(this);

  int size = length * element_type->size;
  int step = element_type->size;

  /* Allocate memory for the data */
  abase.data = new char[size];
  
  /* Initialize each element of the array */
  for (int i = 0; i < size; i+=step)
    element_type->init(&abase.data[i]);
}


void *
array_info::create()
{
  /* Get memory for array object */
  array_base &abase = *(array_base*)internal_alloc();
  abase.info = NULL;
  abase.data = NULL;
  /* Initialize object */
  init(&abase);

  return (void*)&abase;
}



void 
array_info::clear(void *src)
{
  array_base &abase = *(array_base*)src;
  array_info &ainfo = *abase.info;
  
  /* The current array info intance is not referenced by this
   * object any more. Hence, decrement its reference counter */
  ainfo.remove_ref();

  /* Clear each element of the array if they are not scalar! */
  if (!element_type->scalar()) {
    int size = ainfo.length * ainfo.element_type->size;
    int step = ainfo.element_type->size;
    for (int i = 0; i < size; i+=step)
      element_type->clear(&abase.data[i]);
  }

  /* Remove array memory */
  delete[] abase.data;
}


void 
array_info::remove(void *src) 
{
  /* Clear/Remove data structures of the array instance */
  clear(src);
  internal_remove(src);
}


void *
array_info::copy(void *dest, const void *src)
{
  array_base &adest = *(array_base*)dest;
  array_base &asrc = *(array_base*)src;
  
  /* Check whether array bounds are compatible */
  if (adest.info != asrc.info && adest.info->length != asrc.info->length)
    error(ERROR_INCOMPATIBLE_ARRAYS);

  /* Copy the data part of the arrays */
  int element_size = adest.info->element_type->size;
  int length = adest.info->length; 
  char *ps = asrc.data, *pd = adest.data; 
  for (int i = 0; i < length; i++) {
    adest.info->element_type->copy(pd, ps);
    ps += element_size;
    pd += element_size;
  }

  return dest;
}
