#include <freehdl/vaul.h>
#include "v2cc-chunk.h"
#include <strstream>
#include <algo.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>
#include <unistd.h>
#include <math.h>

#include "mapping.h"
#include "v2cc-util.h"



// Used to generate error messages
extern vaul_error_printer codegen_error;



// Print out an internal error message and stop compilation if "stop"
// is true
extern void print_internal_error(char *file, int line, bool stop);

#define internal_error(b) print_internal_error(__FILE__, __LINE__, b)



// ******************************************************************************************
// Name: m_constant_fold, generic function
//
// Description: performs constant folding optimizations on nodes
//
// Return value: returns number of errors found during folding
//
// ******************************************************************************************

int
m_constant_fold (pIIR_EntityDeclaration e, RegionStack &rstack)
{
  if (optimized(e) & CONST_FOLD) return 0; else  optimized(e) |= CONST_FOLD;

  int error_count = 0;
  rstack.push(e);

  pIIR_InterfaceList ports = e->port_clause, generics = e->generic_clause;

  // Explore initial values of port signals
  for (pIIR_InterfaceList dl = ports; dl; dl = dl->rest) {
    pIIR_InterfaceDeclaration decl = (pIIR_InterfaceDeclaration)dl->first;
    if (decl->initial_value &&
	decl->initial_value->is(IR_EXPRESSION))
       error_count += constant_fold(decl->initial_value, rstack);
  }
  
  // Explore initial values of generics
  for (pIIR_InterfaceList dl = generics; dl; dl = dl->rest) {
    pIIR_InterfaceDeclaration decl = (pIIR_InterfaceDeclaration)dl->first;
    if (decl->initial_value)
      error_count += constant_fold(decl->initial_value, rstack);
  }

  // Explore entity declarations (not ports and generics; those are
  // handled above)
  if (e->declarations)
    error_count += constant_fold(e->declarations, rstack);

  // Explore concurrent statements of entity
  if (e->entity_statement_part)
    error_count += constant_fold(e->entity_statement_part, rstack);

  rstack.pop();

  return error_count;
}


int
m_constant_fold (pIIR_DeclarationList dl, RegionStack &rstack)
{
  if (optimized(dl) & CONST_FOLD) return 0; else  optimized(dl) |= CONST_FOLD;

  int error_count = 0;
  for (pIIR_DeclarationList d = dl; d; d = d->rest)
    if (d->first->is(IR_OBJECT_DECLARATION)) {
      // Object declaration
      pIIR_ObjectDeclaration od = pIIR_ObjectDeclaration(d->first);
      // Explore initial value of declarations
      if (od->initial_value &&
	  od->initial_value->is(IR_EXPRESSION))
	error_count += constant_fold(od->initial_value, rstack);
      
    } else if (d->first->is(IR_TYPE_DECLARATION)) {
      // Type declaration
      pIIR_TypeDeclaration tdecl = pIIR_TypeDeclaration(d->first);
      error_count += constant_fold(tdecl->type, rstack);
    }

  return error_count;
}


int
m_constant_fold (pIIR_Type st, RegionStack &rstack)
{
  if (optimized(st) & CONST_FOLD) return 0; else  optimized(st) |= CONST_FOLD;

  return 0;
}


int
m_constant_fold (pIIR_EnumerationType et, RegionStack &rstack)
{
  if (optimized(et) & CONST_FOLD) return 0; else  optimized(et) |= CONST_FOLD;

  return 0;
}

int
m_constant_fold (pIIR_ArchitectureDeclaration a, RegionStack &rstack)
{
  if (optimized(a) & CONST_FOLD) return 0; else  optimized(a) |= CONST_FOLD;

  int error_count = 0;
  rstack.push(a);

  // Explore architecture declarations 
  error_count += constant_fold(a->declarations, rstack);

  // Explore concurrent statements of entity
  error_count += constant_fold(a->architecture_statement_part, rstack);

  rstack.pop();
  return error_count;
}


int
m_constant_fold (pIIR_ConcurrentStatementList csl, RegionStack &rstack)
{
  if (optimized(csl) & CONST_FOLD) return 0; else  optimized(csl) |= CONST_FOLD;

  int error_count = 0;
  while (csl)
    {
      error_count += constant_fold(csl->first, rstack);  
      csl = csl->rest;
    }

  return error_count;
}


int
m_constant_fold (pIIR_ProcessStatement p, RegionStack &rstack)
{
  if (optimized(p) & CONST_FOLD) return 0; else  optimized(p) |= CONST_FOLD;

  int error_count = 0;
  rstack.push(p);

  // Explore local declarations of process
  if (p->declarations != NULL)
    error_count += constant_fold(p->declarations, rstack);

  // explore and check sequential statements
  if (p->process_statement_part != NULL)
    error_count += constant_fold(p->process_statement_part, rstack);

  rstack.pop();

  return error_count;
}




int
m_constant_fold (pIIR_PackageDeclaration p, RegionStack &rstack)
{
  if (optimized(p) & CONST_FOLD) return 0; else  optimized(p) |= CONST_FOLD;

  int error_count = 0;
  rstack.push(p);
  rstack.pop();

  return error_count;
}

int
m_constant_fold (pIIR_PackageBodyDeclaration pb, RegionStack &rstack)
{
  if (optimized(pb) & CONST_FOLD) return 0; else  optimized(pb) |= CONST_FOLD;

  int error_count = 0;
  rstack.push(pb);
  rstack.pop();

  return error_count;
}


int
m_constant_fold (pIIR_ConfigurationDeclaration c, RegionStack &rstack)
{
  return 0;
}

int
m_constant_fold (pIIR_ComponentDeclaration c, RegionStack &rstack)
{
  return 0;
}


int
m_constant_fold(pIIR_SequentialStatementList sl, RegionStack &rstack)
{
  if (optimized(sl) & CONST_FOLD) return 0; else  optimized(sl) |= CONST_FOLD;

  int error_count = 0;
  while (sl) {
    error_count += constant_fold(sl->first, rstack);
    sl = sl->rest;
  }

  return error_count;
}


int
m_constant_fold(pIIR_CaseStatement cs, RegionStack &rstack)
{
  if (optimized(cs) & CONST_FOLD) return 0; else  optimized(cs) |= CONST_FOLD;

  int error_count = 0;
  // Explore case expression
  error_count += constant_fold(cs->expression, rstack);

  for (pIIR_CaseStatementAlternativeList al = cs->case_statement_alternatives; al; al = al->rest) {
    pIIR_CaseStatementAlternative a = al->first;

    // Analyze choices
    for (pIIR_ChoiceList cl = a->choices; cl; cl = cl->rest)
      error_count += constant_fold(cl->first, rstack);

    // Explore sequential statements
    error_count += constant_fold(a->sequence_of_statements, rstack);
  }
}


int
m_constant_fold(pIIR_IfStatement ifs, RegionStack &rstack)
{
  if (optimized(ifs) & CONST_FOLD) return 0; else  optimized(ifs) |= CONST_FOLD;

  int error_count = 0;
  // Explore condition expression
  error_count += constant_fold(ifs->condition, rstack);

  // Explore then branch
  error_count += constant_fold(ifs->then_sequence, rstack);

  // Explore else branch (if present)
  if (ifs->else_sequence != NULL)
    error_count += constant_fold(ifs->else_sequence, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_ForLoopStatement fs, RegionStack &rstack)
{
  if (optimized(fs) & CONST_FOLD) return 0; else  optimized(fs) |= CONST_FOLD;

  int error_count = 0;
  // Explore iteration range
  error_count += constant_fold(((pIIR_ScalarSubtype)fs->iterator->subtype)->range, rstack);

  // Explore loop body
  error_count += constant_fold(fs->sequence_of_statements, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_WaitStatement ws, RegionStack &rstack)
{
  if (optimized(ws) & CONST_FOLD) return 0; else  optimized(ws) |= CONST_FOLD;

  int error_count = 0;
  // Explore explicit sensitivity list
  if (ws->sensitivity_list)
    for (pIIR_ExpressionList el = ws->sensitivity_list; el; el = el->rest)
      error_count += constant_fold(el->first, rstack);

  // Explore condition clause
  if (ws->condition_clause)
    error_count += constant_fold(ws->condition_clause, rstack);

  // Explore timeout expression
  if (ws->timeout_clause)
    error_count += constant_fold(ws->timeout_clause, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_SignalAssignmentStatement sas, RegionStack &rstack)
{
  if (optimized(sas) & CONST_FOLD) return 0; else  optimized(sas) |= CONST_FOLD;

  int error_count = 0;
  error_count += constant_fold(sas->target, rstack);

  // Explore reject expression
  if (sas->reject_time_expression)
    error_count += constant_fold(sas->reject_time_expression, rstack);

  // Explore waveforms
  for (pIIR_WaveformList wl = sas->waveform; wl; wl = wl->rest)
    error_count += constant_fold(wl->first->value, rstack);

  return error_count;
}



int
m_constant_fold(pIIR_VariableAssignmentStatement vas, RegionStack &rstack)
{
  if (optimized(vas) & CONST_FOLD) return 0; else  optimized(vas) |= CONST_FOLD;

  int error_count = 0;
  error_count += constant_fold(vas->target, rstack);
  error_count += constant_fold(vas->expression, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_ComponentInstantiationStatement cs, RegionStack &rstack)
{
  if (optimized(cs) & CONST_FOLD) return 0; else  optimized(cs) |= CONST_FOLD;

  int error_count = 0;
  for (pIIR_AssociationList al = cs->port_map_aspect; al; al = al->rest) {
    pIIR_AssociationElement ae = al->first;

    error_count += constant_fold(ae->actual, rstack);
  }

  return error_count;
}


int
m_constant_fold(pIIR_ProcedureCallStatement pc, RegionStack &rstack)
{
  if (optimized(pc) & CONST_FOLD) return 0; else  optimized(pc) |= CONST_FOLD;

  int error_count = 0;
  for (pIIR_InterfaceList il = pc->procedure->interface_declarations; il; il = il->rest) {
    pIIR_InterfaceDeclaration  idecl = il->first;
    // Find actual that corresponds to formal
    pIIR_AssociationElement a = find_matching_actual(pc->actual_parameter_part, il->first);
    if (a != NULL && a->actual != NULL && !a->actual->is(IR_OPEN_EXPRESSION))
      error_count += constant_fold(a->actual, rstack);
    else
      // the parameter is left open. Hence, the default value will be used instead. 
      // explore and check default value if present
      error_count += constant_fold(idecl->initial_value, rstack);	
  }

  return error_count;
}


int
m_constant_fold(pIIR_AttrSigFunc ase, RegionStack &rstack)
{
  if (optimized(ase) & CONST_FOLD) return 0; else  optimized(ase) |= CONST_FOLD;

  int error_count = constant_fold(ase->signal, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_PhysicalLiteral le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  // Currently physical literals are not folded!
  return error_count;
}


int
m_constant_fold(pIIR_ArrayLiteralExpression le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  // Currently array literals are not folded!
  return error_count;
}


int
m_constant_fold(pIIR_EnumLiteralReference le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  folded_value(le).long_value = le->value->enum_pos;
  constant_fold(le->value, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_CharacterLiteral le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  folded_value(le).long_value = (long long int)(le->text.to_chars())[1];
  return error_count;
}


int
m_constant_fold(pIIR_StringLiteral le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  // String literals are currently not folded
  return error_count;
}



int
m_constant_fold(pIIR_EnumerationLiteral el, RegionStack &rstack)
{
  if (optimized(el) & CONST_FOLD) return 0; else  optimized(el) |= CONST_FOLD;

  folded_value(el).long_value = lint(el->enum_pos);
  valid_folded_value(el) = true;

  return 0;
}


int
m_constant_fold(pIIR_IntegerLiteral le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  folded_value(le).long_value = string_to_i(le->text.to_chars());
  return error_count;
}


int
m_constant_fold(pIIR_FloatingPointLiteral le, RegionStack &rstack)
{
  if (optimized(le) & CONST_FOLD) return 0; else  optimized(le) |= CONST_FOLD;

  int error_count = 0;
  folded_value(le).double_value = string_to_d(le->text.to_chars());
  return error_count;
}


int
m_constant_fold(pIIR_AbstractLiteralExpression ale, RegionStack &rstack)
{
  if (optimized(ale) & CONST_FOLD) return 0; else  optimized(ale) |= CONST_FOLD;

  int error_count = 0;
  constant_fold(ale->value, rstack);
  folded_value(ale).long_value = folded_value(ale->value).long_value;
  return error_count;
}



// function is used to fold two scalar values
template<class T, class A>
T perform_operation(pIIR_FunctionCall fc, A a)
{
  extern map_string2 unary_operator_macro;
  string op_str = fc->function->declarator->text.to_chars();

  if (op_str == "\"not\"")
    return T(int(a)? 0 : 1);
  else if (op_str == "\"+\"")
    return T(a);
  else if (op_str == "\"-\"")
    return T(A(-1) * a);
  else
    internal_error(true);
}

// Calculate power for integer parameters.
lint
calculate_pow(pIIR_FunctionCall fc, lint a, lint b)
{
  if (b < 0) return 0; 
  if (b == 1) return 1;
  lint value = a;
  while (--b) {
    value *= a;
    // In case of an overflow or underflow bail out
    if (value > INTEGER_MAX) 
      return lint(INTEGER_MAX) + 1;
    if (value < INTEGER_MIN) 
      return lint(INTEGER_MIN) - 1;
  }
  return value;
}

// Calculate power for double/integer parameters.
double
calculate_pow(pIIR_FunctionCall fc, double a, lint b)
{
  return pow(a,double(int(b)));
}


// function is used to fold two scalar values
template<class T, class A, class B>
T perform_operation(pIIR_FunctionCall fc, A a, B b)
{
  extern map_string2 binary_operator_macro;
  string op_str = fc->function->declarator->text.to_chars();

  if (op_str == "\"and\"")
    return T(int(a) & int(b)? 1 : 0);
  else if (op_str == "\"or\"")
    return T(int(a) | int(b)? 1 : 0);
  else if (op_str == "\"xor\"")
    return T(int(a) ^ int(b) ? 1 : 0);
  else if (op_str == "\"xnor\"")
    return T(1 ^ (int(a) ^ int(b)) ? 1 : 0);
  else if (op_str == "\"nand\"")
    return T(1 ^ (int(a) & int(b)) ? 1 : 0);
  else if (op_str == "\"nor\"")
    return T(1 ^ (int(a) | int(b)) ? 1 : 0);
  else if (op_str == "\"=\"")
    return T((a == b)? 1 : 0);
  else if (op_str == "\">\"")
    return T((a > b)? 1 : 0);
  else if (op_str == "\">=\"")
    return T((a > b)? 1 : 0);
  else if (op_str == "\"<=\"")
    return T((a <= b)? 1 : 0);
  else if (op_str == "\"<\"")
    return T((a < b)? 1 : 0);
  else if (op_str == "\"/=\"")
    return T((a != b)? 1 : 0);
  else if (op_str == "\"=\"")
    return T((a == b)? 1 : 0);
  else if (op_str == "\"+\"")
    return T(a + b);
  else if (op_str == "\"-\"")
    return T(a - b);
  else if (op_str == "\"*\"")
    return T(a * b);
  else if (op_str == "\"/\"")
    return T(a / b);
  else if (op_str == "\"mod\"")
    internal_error(true);
  else if (op_str == "\"rem\"")
    internal_error(true);
  else if (op_str == "\"**\"") {
    return T(calculate_pow(fc, a, b));
  } else
    internal_error(true);
}


int
m_constant_fold(pIIR_FunctionCall fc, RegionStack &rstack)
{
  if (optimized(fc) & CONST_FOLD) return 0; else  optimized(fc) |= CONST_FOLD;

  int error_count = 0;
  int n_args = 0;
  int try_function_folding = true;

  // Try to fold function parameters
  for (pIIR_AssociationList al = fc->parameter_association_list; al; al = al->rest) {
    error_count += constant_fold(al->first->actual, rstack);
    if (al->first->actual->static_level != IR_LOCALLY_STATIC)
      try_function_folding = false;
    n_args++;
  }

  // Return if not all prameters are locally static!
  if (!try_function_folding)
    return 0;

  // Now, try to fold function call
  switch (get_operator_type(fc->function)) { // Analyze function call type

  case STD_OP: // The function call is a standard operator call

    if (n_args == 1) {
      //**********************************************************************
      // Function has one parameter
      //**********************************************************************
      pIIR_Type base_type = get_base_type(fc->parameter_association_list->first->actual->subtype);
      if (base_type->is(IR_INTEGER_TYPE) ||
	  base_type->is(IR_ENUMERATION_TYPE)) {
	lint arg = folded_value(fc->parameter_association_list->first->actual).long_value;
	folded_value(fc).long_value = perform_operation<lint, lint>(fc, arg);

      } else if (base_type->is(IR_FLOATING_TYPE)) {
	double arg = folded_value(fc->parameter_association_list->first->actual).double_value;
	folded_value(fc).double_value = perform_operation<double, double>(fc, arg);

      } else
	internal_error(true);

    } else if (n_args == 2) {
      //**********************************************************************
      // Function has two parameters
      //**********************************************************************
      pIIR_Type base_type1 = get_base_type(fc->parameter_association_list->first->actual->subtype);
      pIIR_Type base_type2 = get_base_type(fc->parameter_association_list->rest->first->actual->subtype);
      pIIR_Type result_type = get_base_type(fc->subtype);
      
      if (base_type1->is(IR_ENUMERATION_TYPE) ||
	  base_type1->is(IR_INTEGER_TYPE)) {
	// Bail out if the second parameter is not an enumeration or
	// integer type
	if (!(base_type2->is(IR_ENUMERATION_TYPE) || 
	      base_type2->is(IR_INTEGER_TYPE)))
	  internal_error(true);

	lint arg1 = folded_value(fc->parameter_association_list->first->actual).long_value;
	lint arg2 = folded_value(fc->parameter_association_list->rest->first->actual).long_value;
	lint result = perform_operation<lint, lint, lint>(fc, arg1, arg2);
	// Check whether result is ok
	if (result > INTEGER_MAX || result < INTEGER_MIN) {
	  result = result > 0? INTEGER_MAX : INTEGER_MIN;
	  codegen_error.error("%:error: overflow/underflow in %n.", fc, fc);
	  error_count++;
	}
	// Store result
	folded_value(fc).long_value = result;
	
      } else if (base_type1->is(IR_FLOATING_TYPE)) {
	if (result_type->is(IR_ENUMERATION_TYPE)) {
	  // Compare operation
	  double arg1 = folded_value(fc->parameter_association_list->first->actual).double_value;
	  double arg2 = folded_value(fc->parameter_association_list->rest->first->actual).double_value;
	  lint result = perform_operation<lint, double, double>(fc, arg1, arg2);
	  // Check whether result is ok
	  if (result > INTEGER_MAX || result < INTEGER_MIN) {
	    result = result > 0? INTEGER_MAX : INTEGER_MIN;
	    codegen_error.error("%:error: overflow/underflow in %n.", fc, fc);
	    error_count++;
	  }
	  // Store result
	  folded_value(fc).long_value = result;

	} else if (base_type2->is(IR_INTEGER_TYPE)) {
	  // Power
	  double arg1 = folded_value(fc->parameter_association_list->first->actual).double_value;
	  lint arg2 = folded_value(fc->parameter_association_list->rest->first->actual).long_value;
	  double result = perform_operation<lint, double, lint>(fc, arg1, arg2);
	  // Check whether result is ok
	  if (result == HUGE_VAL || result == -HUGE_VAL) {
	    codegen_error.error("%:error: overflow/underflow in %n.", fc, fc);
	    error_count++;
	  }
	  // Store result
	  folded_value(fc).double_value = result;

	} else if (base_type2->is(IR_FLOATING_TYPE)) {
	  // Ramining operations
	  double arg1 = folded_value(fc->parameter_association_list->first->actual).double_value;
	  double arg2 = folded_value(fc->parameter_association_list->rest->first->actual).double_value;
	  lint result = perform_operation<lint, double, double>(fc, arg1, arg2);
	  // Check whether result is ok
	  if (result > INTEGER_MAX || result < INTEGER_MIN) {
	    result = result > 0? INTEGER_MAX : INTEGER_MIN;
	    codegen_error.error("%:error: overflow/underflow in %n.", fc, fc);
	    error_count++;
	  }
	  // Store result
	  folded_value(fc).long_value = result;

	} else
	  internal_error(true);
      }
    }

    valid_folded_value(fc) = true; // The node has been optimized
      
    break;

  case BASIC_OP: // The function is a operator defined in an IEEE
    // library. Currently no constant folding is performed for this class
    // of operators
  case USER_OP: // The function is an user defined operator call
  case NO_OP: // A ordinary function call (no operator)
    break;
  }

  return error_count;
}



int
m_constant_fold(pIIR_SimpleReference sr, RegionStack &rstack)
{
  if (optimized(sr) & CONST_FOLD) return 0; else  optimized(sr) |= CONST_FOLD;
  // Return if the expression is not locally static or does not
  // reference a constant
  if (sr->static_level != IR_LOCALLY_STATIC ||
      !sr->object->is(IR_CONSTANT_DECLARATION))
    return 0;

  pIIR_ConstantDeclaration cdecl = pIIR_ConstantDeclaration(sr->object);
  int error_count = 0;
  // Return if constant does not have an initial value otherwise
  // constant fold initial value
  if (cdecl->initial_value == NULL) 
    return 0;
  else
    error_count = constant_fold(cdecl->initial_value, rstack);

  pIIR_Type base_type = get_base_type(sr->subtype); // Get base type
  
  if (base_type->is(IR_INTEGER_TYPE) ||
      base_type->is(IR_ENUMERATION_TYPE)) {
    folded_value(sr).long_value = folded_value(cdecl->initial_value).long_value;
    valid_folded_value(sr) = true;
  } if (base_type->is(IR_FLOATING_TYPE)) {
    folded_value(sr).double_value = folded_value(cdecl->initial_value).double_value;
    valid_folded_value(sr) = true;
  }

  return 0;
}



int
m_constant_fold(pIIR_ArrayReference ar, RegionStack &rstack)
{
  if (optimized(ar) & CONST_FOLD) return 0; else  optimized(ar) |= CONST_FOLD;

  int error_count = constant_fold(ar->array, rstack);
  for (pIIR_ExpressionList el = ar->indices; el; el = el->rest)
    error_count += constant_fold(el, rstack);

  return error_count;
}



int
m_constant_fold(pIIR_ArrayRange ar, RegionStack &rstack)
{
  if (optimized(ar) & CONST_FOLD) return 0; else  optimized(ar) |= CONST_FOLD;

  int error_count = constant_fold(ar->array, rstack);
  error_count += constant_fold(ar->index, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_ExplicitRange er, RegionStack &rstack)
{
  if (optimized(er) & CONST_FOLD) return 0; else  optimized(er) |= CONST_FOLD;

  int error_count = constant_fold(er->left, rstack);
  error_count += constant_fold(er->right, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_OthersIndexedAssociation ia, RegionStack &rstack)
{
  int error_count = constant_fold(ia->value, rstack);
  return error_count;
}


int
m_constant_fold(pIIR_RangeIndexedAssociation ia, RegionStack &rstack)
{
  if (optimized(ia) & CONST_FOLD) return 0; else  optimized(ia) |= CONST_FOLD;

  int error_count = constant_fold(ia->value, rstack);
  error_count += constant_fold(ia->index_range, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_SingleIndexedAssociation ia, RegionStack &rstack)
{
  if (optimized(ia) & CONST_FOLD) return 0; else  optimized(ia) |= CONST_FOLD;

  int error_count = 0;
  if (ia->index != NULL)
    error_count += constant_fold(ia->index, rstack);

  error_count += constant_fold(ia->value, rstack);

  return error_count;
}


int
m_constant_fold (pIIR_SliceIndexedAssociation sia, RegionStack &rstack)
{
  if (optimized(sia) & CONST_FOLD) return 0; else  optimized(sia) |= CONST_FOLD;

  int error_count = constant_fold(sia->value, rstack);
  error_count += constant_fold(sia->index_range, rstack);

  return error_count;
}


int
m_constant_fold(pIIR_SliceReference sr, RegionStack &rstack)
{
  if (optimized(sr) & CONST_FOLD) return 0; else  optimized(sr) |= CONST_FOLD;

  int error_count = 0;
  error_count += constant_fold(sr->array, rstack);
  error_count += constant_fold(sr->range, rstack);

  return error_count;
}



int
m_constant_fold(pIIR_RecordReference rr, RegionStack &rstack)
{
  if (optimized(rr) & CONST_FOLD) return 0; else  optimized(rr) |= CONST_FOLD;

  int error_count = 0;
  error_count += constant_fold(rr->record, rstack);

  return error_count;
}



int
m_constant_fold(pIIR_Expression e, RegionStack &rstack)
{
  printf ("/* No constant folding performed on:  %s */\n",  e->kind_name());
}


int
m_constant_fold(pIIR_ExpressionList el, RegionStack &rstack)
{
  if (optimized(el) & CONST_FOLD) return 0; else  optimized(el) |= CONST_FOLD;

  int error_count = 0;
  for ( ; el; el = el->rest)
    error_count += constant_fold(el->first, rstack);

  return error_count;
}



int
m_constant_fold (pIIR_ArrayAggregate aa, RegionStack &rstack)
{
  if (optimized(aa) & CONST_FOLD) return 0; else  optimized(aa) |= CONST_FOLD;

  int error_count = 0;
  // First, determine context of asscociations
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest)
    error_count += constant_fold(al->first, rstack);

  return error_count;
}



