#include <freehdl/vaul.h>
#include "v2cc-chunk.h"
#include <strstream>
#include <algo.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>
#include <unistd.h>

#include "mapping.h"
#include "v2cc-util.h"



// Used to generate error messages
extern vaul_error_printer codegen_error;



// Print out an internal error message and stop compilation if "stop"
// is true
extern void print_internal_error(char *file, int line, bool stop);

#define internal_error(b) print_internal_error(__FILE__, __LINE__, b)


// ******************************************************************************************
// Name: m_explore_and_check, generic function
//
// Description: explore (collect informations about node) and check
// node for correctness
//
// Return value: returns number of errors found
//
// ******************************************************************************************

int
m_explore_and_check (pIIR_EntityDeclaration e, RegionStack &rstack)
{
  rstack.push(e);
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  pIIR_InterfaceList ports = e->port_clause, generics = e->generic_clause;

  // Explore initial values of port signals
  for (pIIR_InterfaceList dl = ports; dl; dl = dl->rest) {
    pIIR_InterfaceDeclaration decl = (pIIR_InterfaceDeclaration)dl->first;
    if (decl->initial_value &&
	decl->initial_value->is(IR_EXPRESSION)) {
      get_context(decl->initial_value, ctxt, rstack, false, 0);
      error_count += constant_fold(decl->initial_value, rstack);
      error_count += check_expression(decl->initial_value, rstack);
    }
  }
  
  // Explore initial values of generics
  for (pIIR_InterfaceList dl = generics; dl; dl = dl->rest) {
    pIIR_InterfaceDeclaration decl = (pIIR_InterfaceDeclaration)dl->first;
    if (decl->initial_value) {
      get_context(decl->initial_value, ctxt, rstack, false, 0);
      error_count += constant_fold(decl->initial_value, rstack);
      error_count += check_expression(decl->initial_value, rstack);
    }
  }

  // Explore entity declarations (not ports and generics; those are
  // handled above)
  if (e->declarations)
    explore_and_check(e->declarations, rstack);

  // Explore concurrent statements of entity
  if (e->entity_statement_part)
    explore_and_check(e->entity_statement_part, rstack);

  rstack.pop();

  return 0; // Flag no errors found
}


int
m_explore_and_check (pIIR_DeclarationList dl, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);  
  int error_count = 0;

  for (pIIR_DeclarationList d = dl; d; d = d->rest)
    if (d->first->is(IR_OBJECT_DECLARATION)) {
      // Object declaration
      pIIR_ObjectDeclaration od = pIIR_ObjectDeclaration(d->first);
      // Explore initial value of declarations
      if (od->initial_value &&
	  od->initial_value->is(IR_EXPRESSION)) {
	get_context(od->initial_value, ctxt, rstack, false, 0);
	error_count += constant_fold(od->initial_value, rstack);
	error_count += check_expression(od->initial_value, rstack);
      }
      
    } else if (d->first->is(IR_TYPE_DECLARATION)) {
      // Type declaration
      pIIR_TypeDeclaration tdecl = pIIR_TypeDeclaration(d->first);
      error_count += explore_and_check(tdecl->type, rstack);
    }

  return error_count; // Return number of errors
}


int
m_explore_and_check (pIIR_Type st, RegionStack &rstack)
{
  return 0;
}


int
m_explore_and_check (pIIR_EnumerationType et, RegionStack &rstack)
{
  int enum_count = 0;

  // Count number of elements
  for (pIIR_EnumerationLiteralList ell = et->enumeration_literals; ell; ell = ell->rest)
    enum_count++;
  // And store result into node structure
  enum_item_number(et) = enum_count;

  return 0;
}

int
m_explore_and_check (pIIR_ArchitectureDeclaration a, RegionStack &rstack)
{
  rstack.push(a);
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  // Explore architecture declarations 
  error_count = explore_and_check(a->declarations, rstack);

  // Explore concurrent statements of entity
  error_count += explore_and_check(a->architecture_statement_part, rstack);

  rstack.pop();

  return error_count; // Return number of errors
}


int
m_explore_and_check (pIIR_ConcurrentStatementList csl, RegionStack &rstack)
{
  int error_count = 0;

  while (csl)
    {
      error_count += explore_and_check (csl->first, rstack);  
      csl = csl->rest;
    }

  return error_count;
}


int
m_explore_and_check (pIIR_ProcessStatement p, RegionStack &rstack)
{
  int error_count = 0;
  // Check whether a name has been associated with the process 
  if (p->declarator == NULL) {
    // If the process has no name then create a default name
    string process_name = i_to_string(p->seqno) + process_default_postfix;
    p->declarator = new IIR_TextLiteral(0, process_name.c_str());
  }

  rstack.push(p);
  ContextInfo &ctxt = *ActiveContext(rstack);

  // if the process has a sensitivity list then a virtual wait
  // statement is created in order to setup a corresponding wait_info
  // instance. Further, some info about the sensitvity list must be
  // collected.
  if (p->is(IR_SENSITIZED_PROCESS_STATEMENT)) {
    // Add a virtual wait statement
    pIIR_SensitizedProcessStatement sp = (pIIR_SensitizedProcessStatement)p;
    // Constant-Fold on sensitivity list
    error_count += constant_fold(sp->sensitivity_list, rstack);
    pIIR_WaitStatement ws = new IIR_WaitStatement(NULL, NULL, NULL, NULL, sp->sensitivity_list);
    get_context(ws, ctxt, rstack, false, 0);
  }

  // Explore local declarations of process
  if (p->declarations != NULL)
    error_count += explore_and_check(p->declarations, rstack);

  // explore and check sequential statements
  if (p->process_statement_part != NULL)
    error_count += explore_and_check(p->process_statement_part, rstack);

  rstack.pop();
  
  return error_count;
}




int
m_explore_and_check (pIIR_PackageDeclaration p, RegionStack &rstack)
{
  rstack.push(p);
  rstack.pop();

  return 0; // Flag no errors found
}

int
m_explore_and_check (pIIR_PackageBodyDeclaration pb, RegionStack &rstack)
{
  rstack.push(pb);
  rstack.pop();

  return 0; // Flag no errors found
}


int
m_explore_and_check (pIIR_ConfigurationDeclaration c, RegionStack &rstack)
{
  return 0; // Flag no errors found
}

int
m_explore_and_check (pIIR_ComponentDeclaration c, RegionStack &rstack)
{
  return 0; // Flag no errors found
}


int
m_explore_and_check(pIIR_SequentialStatementList sl, RegionStack &rstack)
{
  int error_count = 0;

  while (sl)
    {
      error_count += explore_and_check(sl->first, rstack);
      sl = sl->rest;
    }

  return error_count; // Flag no errors found
}


int
m_explore_and_check(pIIR_CaseStatement cs, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0, choice_error_count = 0;

  // Explore case expression
  get_context(cs->expression, ctxt, rstack, false, 0);
  error_count = constant_fold(cs->expression, rstack);
  error_count += check_expression(cs->expression, rstack);
  
  if (is_scalar_type(cs->expression->subtype)) {
    //****************************************************
    // Expression is scalar. 
    //****************************************************

    // First, determine bounds of type
    vector<RangeDescriptor> range_desc = 
      get_discrete_range(cs->expression->subtype, rstack, IR_LOCALLY_STATIC);
    // Determine not static range and check whether both ranges are
    // the same
    vector<RangeDescriptor> not_static_range_desc = 
      get_discrete_range(cs->expression->subtype, rstack, IR_NOT_STATIC);

    if (!range_desc[0].is_equal_to(not_static_range_desc[0])) {
      // If both ranges are NOT the same then print a notification if
      // desired. Further, the range is determined from the base type.
      range_desc = 
	get_discrete_range(get_base_type(cs->expression->subtype), rstack, IR_LOCALLY_STATIC);
      // Note the following message is NOT an error. It's just a hint
      // for the user that the actual range which must be covered by
      // the case statement might be "unexpected".
      string left_str, right_str, dir_str;
      range_desc[0].rangedes_to_int_string(left_str, dir_str, right_str, rstack);

      codegen_error.error("%:note: range of (sub)type %n is not locally static.", cs, cs->expression->subtype);
      codegen_error.error("%:note: determining locally static range from base type...", cs);
      codegen_error.error("%:note: ...static range is %s %s %s.", 
			  cs, left_str.c_str(), dir_str.c_str(), right_str.c_str());
    }

    // Convert range to ints
    int left, right; 
    IR_Direction dir;
    error_count += range_desc[0].constant_fold_rangedes(rstack);
    range_desc[0].rangedes_to_int(left, dir, right, rstack);

    long long int lower_type_bound; // Lower bound of expression type
    long long int upper_type_bound; // Upper bound of expression type
    bool to_type_range = true;
    if (left < right) {
      lower_type_bound = left;
      upper_type_bound = right;
    } else {
      lower_type_bound = right;
      upper_type_bound = left;
      to_type_range = false; // Descending range
    }
    // Number of discrete items 
    long long int item_type_count = upper_type_bound - lower_type_bound + 1; 
    long long int item_count = 0;
    
    // Now analyze each alternative
    vector<long long int> lower_vec; // Stores lower bounds of choices
    vector<long long int> upper_vec; // Stores upper bounds of choices
    vector<pIIR_Choice> choice_vec; // Points to choice associated
                                    // with lower_vec and upper_vec
    for (pIIR_CaseStatementAlternativeList al = cs->case_statement_alternatives; al; al = al->rest) {
      pIIR_CaseStatementAlternative a = al->first;

      // Analyze choices
      for (pIIR_ChoiceList cl = a->choices; cl; cl = cl->rest) {

	if (cl->first->is(IR_CHOICE_BY_RANGE)) {
	  //***************************************
	  // Choice by range
	  //***************************************

	  pIIR_ChoiceByRange cbr = pIIR_ChoiceByRange(cl->first);

	  // Constant fold choice expression
	  error_count += constant_fold(cbr->range, rstack);

	  // Get choice range. Note that choice must be locally static
	  long long int left = lower_type_bound;
	  long long int right = lower_type_bound;
	  IR_Direction dir = IR_DIRECTION_UP;
	  // Get range. Note that range_desc is empty if range is not
	  // locally static!
	  vector<RangeDescriptor> range_desc = get_discrete_range(cbr->range, rstack, IR_LOCALLY_STATIC);
	  if (range_desc.size() == 0) {
	    codegen_error.error("%:error: choice %n is not locally static.", cbr->range, cbr->range);
	    choice_error_count++;
	  } else {
	    // convert string values to long long ints
	    int l, r;
	    range_desc[0].rangedes_to_int(l, dir, r, rstack);
	    left = l;
	    right = r;
	  }

	  long long int lower, upper;
	  if (left == right) {
	    lower = left;
	    upper = right;
	  } else if (left < right && dir == IR_DIRECTION_UP) {
	    lower = left;
	    upper = right;
	  } else if (left > right && dir != IR_DIRECTION_UP) {
	    lower = right;
	    upper = left;
	  }

	  // Check whether choice is unique
	  for (int i = 0; i < lower_vec.size(); i++) {
	    if ((lower <= lower_vec[i] && upper >= lower_vec[i]) ||
		(lower <= upper_vec[i] && upper >= upper_vec[i])) {
	      codegen_error.error("%:error: choice %n (line %i) conflicts with %n (line %i).", 
				  cbr, cbr->range, get_line_number(cbr->range),
				  (choice_vec[i]->is(IR_CHOICE_BY_RANGE)? 
				   pIIR_ChoiceByRange(choice_vec[i])->range : 
				   pIIR_ChoiceByExpression(choice_vec[i])->value),
				  get_line_number(choice_vec[i]));
	      choice_error_count++;
	    }
	  }

	  // Add bounds to vectors 
	  lower_vec.push_back(lower);
	  upper_vec.push_back(upper);
	  choice_vec.push_back(cbr);
	  // Update item counter
	  item_count += upper - lower + 1;

	} else if (cl->first->is(IR_CHOICE_BY_EXPRESSION)) {
	  //***************************************
	  // Choice by expression
	  //***************************************
	  
	  long long int value;
	  pIIR_ChoiceByExpression cbe = pIIR_ChoiceByExpression(cl->first);

	  // Constant fold choice expression
	  error_count += constant_fold(cbe->value, rstack);

	  // Choice must be locally static
	  if (cbe->value->static_level != IR_LOCALLY_STATIC) {
	    codegen_error.error("%:error: choice %n is not locally static.", 
				cl->first, cbe->value);
	    choice_error_count++;
	    value = lower_type_bound;
	  } else
	    value = folded_value(cbe->value).long_value;

	  // Check whether the value is within the type bounds
	  if (value < lower_type_bound || value > upper_type_bound) {
	    codegen_error.error("%:error: choice %n is not within bounds of type %n.", 
				cbe->value, cbe->value, cs->expression->subtype);
	    choice_error_count++;
	  }
	    
	  // Check whether choice is unique
	  for (int i = 0; i < lower_vec.size(); i++) {
	    if (value >= lower_vec[i] && value <= upper_vec[i]) {
	      codegen_error.error("%:error: choice %n (line %i) conflicts with %n (line %i).", 
				  cbe->value, cbe->value, get_line_number(cbe->value), 
				  choice_vec[i], get_line_number(choice_vec[i]));
	      choice_error_count++;
	    }
	  }

	  // Add bounds to vectors 
	  lower_vec.push_back(value);
	  upper_vec.push_back(value);
	  choice_vec.push_back(cbe);
	  // A single element was specified. Hence, increment item counter.
	  item_count++;

	} else if (cl->first->is(IR_CHOICE_BY_OTHERS)) {
	  //***************************************
	  // Choice by others
	  //***************************************

	  // Check whether the others choice is last alternative in
	  // choice list
	  if (cl->rest != NULL) {
	    codegen_error.error("%:error: others choice must be last alternative in choice list.", cl->first);
	    choice_error_count++;
	  }
	  // Ok, due to the others statement all discrete items of the
	  // type are covered. Hence, set item_count to the number of
	  // elements the expression type consists of
	  item_count = item_type_count;
	}
      }
      
      // Analyze sequential statements associated with alternative
      error_count += explore_and_check(a->sequence_of_statements, rstack);
     
    }

    // Update error counter
    error_count += choice_error_count;

    // Check whether all items are covered by the choices
    if (choice_error_count != 0) {
      // If the choices contained errors then no choice coverage
      // checking is done but we warn the user.
      codegen_error.error("%:warning: no choice coverage checking is done due to previous errors.", cs);

    } else if (item_count != item_type_count) {
      // Not all items are covered! 

      // Determine missing choices and output some useful error
      // messages
      sort(lower_vec.begin(), lower_vec.end());
      sort(upper_vec.begin(), upper_vec.end());
      vector<long long int> missing_lower, missing_upper;
      long long int search_bound = lower_type_bound;
      // Determine missing ranges
      for (int i = 0; i < lower_vec.size(); i++) {
	if (search_bound < lower_vec[i]) {
	  missing_lower.push_back(search_bound);
	  missing_upper.push_back(lower_vec[i] - 1);
	}
	search_bound = upper_vec[i] + 1;
      }
      if (search_bound <= upper_type_bound) {
	missing_lower.push_back(search_bound);
	missing_upper.push_back(upper_type_bound);
      }
      // Generate error message string reporting the missing ranges
      string error_str;
      bool first = true;
      pIIR_Type base_type = get_base_type(cs->expression->subtype);
      for (int i = 0; i < missing_lower.size(); i++, first = false) {
	string low_value;
	string high_value;
	if (base_type->is(IR_ENUMERATION_TYPE)) {
	  // If expression is an enumeration type then convert
	  // position numbers into appropriate strings
	  low_value = pos_to_literal(pIIR_EnumerationType(base_type)->enumeration_literals, (int)missing_lower[i])
	    ->declarator->text.to_chars();
	  high_value = pos_to_literal(pIIR_EnumerationType(base_type)->enumeration_literals, (int)missing_upper[i])
	    ->declarator->text.to_chars();
	} else {
	  // If expression is an integer type then convert integer
	  // values to strings
	  low_value = i_to_string((int)missing_lower[i]);
	  high_value = i_to_string((int)missing_upper[i]);
	}
	if (missing_lower[i] == missing_upper[i])
	  error_str += (first?"":", ") + low_value;
	else
	  error_str += (first?"":", ") + 
	    (to_type_range? low_value + " to " + high_value : high_value + " downto " + low_value);
      }

      codegen_error.error("%:error: not all possible values of type %n are covered by the case statement.", 
			  cs, cs->expression->subtype);
      codegen_error.error("%:error: missing element(s): %s.", cs, error_str.c_str());
      error_count++;      
    }

    return error_count; // We are done. Return number of errors found

  } else {
    //****************************************************
    // The expression is an array where the elements are characters
    //****************************************************

    return error_count; // We are done. Return number of errors found
  }
}


int
m_explore_and_check(pIIR_IfStatement ifs, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;
  
  // Explore condition expression
  get_context(ifs->condition, ctxt, rstack, false, 0);
  error_count = constant_fold(ifs->condition, rstack);
  error_count += check_expression(ifs->condition, rstack);

  // Explore then branch
  error_count += explore_and_check(ifs->then_sequence, rstack);

  // Explore else branch (if present)
  if (ifs->else_sequence != NULL)
    error_count += explore_and_check(ifs->else_sequence, rstack);

  return error_count;
}


int
m_explore_and_check(pIIR_NullStatement n, RegionStack &rstack)
{
  return 0;
}


int
m_explore_and_check(pIIR_ForLoopStatement fs, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  // Explore iteration range
  get_context(((pIIR_ScalarSubtype)fs->iterator->subtype)->range, ctxt, rstack, false, 0);
  error_count = constant_fold(((pIIR_ScalarSubtype)fs->iterator->subtype)->range, rstack);
  error_count += check_expression(((pIIR_ScalarSubtype)fs->iterator->subtype)->range, rstack);

  // Explore loop body
  error_count += explore_and_check(fs->sequence_of_statements, rstack);

  return error_count; // Flag no errors found
}


int
m_explore_and_check(pIIR_LoopStatement ls, RegionStack &rstack)
{
  int error_count = 0;

  // Explore loop body
  error_count += explore_and_check(ls->sequence_of_statements, rstack);
}


int
m_explore_and_check(pIIR_WhileLoopStatement wls, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  get_context(wls->condition, ctxt, rstack, false, 0);
  error_count = constant_fold(wls->condition, rstack);
  error_count += check_expression(wls->condition, rstack);
  
  // Explore loop body
  error_count += explore_and_check(wls->sequence_of_statements, rstack);
}


int
m_explore_and_check(pIIR_WaitStatement ws, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  // Add ws to the list of wait statements of the current context
  ctxt.wait_statements.push_back(ws);

  if (ws->sensitivity_list) {
    // Wait statemen has an explicit sensitivity list
    for (pIIR_ExpressionList el = ws->sensitivity_list; el; el = el->rest)
      {
	ContextInfo tmp_ctxt;
	get_context(el->first, tmp_ctxt, rstack, false, 0);
	error_count += constant_fold(el->first, rstack);
	error_count += check_expression(el->first, rstack);
	(*tmp_ctxt.accessed_objects.begin()).access_type = SENSITIVE;
	ctxt.accessed_objects.push_back(*tmp_ctxt.accessed_objects.begin());
      }

    if (ws->condition_clause) {
      get_context(ws->condition_clause, ctxt, rstack, false, 0);
      error_count += constant_fold(ws->condition_clause, rstack);
      error_count += check_expression(ws->condition_clause, rstack);
    }

  } else {
    // Wait statement does not have an explicit sensitivity
    // list. Hence, extract one from the wait condition.
    ContextInfo tmp_ctxt;
    get_context(ws->condition_clause, tmp_ctxt, rstack, false, 0);
    error_count += constant_fold(ws->condition_clause, rstack);
    error_count += check_expression(ws->condition_clause, rstack);
    for (access_list::iterator i = tmp_ctxt.accessed_objects.begin(); 
	 i != tmp_ctxt.accessed_objects.end(); i++)
      ctxt.accessed_objects.push_back(AccessDescriptor((*i).declaration, (*i).object_ref, SENSITIVE | READ));
  }

  // Explore timeout expression
  if (ws->timeout_clause) {
    get_context(ws->timeout_clause, ctxt, rstack, false, 0);
    error_count += constant_fold(ws->timeout_clause, rstack);
    error_count += check_expression(ws->timeout_clause, rstack);
  }

  return error_count;
}


int
m_explore_and_check(pIIR_SignalAssignmentStatement sas, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  // Add sas to list of signal assignment statements
  ctxt.signal_assignment_statements.push_back(sas);

  // Collect objects in target expression
  get_context(sas->target, ctxt, rstack, true /* = true, because the signal is a target */, 0);
  // Check target expression
  error_count = constant_fold(sas->target, rstack);
  error_count += check_expression(sas->target, rstack);

  // Collect signals in reject expression
  if (sas->reject_time_expression) {
    get_context(sas->reject_time_expression, ctxt, rstack, true, 0); 
    error_count += constant_fold(sas->reject_time_expression, rstack);
    error_count += check_expression(sas->reject_time_expression, rstack);
  }

  // Collect signals in waveforms
  for (pIIR_WaveformList wl = sas->waveform; wl; wl = wl->rest) {
    get_context(wl->first->value, ctxt, rstack, false, 0);
    error_count += constant_fold(wl->first->value, rstack);
    error_count += check_expression(wl->first->value, rstack);
  }

  return error_count;
}



int
m_explore_and_check(pIIR_VariableAssignmentStatement vas, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  // Collect objects in target expression
  get_context(vas->target, ctxt, rstack, true /* = true, because the variable is a target */, 0);
  error_count = constant_fold(vas->target, rstack);
  error_count += check_expression(vas->target, rstack);

  // Collect objects in assignment expression
  get_context(vas->expression, ctxt, rstack, false, 0);
  error_count += constant_fold(vas->expression, rstack);
  error_count += check_expression(vas->expression, rstack);
  
  return error_count;
}


int
m_explore_and_check(pIIR_ComponentInstantiationStatement cs, RegionStack &rstack)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  int error_count = 0;

  for (pIIR_AssociationList al = cs->port_map_aspect; al; al = al->rest) {
    pIIR_AssociationElement ae = al->first;

    get_context(ae->actual, ctxt, rstack, false, 0);
    error_count += constant_fold(ae->actual, rstack);
    error_count += check_expression(ae->actual, rstack);
  }

  return error_count;
}


int
m_explore_and_check(pIIR_ProcedureCallStatement pc, RegionStack &rstack)
{
  int error_count = 0;
  ContextInfo &ctxt = *ActiveContext(rstack);

  for (pIIR_InterfaceList il = pc->procedure->interface_declarations; il; il = il->rest) {
    pIIR_InterfaceDeclaration  idecl = il->first;
    // Find actual that corresponds to formal
    pIIR_AssociationElement a = find_matching_actual(pc->actual_parameter_part, il->first);

    // check whether the corresponding actual is left open
    if (a != NULL && a->actual != NULL && !a->actual->is(IR_OPEN_EXPRESSION)) {
      // The parameter was NOT left open...
  
      /* if the mode is INOUT, OUT or BUFFER then the actual is written */
      if (idecl->mode == IR_INOUT_MODE || 
	  idecl->mode == IR_OUT_MODE ||
	  idecl->mode == IR_BUFFER_MODE)
	get_context(a->actual, ctxt, rstack, true /* = true, because the actual is a target */, 0);
      
      /* if the mode is IN or BUFFER then the actual is read */
      if (idecl->mode == IR_INOUT_MODE || 
	  idecl->mode == IR_BUFFER_MODE)
	get_context(a->actual, ctxt, rstack, false, 0);
      
      // fold and check actual expression
      error_count += constant_fold(a->actual, rstack);	
      error_count += check_expression(a->actual, rstack);	

    } else
      // the parameter is left open. Hence, the default value will be used instead. 
      // explore and check default value if present
      if (idecl->initial_value != NULL) {
	get_context(idecl->initial_value, ctxt, rstack, false, 0);
	error_count += constant_fold(idecl->initial_value, rstack);	
	error_count += check_expression(idecl->initial_value, rstack);	
      }
  }

  return error_count;
}


// ******************************************************************************************
// Name: m_get_context, generic function
//
// Description: Analyzes statements and collects all abjects which are
// accessed. The information about the accessed objects are stored in
// a ContextInfo instance. A ContextInfo instance stores three
// different kinds of informations: 
//
// 1. accessed_objects = all objects
// which are accessed (read, written, sensitive).
//
// 2. wait_statements = pointer to the wait statements included in the
// current context
//
// 3. signal_assignment_statements = list of pointers to all signal
// assignment statements included in the current context
//
// 4. internal_vars = internal objects required by the generated code
//
// Parameter: ctxt = context info instance
//   target = is true if the expression is target of an assignment
//   level = level the object is located at. The entire statement
// is assigned level 0. The level is increased each time an object is
// included in an argument to a subroutine or an index value of an
// array reference.
// 
// Return value: returns a pointer to the corresponding access
// descriptor if an expression is analyzed. Otherwise, NULL is
// returned.
//
// ******************************************************************************************


pAccessDescriptor
m_get_context(pIIR_AttrSigFunc aev, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  pAccessDescriptor p = get_context(aev->signal, ctxt, rstack, false, level + 1);
  // Store that an attribute of the signal has been used
  p->access_type |= ATTRIBUTE; 

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_FunctionCall fc, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  for (pIIR_AssociationList al = fc->parameter_association_list; al; al = al->rest)
    get_context(al->first->actual, ctxt, rstack, false, level + 1);

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_SimpleReference sr, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  if (sr->object->is(IR_SIGNAL_DECLARATION) || sr->object->is(IR_SIGNAL_INTERFACE_DECLARATION) || 
      sr->object->is(IR_VARIABLE_DECLARATION)) {
    ctxt.accessed_objects.push_back(AccessDescriptor(sr->object, sr, target? WRITE : READ, level));
    return &ctxt.accessed_objects.back();
  } else
    return NULL;
}



pAccessDescriptor
m_get_context(pIIR_ArrayReference ar, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  pAccessDescriptor p = get_context(ar->array, ctxt, rstack, target, level);
  p->object_ref = ar;
  for (pIIR_ExpressionList el = ar->indices; el; el = el->rest)
    get_context(el, ctxt, rstack, false, level + 1);

  return p;
}




pAccessDescriptor
m_get_context(pIIR_ArrayRange ar, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  pAccessDescriptor p = get_context(ar->array, ctxt, rstack, false, level);
  get_context(ar->index, ctxt, rstack, false, level + 1);

  return p;
}


pAccessDescriptor
m_get_context(pIIR_ExplicitRange er, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  get_context(er->left, ctxt, rstack, false, level);
  get_context(er->right, ctxt, rstack, false, level);

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_OthersIndexedAssociation ia, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  get_context(ia->value, ctxt, rstack, false, level);

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_RangeIndexedAssociation ia, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  get_context(ia->index_range, ctxt, rstack, false, level);
  get_context(ia->value, ctxt, rstack, false, level);

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_SingleIndexedAssociation ia, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  if (ia->index != NULL)
    get_context(ia->index, ctxt, rstack, false, level);
  get_context(ia->value, ctxt, rstack, false, level);

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_SliceReference sr, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  pAccessDescriptor p = get_context(sr->array, ctxt, rstack, target, level);
  p->object_ref = sr;
  get_context(sr->range, ctxt, rstack, false, level + 1);

  return p;
}



pAccessDescriptor
m_get_context(pIIR_RecordReference rr, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  pAccessDescriptor p = get_context(rr->record, ctxt, rstack, target, level);
  p->object_ref = rr;

  return p;
}




pAccessDescriptor
m_get_context(pIIR_Expression e, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  printf ("/* No signals extracted from expression type:  %s */\n",  e->kind_name());
  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_ExpressionList el, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  for ( ; el; el = el->rest)
    get_context(el->first, ctxt, rstack, target, level);

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_WaitStatement ws, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  // Add ws to the list of wait statements of the current context
  ctxt.wait_statements.push_back(ws);

  if (ws->sensitivity_list) {
    // Wait statemen has an explicit sensitivity list
    for (pIIR_ExpressionList el = ws->sensitivity_list; el; el = el->rest)
      {
	ContextInfo tmp_ctxt;
	get_context(el->first, tmp_ctxt, rstack, false, 0);
	(*tmp_ctxt.accessed_objects.begin()).access_type = SENSITIVE;
	ctxt.accessed_objects.push_back(*tmp_ctxt.accessed_objects.begin());
      }

    if (ws->condition_clause) {
      get_context(ws->condition_clause, ctxt, rstack, false, 0);
    }

  } else {
    // Wait statement des not have an explicit sensitivity
    // list. Hence, extract one from the wait condition.
    ContextInfo tmp_ctxt;
    get_context(ws->condition_clause, tmp_ctxt, rstack, false, 0);
    for (access_list::iterator i = tmp_ctxt.accessed_objects.begin(); 
	 i != tmp_ctxt.accessed_objects.end(); i++)
      ctxt.accessed_objects.push_back(AccessDescriptor((*i).declaration, (*i).object_ref, SENSITIVE | READ));
  }

  // Explore timeout expression
  if (ws->timeout_clause) {
    get_context(ws->timeout_clause, ctxt, rstack, false, 0);
  }

  return NULL;
}


pAccessDescriptor
m_get_context(pIIR_ArrayAggregate aa, ContextInfo &ctxt, RegionStack &rstack, 
	      bool target, int level)
{
  // First, determine context of asscociations
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest)
    get_context(al->first, ctxt, rstack, false, level + 1);

  return NULL;
}




// ******************************************************************************************
// Name: m_check_expression , generic function
//
// Description: checks whether expression is valid and contains no
// errors. Any errors found are reportet. Parameter rstack is a
// reference to the current regions stack
// 
// Return value: returns number of errors which were found in the expression
//
// ******************************************************************************************



int
m_check_expression(pIIR_AttrSigFunc ase, RegionStack &rstack)
{
  int error_count = check_expression(ase->signal, rstack);

  return error_count;
}


int
m_check_expression(pIIR_FunctionCall fc, RegionStack &rstack)
{
  int error_count = 0;

  for (pIIR_AssociationList al = fc->parameter_association_list; al; al = al->rest)
    error_count += check_expression(al->first->actual, rstack);

  return error_count;
}



int
m_check_expression(pIIR_SimpleReference sr, RegionStack &rstack)
{
  return 0;
}



int
m_check_expression(pIIR_ArrayReference ar, RegionStack &rstack)
{
  int error_count = check_expression(ar->array, rstack);
  for (pIIR_ExpressionList el = ar->indices; el; el = el->rest)
    error_count += check_expression(el, rstack);

  return error_count;
}



int
m_check_expression(pIIR_ArrayRange ar, RegionStack &rstack)
{
  int error_count = check_expression(ar->array, rstack);
  error_count += check_expression(ar->index, rstack);

  return error_count;
}


int
m_check_expression(pIIR_ExplicitRange er, RegionStack &rstack)
{
  int error_count = check_expression(er->left, rstack);
  error_count += check_expression(er->right, rstack);

  return error_count;
}


int
m_check_expression(pIIR_OthersIndexedAssociation ia, RegionStack &rstack)
{
  int error_count = check_expression(ia->value, rstack);

  return error_count;
}


int
m_check_expression(pIIR_RangeIndexedAssociation ia, RegionStack &rstack)
{
  int error_count = check_expression(ia->index_range, rstack);
  error_count += check_expression(ia->value, rstack);

  return error_count;
}


int
m_check_expression(pIIR_SingleIndexedAssociation ia, RegionStack &rstack)
{
  int error_count = 0;

  if (ia->index != NULL)
    error_count = check_expression(ia->index, rstack);
  error_count += check_expression(ia->value, rstack);

  return error_count;
}


int
m_check_expression(pIIR_SliceReference sr, RegionStack &rstack)
{
  int error_count = check_expression(sr->array, rstack);
  error_count += check_expression(sr->range, rstack);

  return error_count;
}



int
m_check_expression(pIIR_RecordReference rr, RegionStack &rstack)
{
  return check_expression(rr->record, rstack);
}




int
m_check_expression(pIIR_Expression e, RegionStack &rstack)
{
  printf ("/* No checks performed on:  %s */\n",  e->kind_name());
  return 0;
}


int
m_check_expression(pIIR_ExpressionList el, RegionStack &rstack)
{
  int error_count = 0;
  for ( ; el; el = el->rest)
    error_count += check_expression(el->first, rstack);

  return error_count;
}



int
m_check_expression (pIIR_ArrayAggregate aa, RegionStack &rstack)
{
  int error_count = 0;

  // First, determine context of asscociations
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest)
    error_count += check_expression(al->first, rstack);

  // Subtype of aggregate expression. At least the range direction will
  // be used...
  int dim_number = 1;
  pIIR_Type dest_type = aa->subtype;
  if (dest_type->is(VAUL_SUBARRAY_TYPE)) {
    // If the aggreate is an multi-dimensional then a node
    // VAUL_SubarrayType is used to describe the (sub)type of the
    // sub-aggregate. First, determine dimension of the main array the
    // aggregate belongs to
    dest_type = pVAUL_SubarrayType(aa->subtype)->complete_type->declaration->type;
    int dim_counter = 0;
    for (pIIR_TypeList tl = pVAUL_SubarrayType(aa->subtype)->complete_type->index_types; tl; tl = tl->rest)
      dim_counter++;
    // Next, determine the index associated with the array
    // aggregate. Note that we can only count how many dimensions are
    // left from where the aggregate starts.
    int current_dim_counter = 0;
    for (pIIR_TypeList tl = pVAUL_SubarrayType(aa->subtype)->index_types; tl; tl = tl->rest)
      current_dim_counter++;
    // Now, determine the index number of the aggregate
    dim_number = dim_counter - current_dim_counter + 1;

  } else if (dest_type->is(IR_ARRAY_TYPE))
    dest_type = aa->subtype->declaration->type;

  // Get range type of aggregate and determine left bound, right
  // bound, direction and length of aggregat. Note that if the
  // aggregate subtype cannot be determined from the context then
  // dest_range will point to the corresponding index range of the
  // unconstrained array associated with the aggregate.
  dest_length(aa) = -1; 
  known_subtype(aa) = true;
  // Get left and right bound as well as direction of the aggregate. If
  // the bounds are locally static then calculate length of aggregate.
  vector<RangeDescriptor> range_desc = 
    get_discrete_range (dest_type, rstack, IR_NOT_STATIC);
  if (range_desc[dim_number-1].rangedes_to_int(dest_left(aa), dest_direction(aa), dest_right(aa), rstack))
    dest_length(aa) = abs(dest_left(aa) - dest_right(aa)) + 1;
  // If type of aggregate is an unconstrained array then set length of
  // aggregate to -1 (unknown) and known_subtype to false.
  if (!is_constrained_array_type(aa->subtype)) {
    dest_length(aa) = -1;
    known_subtype(aa) = false;
  }

  has_others(aa) = false; // Is set to true if aggregate includes others choice
  min_index(aa) = INT_MAX; // Min index value
  max_index(aa) = INT_MIN; // Max index value
  locally_static_ranges(aa) = true; // We assume that all choice ranges are locally static
  total_length(aa) = -1; // Total length
  int choice_counter = 0;

  /* First, take a look at all choices to determine the number of
   * choices, whether an others choice is present and determine lower
   * and upper index range values. */
  int named_association_count = 0;
  int positional_association_count = 0;
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest) { 

    if (al->first->is(IR_SINGLE_INDEXED_ASSOCIATION)) { // Single indexed association
      pIIR_SingleIndexedAssociation saa = pIIR_SingleIndexedAssociation(al->first);
      int value;
      if (saa->index != NULL) {	// Named association
	named_association_count++;
	if (expr_to_int(saa->index, value, rstack)) {
	  // Determine min/max values
	  min_index(aa) = min(value, min_index(aa));
	  max_index(aa) = max(value, max_index(aa));
	  total_length(aa) = max(total_length(aa), 0) + 1; // Increment length (min length is 1)
	  // Set values of choice
	  min_index(saa) = value;
	  max_index(saa) = value;
	  locally_static_range(saa) = true;
	  length(saa) = 1;
	} else {
	  locally_static_ranges(aa) = false;
	  locally_static_range(saa) = false;
	}

      } else { // Positional association
	positional_association_count++;
	locally_static_range(saa) = true;
	length(saa) = 1;
	// Store number of choice into min_index and
	// max_index. Further, max_index of the aggregate expression
	// is set to the current choice counter and min_index of the
	// aggregate expression is set to 0.
	min_index(saa) = choice_counter;
	max_index(saa) = choice_counter;
	max_index(aa) = choice_counter;
	min_index(aa) = 0;
      }

    } else if (al->first->is(IR_RANGE_INDEXED_ASSOCIATION)) { // Range association
      named_association_count++; // Range indexed association is
      // always named association
      pIIR_RangeIndexedAssociation raa = pIIR_RangeIndexedAssociation(al->first);
      int from, to;
      IR_Direction direction;
      vector<RangeDescriptor> range_desc = get_discrete_range(raa->index_range, rstack, IR_NOT_STATIC);
      if (range_desc[0].rangedes_to_int(from, direction, to, rstack)) {
	// Test whether range bound matches the direction
	if (from == to ||
	    (from < to && direction == IR_DIRECTION_UP) ||
	    (from > to && direction == IR_DIRECTION_DOWN)) {
	  // Determine min/max values
	  min_index(aa) = min(from, min_index(aa));
	  max_index(aa) = max(from, max_index(aa));
	  min_index(aa) = min(to, min_index(aa));
	  max_index(aa) = max(to, max_index(aa));
	  // Update length (min length is 1)
	  total_length(aa) = max(max(total_length(aa), 0) + abs(from - to) + 1, 1); 
	  // Set values of choice
	  min_index(raa) = min(from, to);
	  max_index(raa) = max(from, to);
	  locally_static_range(raa) = true;
	  length(raa) = abs(from - to) + 1;
	}
      } else {
	locally_static_ranges(aa) = false;
	locally_static_range(raa) = false;
      }
      
    } else if (al->first->is(IR_OTHERS_INDEXED_ASSOCIATION)) // Others clause
      has_others(aa) = true;

    choice_counter++; // Increment counter
  }

  // Set named_association flag to false if at least one positional
  // association was found.
  if (positional_association_count != 0) 
    named_association(aa) = false;
  else
    named_association(aa) = true;


  // Check whether only named or only positional association has been used
  if (named_association_count != 0 && positional_association_count != 0)
    codegen_error.error("%:error: array aggregates cannot contain positional and named associations", aa);

  // Check whether all ranges are locally static if more than a single
  // choice was given
  if (!locally_static_ranges(aa))
    if (choice_counter != 1) {
      internal_error(true);
      return error_count + 1;
    } else
      return error_count;
  
  // Next, make some checks to test the range of the choices. First,
  // check whether ranges do not overlap. 
  vector<int> from_vec, to_vec;
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest) { 
    if (al->first->is(IR_OTHERS_INDEXED_ASSOCIATION)) continue;
    // Check new range against previous ranges
    for (int i = 0; i < from_vec.size(); i++) {
      if ((min_index(al->first) <= from_vec[i] && max_index(al->first) >= from_vec[i]) ||
	  (min_index(al->first) <= to_vec[i] && max_index(al->first) >= to_vec[i]))
	codegen_error.error("%:error: choice ranges do overlap (%i to %i conflicts with %i to %i)", 
			    aa, min_index(al->first), max_index(al->first), from_vec[i], to_vec[i]);
    }
    // Add new index range to vectors
    from_vec.push_back(min_index(al->first));
    to_vec.push_back(max_index(al->first));
  }
  // If aggregate has others choice then we are done
  if (has_others(aa)) 
    return error_count;
  
  // Otherwise, check whether there are no "holes" bezween the
  // ranges. We already checked that the ranges do not overlap. Hence,
  // we simply sort both arrays from_vec and to_vec which will move
  // associated entries in both vectors to the SAME position.
  sort(from_vec.begin(), from_vec.end());
  sort(to_vec.begin(), to_vec.end());
  for (int i = 1; i < from_vec.size(); i++)
    if (to_vec[i - 1] + 1 != from_vec[i]) {
      codegen_error.error("%:error: index %i of aggregate is undefined", 
			  aa, from_vec[i] - 1);
      return error_count + 1;
    }

  // Test whether the length as determined from the choices matches
  // the length of the aggregate subtype (if this length is kown from
  // the context).
  if (dest_length(aa) != -1 &&
      dest_length(aa) != total_length(aa)) {
    codegen_error.error("%:error: length of array aggregate (=%i) does not match length of target (=%i)",
			aa, total_length(aa), dest_length(aa));
    error_count++;
  }

  return error_count;
}



