
#include <strstream>
#include "v2cc-util.h"


// ******************************************************************************************
// Some global variables
// ******************************************************************************************

// used to generate error messages
extern vaul_error_printer codegen_error;
int dbg=0;

// Counter to produce unique names for internal variables
int internal_counter = 0;
const string internal_prefix_start = "itn";

// Prefix used to build a default name for a process
const string process_default_postfix = "pn";

// Prefix used to build a default name for loop statement
const string loop_default_postfix = "ln";

// Integer IIR_Range instance
pIIR_ExplicitRange iir_integer_range = NULL;

// ******************************************************************************************


// Converts a string item into an IR_TextLiteral
pIIR_TextLiteral
to_TextLiteral(const string &str)
{
  return new IIR_TextLiteral(NULL, IR_String(str.c_str()));
}

// Print out an internal error message and stop compilation if "stop"
// is true
void
print_internal_error(char *file, int line, bool stop)
{
  fprintf (stderr, "Internal code generator error in file %s at line %i.\nCompilation aborted...\n",
	   file, line);
  if (stop)
    exit(1);
}

#define internal_error(b) print_internal_error(__FILE__, __LINE__, b)


/* Get line number associated with a node */
int
get_line_number(pIIR_Root n)
{
  if (n->pos->is(IR_POS_INFO_TEXT_FILE))
    return pIIR_PosInfo_TextFile(n->pos)->line_number;
  else if (n->pos->is(IR_POS_INFO_SHEET))
    return pIIR_PosInfo_Sheet(n->pos)->x_coordinate;
  else
    internal_error(true);
}


/* Select association element from an association list which
 * corresponds with given formal */
pIIR_AssociationElement
find_matching_actual(pIIR_AssociationList assocs, pIIR_InterfaceDeclaration formal)
{
  for (pIIR_AssociationList al = assocs; al; al = al->rest)
    {
      pIIR_AssociationElement a = al->first;
      if (a->formal == formal) // break if corresponding actual is found
	return a;
    }

  return NULL;
}


/* Creates an array info object for an array subtype */
string
create_array_info_obj(pIIR_ArraySubtype as, RegionStack &rstack, const bool static_object)
{
  string result = "new array_info(";
  
  pIIR_TypeDeclaration type_declaration = as->immediate_base->declaration;
  result += qid(type_declaration,"_",INFO) + "_INFO,";

  // Extract bounds and range direction from expression
  vector<RangeDescriptor> range_desc = get_discrete_range (as->constraint->first, rstack, IR_NOT_STATIC);
  string left, dir, right;
  range_desc[0].rangedes_to_int_string(left, dir, right, rstack);
  result += left + "," + dir + "," + right;

  // If the array_info instance will be static, then set the last
  // argument to -1. Otherwise, append 0. Static array_info instances
  // are not removed even when no array instance is using it!
  if (static_object)
    result += ",-1)";
  else
    result += ",0)";

  return result;
}





/* Emit code to create an new object of a specific type */
string
create_default_instance(pIIR_Type type, RegionStack &rstack)
{
  string str;
  pIIR_Type im_base_type = get_immediate_base_type(type);
  pIIR_TypeDeclaration type_decl = get_declaration(type);
  
  // First, check whether the type is an array type
  if (im_base_type->is(IR_ARRAY_TYPE)) {
    // Bail out if the type is not constrained!
    if (!is_constrained_array_type(type))
      internal_error(true);

    str += qid(type_decl, "_", TYPE) + "(";
    str += create_array_info_obj(pIIR_ArraySubtype(type), rstack, false);
    str += "," + create_default_instance(pIIR_ArrayType(im_base_type)->element_type, rstack) + ")";
    
  } else {
    // Emit simple constructor for the type
    pIIR_TypeDeclaration type_decl = get_declaration(type);
    str += qid(type_decl, "_", TYPE) + "()";
  }

  return str;
}



/* Test whether a type is a scalar */
bool
is_scalar_type(pIIR_Type t)
{
  return (t->is(IR_SCALAR_TYPE) || t->is(IR_SCALAR_SUBTYPE));
}


/* Test whether an type is an constrained array type */
bool
is_constrained_array_type(pIIR_Type t)
{
  if (t->is(VAUL_SUBARRAY_TYPE)) {
    if (pVAUL_SubarrayType(t)->complete_type->declaration->type->is(IR_ARRAY_SUBTYPE))
      return true;
    else
      return false;
    
  } else if (t->is(IR_ARRAY_SUBTYPE)) {
    return true;
    
  } else if (t->is(IR_ARRAY_TYPE)) {
    if (pIIR_ArrayType(t)->declaration->type->is(IR_ARRAY_SUBTYPE))
      return true;
    else
      return false;
    
  } else
    return false;
}


bool
is_implicit_array_subtype(pIIR_Type t)
{
  if (!t->is(VAUL_SUBARRAY_TYPE) &&
      !t->is(IR_ARRAY_SUBTYPE) &&
      !t->is(IR_ARRAY_TYPE))
    return false;

  return  t->declaration == NULL;
}


/* Convert an integer value into a string */
string
i_to_string(int i)
{
  strstream lstr;
  lstr << i << (char)'\0';
  return string(lstr.str());
}


/* Convert a double value into a string */
string
d_to_string(double d)
{
  strstream lstr;
  lstr << d << (char)'\0';
  return string(lstr.str());
}


/* Convert an string value into a integer */
int
string_to_i(const string &str)
{
  return atoi(str.c_str());
}

/* Convert an string value into a double */
double
string_to_d(const string &str)
{
  return strtod(str.c_str(), NULL);
}

/* Concatenate all elements of a string list */
string
concat(list<string> lstr, string sep)
{
  if (lstr.size() == 0) return string("");

  string result = lstr.front();

  list<string>::iterator iter = lstr.begin();
  while (++iter != lstr.end())
    result += sep + (*iter);

  return result;
}


/* Concatenate all elements of a string vector */
string
concat(vector<string> vstr, string sep)
{
  if (vstr.size() == 0) return string("");

  string result = vstr.front();

  vector<string>::iterator iter = vstr.begin();
  while (++iter != vstr.end())
    result += sep + (*iter);

  return result;
}


/* Return a string of L spaces, but at most 30.  */
string 
spaces(int l)
{
  if(l > 30)
    l = 30;
  if(l < 0)
    l = 0;
  return string("                              " + (30-l));
}

// Determine name of a subprogram. The name is generated as usual,
// however an sequence number is added to the name in order to
// separate functions with the same parameter set but different return
// value. The sequence number is a unique number within a design
// unit. Note that in VHDL two functions with the same parameter set
// but different return type may be defined while in C++ this is not
// permitted.
string get_subprogram_name(pIIR_SubprogramDeclaration spd)
{
  return  qid(spd, "_", DEFAULT) + "_i" + i_to_string(spd->seqno);
}


/* Emit an UglyIdentifier as a beautiful_identifier.  */
void emit_id(pIIR_TextLiteral idnode, string &str)
{
  char *id = idnode->text.to_chars();
  if (*id == '\\' || *id == '\'')
    str += id;
  else 
    {
      bool break_on_caps = false;
      while (*id) 
	{
	  if (isupper(*id) && break_on_caps)
	    str += '_';
	  break_on_caps = islower(*id);
	  str += tolower(*id++);
	}
    }
}



// get a fully qualified identifier
string
qid (pIIR_Declaration d, char *qual="_", id_type typ)
{
  string id;

  if (d->is (IR_TYPE_DECLARATION) && 
      typ.object() == TYPE &&
      is_scalar_type(pIIR_TypeDeclaration(d)->type)) 
    {
      pIIR_Type base_type = get_base_type(pIIR_TypeDeclaration(d)->type);

      if (base_type->is(IR_INTEGER_TYPE))
	return "integer";
      else if (base_type->is(IR_ENUMERATION_TYPE))
	return "enumeration";
      else if (base_type->is(IR_PHYSICAL_TYPE))
	return "physical";
      else if (base_type->is(IR_FLOATING_TYPE))
	return "floatingpoint";
      else
	internal_error(true);
    }
  else if (d->is (IR_ARCHITECTURE_DECLARATION))			// Ent_Arch
    {
      id+=qid(((pIIR_ArchitectureDeclaration) d)->entity, qual, typ);
    }
  else if (d->is (IR_LIBRARY_UNIT) && !(typ.object() == BARE))
    {
      id+=nid(pIIR_LibraryUnit(d),LIBRARY);
    }
  else if (d->is (IR_LIBRARY_UNIT) && (typ.object() == BARE))
    id+=pIIR_LibraryUnit(d)->library_name->text.to_chars();		
  else if (d->declarative_region)
    id+=qid(d->declarative_region, qual, typ);

  id+= qual + nid(d, typ);

  if ((d->is(IR_SIGNAL_DECLARATION) || d->is(IR_SIGNAL_INTERFACE_DECLARATION)) 
      && (typ.access() & ARCHREF) && (typ.object() != READER) && (typ.object() != DRIVER))
    // The ARCHREF flag is used to access object within the
    // constructor of a process class.
    id = "arch->" + id;

  if ((d->is(IR_SIGNAL_DECLARATION) || d->is(IR_SIGNAL_INTERFACE_DECLARATION)) && 
       (typ.access() & DEREF))
      // If a signal shall be printed and the DEREF flag is set then the
      // entire string is prepended with "*"
      id = "(*" + id + ")";

  if (d->is(IR_TYPE_DECLARATION) && (typ.object() == (ALIAS | TYPE)) &&
      (((pIIR_TypeDeclaration)d)->type->is(IR_ARRAY_SUBTYPE) || 
       ((pIIR_TypeDeclaration)d)->type->is(IR_ARRAY_TYPE)))
    // Create an array alias type name
    id = "array_alias<" + id + ">";

  return id;
}


string
nid(pIIR_Declaration d, id_type typ)
{
  string id;
  char *decl;
  if (typ.object() & BARE) {  }
  else if (d->is(IR_PROCEDURE_DECLARATION)) 	 id="X";
  else if (d->is(IR_ARCHITECTURE_DECLARATION)) id="A";
  else if (d->is(IR_ENTITY_DECLARATION))       id="E";
  else if (d->is(IR_SIGNAL_DECLARATION) || d->is(IR_SIGNAL_INTERFACE_DECLARATION))
    {
      if (typ.object() & DRIVER)
	id = "D";
      else if (typ.object() & READER)
	id = "R";
      else if (typ.object() & SIGNAL)
	id = "S";
      else if (typ.object() & DEFAULT)
	id = "S";
    }
  else if (d->is(IR_TYPE_DECLARATION)) 
    {
      if (typ.object() & INFO)
	id = "I";
      else if (typ.object() & DEFAULT)
	id = "T";
      else if (typ.object() & TYPE)
	id = "T";
    }
  else if (d->is(IR_VARIABLE_DECLARATION)) id="V";
  else if (d->is(IR_VARIABLE_INTERFACE_DECLARATION)) id="V";
  else if (d->is(IR_PACKAGE_DECLARATION)) id="Q";
  else if (d->is(IR_CONSTANT_DECLARATION)) id="C";
  else if (d->is(IR_CONSTANT_INTERFACE_DECLARATION)) id="C";
  else if (d->is(IR_LIBRARY_DECLARATION)) id="L";
  else if (d->is(IR_CONCURRENT_STATEMENT)) id="P";
  else if (d->is(IR_COMPONENT_DECLARATION)) id="!";
  else if (d->is(IR_LOOP_DECLARATIVE_REGION)) id="L";
  else if (d->is(IR_ENUMERATION_LITERAL))
    return "enumeration(" + i_to_string(pIIR_EnumerationLiteral(d)->enum_pos) + ")";
  else { 
    cout << endl << "Can't find: " <<  d->kind_name() << endl ;
    id="?";
  }

  if (typ.object() == LIBRARY) 
    {
      id = "L";
      decl = pIIR_LibraryUnit(d)->library_name->text.to_chars();
    }
  else
    decl = d->declarator->text.to_chars();

  // Convert all uppercase letters to lowercase
  string str;
  if (!(typ.object() & BARE)) 
    str += id + i_to_string(strlen(decl));

  while (*decl) { str += (char)tolower(*decl++); }

  return str;
}



/* Return the prefix for the name of a new internal variable */
string
get_new_internal_var_prefix()
{
  return internal_prefix_start + i_to_string(internal_counter++);
}




/* Returns a pointer to the current active context info instance */
ContextInfo *
ActiveContext(RegionStack &rstack)
{
  pIIR_DeclarativeRegion r = ActiveDeclarativeRegion(rstack);

  return r != NULL? &context(r) : NULL;
}

pIIR_DeclarativeRegion
ActiveDeclarativeRegion(RegionStack &rstack)
{
  return rstack.back();
}


/* Returns the next process, subprogram or concurrent statement
   declarative region */

ContextInfo *
BaseContext(RegionStack &rstack)
{
  pIIR_DeclarativeRegion r = BaseDeclarativeRegion(rstack);

  return r != NULL? &context(r) : NULL;
}

pIIR_DeclarativeRegion
BaseDeclarativeRegion(RegionStack &rstack)
{
  if (rstack.size() == 0) return NULL;

  RegionStack::iterator iter = rstack.end();
  iter--;
  while (true) {
    // Search in reverse order until either a concurrent statement
    // region is found or the very first item on the list has been
    // visited
    if (!(*iter)->is(IR_CONCURRENT_STATEMENT) &&
	!(*iter)->is(IR_SUBPROGRAM_DECLARATION))
      return *iter;
    if (iter == rstack.begin()) break;
    iter--;
  }
  return NULL;
}


/* Returns the top level region which is either an architecture
   region, an entity region, a package body region or package declaration
   region */
ContextInfo *
TopContext(RegionStack &rstack) 
{
  pIIR_DeclarativeRegion r = TopDeclarativeRegion(rstack);

  return r != NULL? &context(r) : NULL;
}

pIIR_DeclarativeRegion
TopDeclarativeRegion(RegionStack &rstack)
{
  if (rstack.size() == 0) return NULL;

  pIIR_DeclarativeRegion found = NULL;

  for (RegionStack::iterator iter = rstack.begin(); iter != rstack.end(); iter++)
    if ((*iter)->is(IR_PACKAGE_BODY_DECLARATION) ||
	(*iter)->is(IR_PACKAGE_DECLARATION) ||
	(*iter)->is(IR_ARCHITECTURE_DECLARATION) ||
	(*iter)->is(IR_ENTITY_DECLARATION))
      found = *iter;
  
  return found;
}


// cast an expression
// string
// cast(pIIR_Expression e, string estr)
// {
//   if (e->subtype->declaration == NULL) {
//     if (e->subtype->is(IR_INTEGER_TYPE))
//        return  "integer(" + estr + ")";
//     else if (e->subtype->is(IR_FLOATING_TYPE))
//        return  "real(" + estr + ")";
//   } else
//     return  qid(e->subtype->declaration, "_", TYPE) + "(" + estr + ")";
// }


// un-cast an expression, i.e. return whats enclosed in barckets "()"
// string
// uncast(string &estr)
// {
//   int start = estr.find_first_of("(") + 1;
//   if (start >= estr.size())
//     return estr;

//   int end = estr.find_last_of(")");
//   if (end < start) 
//     internal_error(true);

//   return estr.substr(start, end - start);
// }


// convert an scalar expression into an int value. If boolean simple
// is true then str must contains a literal value.
// string
// to_int_string(string &str, bool simple)
// {
//   if (!simple) 
//     return "(" + str + ").value"; 
//   else 
//     return uncast(str);
// }

/* for debugging */
int
plot_rstack(RegionStack &rstack)
{
  for (RegionStack::iterator iter = rstack.begin(); iter != rstack.end(); iter++) {
    cout << hex << "0x" << (int)*iter << dec <<" = ";
    if ((*iter)->is(IR_PACKAGE_BODY_DECLARATION))
      cout << "IR_PACKAGE_BODY_DECLARATION" << endl;
    else if ((*iter)->is(IR_PACKAGE_DECLARATION))
      cout << "IR_PACKAGE_DECLARATION" << endl;
    else if  ((*iter)->is(IR_ARCHITECTURE_DECLARATION))
      cout << "IR_ARCHITECTURE_DECLARATION" << endl;
    else if ((*iter)->is(IR_ENTITY_DECLARATION))
      cout << "IR_ENTITY_DECLARATION" << endl;
    else if ((*iter)->is(IR_CONCURRENT_STATEMENT))
      cout << "IR_CONCURRENT_STATEMENT" << endl;
    else if ((*iter)->is(IR_SUBPROGRAM_DECLARATION))
      cout << "IR_SUBPROGRAM_DECLARATION" << endl;
    else
      cout << "???" << endl;
  }
}


/* for debugging */
void
plot(string &str) { cout << str; }



// Get the type declaration
pIIR_TypeDeclaration
get_declaration(pIIR_Type type)
{
  if (type->declaration == NULL) {
    if (type->is(IR_ARRAY_SUBTYPE))
      type = ((pIIR_ArraySubtype)(type))->immediate_base;
    else if (type->is(IR_SCALAR_SUBTYPE))
      type = ((pIIR_ScalarSubtype)(type))->immediate_base;
  }

  return type->declaration;
}


// Get the immediate_base type
pIIR_Type
get_immediate_base_type(pIIR_Type type)
{
  if (type->declaration == NULL) {
    if (type->is(IR_ARRAY_SUBTYPE))
      type = ((pIIR_ArraySubtype)(type))->immediate_base;
    else if (type->is(IR_SCALAR_SUBTYPE))
      type = ((pIIR_ScalarSubtype)(type))->immediate_base;
  }

  return type;
}


// Get base type of a type
pIIR_Type
get_base_type(pIIR_Type type)
{
  if (type != type->base)
    return get_base_type(type->base);
  else
    return type;
}


// pIIR_Range
// get_array_range(pIIR_Type t, int index_nr)
// {
//   pIIR_TypeList tlist = NULL;

//   if (t->is(IR_ARRAY_SUBTYPE)) {
//     pIIR_ArraySubtype as = pIIR_ArraySubtype(t);

//     if (as->constraint == NULL)
//       return get_array_range(get_immediate_base_type(as), index_nr);
    
//     tlist = as->constraint;
//     for (int i = 1; i < index_nr; i++)
//       tlist = tlist->rest;

//   } else if (t->is(IR_ARRAY_TYPE)) {
//     pIIR_ArrayType at = pIIR_ArrayType(t);
//     tlist = at->index_types;
//     for (int i = 1; i < index_nr; i++)
//       tlist = tlist->rest;
    
//   } else
//     internal_error(true);

//   if (tlist->first->is(IR_SCALAR_SUBTYPE))
//     return pIIR_ScalarSubtype(tlist->first)->range;
//   else
//     return NULL;
// }


// pIIR_Type
// get_array_range_type(pIIR_Type t, int index_nr)
// {
//     return NULL;
// }


// Convert locally static expression to int value. Returns false if
// the expression is not locally static.
bool
expr_to_int(pIIR_Expression e, int &value, RegionStack &rstack)
{
  // Return false if expression is not locally static
  if (e->static_level != IR_LOCALLY_STATIC)
    return false;

  pIIR_Type base = get_base_type(e->subtype);
  pIIR_AbstractLiteralExpression ae = pIIR_AbstractLiteralExpression(e);
  if (base->is(IR_INTEGER_TYPE)) {
    string val_str;
    if (!emit_expr(e, val_str, rstack, DEFAULT))
      return false;
    value = atoi(val_str.c_str());
  } else if (base->is(IR_ENUMERATION_TYPE)) {
    string literal_str = pIIR_EnumerationLiteral(ae->value)->declarator->text.to_chars();
    value = literal_to_pos(pIIR_EnumerationType(base)->enumeration_literals, literal_str);
  } else
    // Return false if type is not an enumeration type or integer type 
    return false;

  return true;
}


// Get enumeration pos of enumeration item
int 
literal_to_pos(pIIR_EnumerationLiteralList enum_items, const string &literal_str)
{
  for (pIIR_EnumerationLiteralList lit_list = enum_items; lit_list; lit_list = lit_list->rest) {
    if (lit_list->first->declarator->is(IR_IDENTIFIER)) { 
      // Identifier enumeration item
      if (!strcmp(literal_str.c_str(), lit_list->first->declarator->text.to_chars()))
	return lit_list->first->enum_pos;

    } else if (lit_list->first->declarator->text.to_chars()[1] == literal_str.c_str()[1])
      // Character enumeration item
      return lit_list->first->enum_pos;
  }
  return -1;
}


// Get enumeration literal associated with pos
pIIR_EnumerationLiteral
pos_to_literal(pIIR_EnumerationLiteralList enum_items, int literal_pos)
{
  for (pIIR_EnumerationLiteralList lit_list = enum_items; lit_list; lit_list = lit_list->rest)
    if (lit_list->first->enum_pos == literal_pos)
      return lit_list->first;

  return NULL;
}


// Create an internal acl object which is used to store acl values at
// simulation time. Return the name of the internal acl
// object. private_acl determines whether the object will be used
// privately or may be shared by several expressions.
string
create_internal_acl (list<string> &acl_list, RegionStack &rstack, bool private_acl)
{
  pIIR_DeclarativeRegion active_region = ActiveDeclarativeRegion(rstack);

  // generate a name for a new acl object. If the internal object is
  // private then the name is generated by adding a serial number to
  // the standard prefix name. Otherwise, an additional string "_acl"
  // is appended to the prefix and the number determines the max
  // length of the acl instance.
  string internal_var_name = internal_prefix_start + 
    (private_acl? i_to_string(internal_counter++) : "_acl" + i_to_string(acl_list.size()));
  
  if (lookup_internal_object_declaration(active_region, internal_var_name) == NULL)
    insert_internal_object_declaration(NULL, active_region, internal_var_name, "pacl", 
				       "=new(" + i_to_string(acl_list.size()) + ") acl",
				       0);

  return internal_var_name;
}



// Lookup an internal object declaration with name "name" defined in
// the declarative region "region"
pV2CC_InternalObjectDeclaration
lookup_internal_object_declaration(pIIR_DeclarativeRegion region, const string &name)
{
  for (pIIR_DeclarationList decls = region->declarations; decls; decls = decls->rest) {
    pIIR_Declaration obj = decls->first;
    if (obj->is(V2CC_INTERNAL_OBJECT_DECLARATION) &&
	!strcmp(name.c_str(), obj->declarator->text.to_chars()))
      return pV2CC_InternalObjectDeclaration(obj);
  }

  return NULL;
}


// Return address of the last member from the last item on the
// list. If the list is empty then return listp. Note that listp is
// the ADDRESS of an pointer which points to the next item on the list
// (or to NULL).
pIIR_DeclarationList *
get_last_rest_address(pIIR_DeclarationList *listp)
{
  while (*listp)
    listp = &(*listp)->rest;

  return listp;
}


// Return C++ type string of an internal object
string
get_internal_object_type_string(pV2CC_InternalObjectDeclaration decl, RegionStack &rstack)
{
  // Test whether cpp_type_string contains a valid string value
  if (decl->cpp_type_string != "")
    return decl->cpp_type_string;
  else
    return qid(decl, "_", TYPE);
}

// Return inital value string of an internal object
string
get_internal_object_initial_string(pV2CC_InternalObjectDeclaration decl, RegionStack &rstack)
{
  // Test whether cpp_inital_string contains an valid type string
  if (decl->cpp_initial_string != "")
    return decl->cpp_initial_string;
  else {
    string str = "";
    if (decl->initial_value)
      emit_expr (decl->initial_value, str, rstack, DEFAULT);
    
    return str;
  }
}


// Merge two static levels. The resulting level will be equal to the
// smallest level of both.
inline IR_StaticLevel
merge_level(IR_StaticLevel sl1, IR_StaticLevel sl2)
{
  if (IR_LOCALLY_STATIC == sl1 && IR_LOCALLY_STATIC == sl2)
    return IR_LOCALLY_STATIC;
  if (level_match(sl1, IR_GLOBALLY_STATIC) &&
      level_match(sl2, IR_GLOBALLY_STATIC)) 
    return IR_GLOBALLY_STATIC;
  return IR_NOT_STATIC;
}


// Get folded value from IIR_Root node
scalar_data &
get_folded_value(pIIR_Root n) 
{
  if (n->is(IR_EXPRESSION))
    return folded_value(pIIR_Expression(n));
  else if (n->is(IR_ENUMERATION_LITERAL))
    return folded_value(pIIR_EnumerationLiteral(n));
  else if (n->is(IR_LITERAL))
    return folded_value(pIIR_Literal(n));
  else
    internal_error(true);
}

int 
get_scalar_count(pIIR_Type type, bool &success)
{
  if (type->is(IR_ARRAY_TYPE)) {
    
    return get_scalar_count(pIIR_ArrayType(type)->element_type, success);

  } else if (type->is(IR_ARRAY_SUBTYPE)) {

  } else if (type->is(IR_RECORD_TYPE)) {

    // Sorry, but records are not supported yet
    internal_error(true);

  } else {
    success = true;
    return 1;
  }
}


// ******************************************************************************************
// Name: insert_internal_object_declaration, set of functions
//
// Description: Creates an internal object declaration and inserts it
// into the corresponding declaration list. The new item is inserted
// right before *listp. region points to the declarative region the
// new item belongs to. type_str is the C++ type of the item while
// initial_str stores its initial value. Alternatively, the type may
// be specified by a pIIR_Type pointer and the intial value may be
// defined by a pIIR_Expression.
// 
// Return value: returns pointer to new internal object declaration
//
// ******************************************************************************************

pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, const string &type_str, const string &initial_str, 
				   int flags)
{
  // If listp is not specified then append the new item to the end of
  // the list
  if (listp == NULL)
    listp = get_last_rest_address(&region->declarations);

  pV2CC_InternalObjectDeclaration obj =
    new V2CC_InternalObjectDeclaration(NULL /* pIIR_PosInfo */, 
				       to_TextLiteral(name) /* declarator name */, 
				       region /* declaration region */, 
				       NULL /* attribute value list */, 
				       0 /* seqno */ , 
				       NULL /* type */ , 
				       NULL /* initial value expression */, 
				       (string)type_str /* type string */, 
				       (string)initial_str /* initial value string */, 
				       flags /* some flags */);
  *listp = new IIR_DeclarationList(obj->pos, obj, *listp);

  return obj;
}

pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, pIIR_Type type, const string &initial_str, 
				   int flags)
{
  // If listp is not specified then append the new item to the end of
  // the list
  if (listp == NULL)
    listp = get_last_rest_address(&region->declarations);

  pV2CC_InternalObjectDeclaration obj =
    new V2CC_InternalObjectDeclaration(NULL /* pIIR_PosInfo */, 
				       to_TextLiteral(name) /* declarator name */, 
				       region /* declaration region */, 
				       NULL /* attribute value list */, 
				       0 /* seqno */ , 
				       type /* type */ , 
				       NULL /* initial value expression */, 
				       "" /* type string */, 
				       (string)initial_str /* initial value string */, 
				       flags /* some flags */);
  *listp = new IIR_DeclarationList(obj->pos, obj, *listp);

  return obj;
}


pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, const string &type_str, pIIR_Expression initial, 
				   int flags)
{
  // If listp is not specified then append the new item to the end of
  // the list
  if (listp == NULL)
    listp = get_last_rest_address(&region->declarations);

  pV2CC_InternalObjectDeclaration obj =
    new V2CC_InternalObjectDeclaration(NULL /* pIIR_PosInfo */, 
				       to_TextLiteral(name) /* declarator name */, 
				       region /* declaration region */, 
				       NULL /* attribute value list */, 
				       0 /* seqno */ , 
				       NULL /* type */ , 
				       initial /* initial value expression */, 
				       (string)type_str /* type string */, 
				       "" /* initial value string */, 
				       flags /* some flags */);
  *listp = new IIR_DeclarationList(obj->pos, obj, *listp);

  return obj;
}


pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, pIIR_Type type, pIIR_Expression initial, int flags)
{
  // If listp is not specified then append the new item to the end of
  // the list
  if (listp == NULL)
    listp = get_last_rest_address(&region->declarations);

  pV2CC_InternalObjectDeclaration obj =
    new V2CC_InternalObjectDeclaration(NULL /* pIIR_PosInfo */, 
				       to_TextLiteral(name) /* declarator name */, 
				       region /* declaration region */, 
				       NULL /* attribute value list */, 
				       0 /* seqno */ , 
				       type /* type */ , 
				       initial /* initial value expression */, 
				       "" /* type string */, 
				       "" /* initial value string */, 
				       flags /* some flags */);
  *listp = new IIR_DeclarationList(obj->pos, obj, *listp);

  return obj;
}


// ******************************************************************************************
// Name: insert_internal_code, function
//
// Description: Creates an internal object which contains code to be
// included into the generated C++ code. hdr contains header code to
// be prepended to the C++ code while impl stores the actual
// implementation code. Usually, this object is used to store the code
// generated from VHDL subporogram declarations. In this case, hdr
// contains the C++ interface definition of the subprogram while impl
// stores its implementation code.
// 
// Return value: returns pointer to new internal object declaration
//
// ******************************************************************************************

pV2CC_InternalCode
insert_internal_code(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
		     const string &name, const string &hdr, const string &impl,
		     int flags)
{
  // If listp is not specified then append the new item to the end of
  // the list
  if (listp == NULL)
    listp = get_last_rest_address(&region->declarations);

  pV2CC_InternalCode obj =
    new V2CC_InternalCode(NULL /* pIIR_PosInfo */, 
			  to_TextLiteral(name) /* declarator name */, 
			  region /* declaration region */, 
			  NULL /* attribute value list */, 
			  0 /* seqno */ , 
			  NULL /* type */ , 
			  NULL /* initial value expression */, 
			  (string)hdr /* header string */, 
			  (string)impl /* initial value string */,
			  flags /* some flags */ );
  *listp = new IIR_DeclarationList(obj->pos, obj, *listp);

  return obj;
}


// ******************************************************************************************
// Name: m_get_object_declaration , generic function
//
// Description: Returns the object declaration a reference expression
// is based on. E.g. an expression "sig(3 downto 1)" will return the
// object delcration pointer associated with "sig". Note that if the
// prefix is a signal kind attribute then this signal attribute object
// is returned!
// 
// Return value: returns object declaration
//
// ******************************************************************************************


pIIR_ObjectDeclaration
m_get_object_declaration(pIIR_ObjectReference or)
{
  return NULL;
}


pIIR_ObjectDeclaration
m_get_object_declaration(pIIR_SimpleReference sr)
{
  return sr->object;
}


pIIR_ObjectDeclaration
m_get_object_declaration(pIIR_SliceReference slr)
{
  return get_object_declaration(slr->array);
}


pIIR_ObjectDeclaration
m_get_object_declaration(pIIR_AttrSignalRef asr)
{
  codegen_error.error("%:error: sorry, currently no signal kind attributes are supported", asr);
  return NULL;
}


pIIR_ObjectDeclaration
m_get_object_declaration(pIIR_ArrayReference ar)
{
  return get_object_declaration(ar->array);
}


pIIR_ObjectDeclaration
m_get_object_declaration(pIIR_RecordReference rr)
{
  return get_object_declaration(rr->record);
}


// ******************************************************************************************
// Name: m_get_discrete_range , generic function
//
// Description: Returns an array of RangeDescriptors derived from the
// first parameter. Note that usually this function is called to
// determine the bounds on an scalar type or the index bounds on an
// array. In most cases it will return an array containing only a
// single element. However, when called to extract the index ranges
// from an array several RangeDescriptors may be returned where each
// one covers one index range.
// 
// Return value: RangeDescriptor (see v2cc.h)
//
// ******************************************************************************************

vector<RangeDescriptor>
m_get_discrete_range(pIIR_EnumerationType et, RegionStack &rstack, IR_StaticLevel slevel)
{
  vector<RangeDescriptor> bounds;

  pIIR_EnumerationLiteral left, right;
  
  left = et->enumeration_literals->first;
  for (pIIR_EnumerationLiteralList el = et->enumeration_literals; el; el = el->rest)
    if (el->rest == NULL)
      // Store right bound
      right = el->first;
      
  bounds.push_back(RangeDescriptor(left, IR_DIRECTION_UP, right, IR_LOCALLY_STATIC));

  return bounds;
}


vector<RangeDescriptor>
m_get_discrete_range(pIIR_IntegerType it, RegionStack &rstack, IR_StaticLevel slevel)
{
  return get_discrete_range(pIIR_ScalarSubtype(it->declaration->type)->range, rstack, slevel);
}


vector<RangeDescriptor>
m_get_discrete_range(pIIR_PhysicalType pt, RegionStack &rstack, IR_StaticLevel slevel)
{
  // Not tested!
  return get_discrete_range(pt->base, rstack, slevel);
}


vector<RangeDescriptor>
m_get_discrete_range(pIIR_ScalarSubtype st, RegionStack &rstack, IR_StaticLevel slevel)
{
  if (st->range == NULL) {
    // If no discrete array range is defined then try to get the range
    // from the base type of the current type
    return  get_discrete_range(st->immediate_base, rstack, slevel);

  } else {
    // Get the range of the type
    vector<RangeDescriptor> bounds = get_discrete_range(st->range, rstack, slevel);
    // Test static level of the exracted range(s). If no bound with
    // the required static range could be extracted then try to
    // extract an appropriate range from the base type of the current
    // type. Otherwise return bounds.
    if (bounds.size() == 0)
      return get_discrete_range(st->immediate_base, rstack, slevel);
    else
      return bounds;
  }
}


vector<RangeDescriptor>
m_get_discrete_range(pIIR_ExplicitRange r, RegionStack &rstack, IR_StaticLevel slevel)
{
  vector<RangeDescriptor> bounds;

  // If both bounds matche the required static level then create a
  // corresponding RangeDescriptor from the range. Otherwise return an
  // empty RangeDescriptor vector.
  if (level_match(r->left->static_level, slevel) &&
      level_match(r->right->static_level, slevel)) {
    IR_StaticLevel new_level = merge_level(r->left->static_level, r->right->static_level);
    bounds.push_back(RangeDescriptor(r->left, r->direction, r->right, new_level));
  }

  return bounds;
}


vector<RangeDescriptor>
m_get_discrete_range(pIIR_ArraySubtype ast, RegionStack &rstack, IR_StaticLevel slevel)
{
  vector<RangeDescriptor> bounds;

  for (pIIR_TypeList tl = ast->constraint; tl; tl = tl->rest) {
    vector<RangeDescriptor> new_bound = get_discrete_range(tl->first, rstack, slevel);
    if (new_bound.size() > 1)
      internal_error(true);
    bounds.push_back(new_bound[0]);
  }

  return bounds;
}


vector<RangeDescriptor>
m_get_discrete_range(pIIR_ArrayType at, RegionStack &rstack, IR_StaticLevel slevel)
{
  vector<RangeDescriptor> bounds;

  for (pIIR_TypeList tl = at->index_types; tl; tl = tl->rest) {
    vector<RangeDescriptor> new_bound = get_discrete_range(tl->first, rstack, slevel);
    if (new_bound.size() > 1)
      internal_error(true);
    bounds.push_back(new_bound[0]);
  }

  return bounds;
}


