
#ifndef V2CC_UTIL_H 
#define V2CC_UTIL_H

#include <freehdl/vaul.h>
#include "v2cc-chunk.h"

// ******************************************************************************************
// Some global variables
// ******************************************************************************************

// Counter to produce unique names for internal variables
extern int internal_counter;
extern const string internal_prefix_start;

// Prefix used to build a default name for a process
extern const string process_default_postfix;

// Prefix used to build a default name for loop statement
extern const string loop_default_postfix;

// ******************************************************************************************

// Print out an internal error message and stop compilation if "stop"
// is true
void
print_internal_error(char *file, int line, bool stop);
#define internal_error(b) print_internal_error(__FILE__, __LINE__, b)

/* Get line number associated with a node */
int
get_line_number(pIIR_Root n);

// Converts a string item into an IR_TextLiteral
pIIR_TextLiteral
to_TextLiteral(const string &str);

/* Return a string of L spaces, but at most 30.
*/
string 
spaces(int l);

/* Emit an UglyIdentifier as a beautiful_identifier.
*/
void emit_id(pIIR_TextLiteral idnode, string &str);

// get a fully qualified identifier
string
qid (pIIR_Declaration d, char *qual="_", id_type typ=id_type());

string
nid(pIIR_Declaration d, id_type typ=id_type());

/* Select association element from an association list which
 * corresponds with given formal */
pIIR_AssociationElement
find_matching_actual(pIIR_AssociationList assocs, pIIR_InterfaceDeclaration formal);

/* Test whether a type is a scalar */
extern bool
is_scalar_type(pIIR_Type t);

/* Test whether an type is an constrained array type */
bool
is_constrained_array_type(pIIR_Type t);

/* Test whether an type is an implicit defined array
 * subtype. E.g. when a variable "varaible var : bit_vector(0 to 7)"
 * is declared then an implicit subtype "bit_vector(0 to 7)" is
 * defined. */
bool 
is_implicit_array_subtype(pIIR_Type t);

/* Convert an integer value into a string */
string
i_to_string(int i);

/* Convert a double value into a string */
string
d_to_string(double d);

/* Convert an string value into a integer */
int
string_to_i(const string &str);

/* Convert an string value into a double */
double
string_to_d(const string &str);

// Get folded value from IIR_Root node
scalar_data &
get_folded_value(pIIR_Root n);

/* Concatenate all elements of a string list */
string
concat(list<string> lstr, string sep);

/* Concatenate all elements of a string vector */
string
concat(vector<string> vstr, string sep);

/* Return the prefix for the name of a new internal variable */
string
get_new_internal_var_prefix();

/* Returns a pointer to the current active context info instance */
ContextInfo *
ActiveContext(RegionStack &rstack);

pIIR_DeclarativeRegion
ActiveDeclarativeRegion(RegionStack &rstack);

/* Returns the next process, subprogram or concurrent statement
   declarative region */
extern ContextInfo *
BaseContext(RegionStack &rstack);

pIIR_DeclarativeRegion
BaseDeclarativeRegion(RegionStack &rstack);

/* Returns the top level region which is either an architecture
   region, an entity region, a package body region or package declaration
   region */
ContextInfo *
TopContext(RegionStack &rstack);

pIIR_DeclarativeRegion
TopDeclarativeRegion(RegionStack &rstack);

/* for debugging */
extern int
plot_rstack(RegionStack &rstack);

/* for debugging */
extern void
plot(string &str);

// Determine name of a subprogram. The name is generated as usual,
// however an sequence number is added to the name in order to
// separate functions with the same parameter set but different return
// value. The sequence number is a unique number within a design
// unit. Note that in VHDL two functions with the same parameter set
// but different return type may be defined while in C++ this is not
// permitted.
string get_subprogram_name(pIIR_SubprogramDeclaration spd);

// Get the type declaration
pIIR_TypeDeclaration
get_declaration(pIIR_Type type);

// Get the immediate_base type
pIIR_Type
get_immediate_base_type(pIIR_Type type);

// Get base type of a type
pIIR_Type
get_base_type(pIIR_Type type);

/* pIIR_Range */
/* get_array_range(pIIR_Type t, int index_nr); */

/* pIIR_Type */
/* get_array_range_type(pIIR_Type t, int index_nr); */

/* Emit code to create an new object of a specific type */
string
create_default_instance(pIIR_Type type, RegionStack &rstack);

/* Creates an array info object for an array subtype */
string
create_array_info_obj(pIIR_ArraySubtype as, RegionStack &rstack, const bool static_object);


// Convert locally static expression to int value. Returns false if
// the expression is not locally static.
bool
expr_to_int(pIIR_Expression e, int &value, RegionStack &rstack);

// Get enumeration pos of enumeration item
int 
literal_to_pos(pIIR_EnumerationLiteralList enum_items, const string &literal_str);

// Get enumeration literal associated with pos
pIIR_EnumerationLiteral
pos_to_literal(pIIR_EnumerationLiteralList enum_items, int literal_pos);

// Create an internal acl object which is used to store acl values at
// simulation time. Return the name of the internal acl
// object. private_acl determines whether the object will be used
// privately or may be shared by several expressions.
string
create_internal_acl (list<string> &acl_list, RegionStack &rstack, bool private_acl);

// Lookup an internal object declaration with name "name" defined in
// the declarative region "region"
pV2CC_InternalObjectDeclaration
lookup_internal_object_declaration(pIIR_DeclarativeRegion region, const string &name);


// Return C++ type string of an internal object
string
get_internal_object_type_string(pV2CC_InternalObjectDeclaration decl, RegionStack &rstack);

// Return inital value string of an internal object
string
get_internal_object_initial_string(pV2CC_InternalObjectDeclaration decl, RegionStack &rstack);


// Test whether static level level matches the required static level
// test. The function returns true if the static level "level" is
// greater or equal compared to level test.
inline bool 
level_match(IR_StaticLevel test, IR_StaticLevel level) {
  switch (level) {
  case IR_LOCALLY_STATIC:
    if (test == IR_GLOBALLY_STATIC) return false;
  case IR_GLOBALLY_STATIC:
    if (test == IR_NOT_STATIC) return false;
  case IR_NOT_STATIC:
    break;
  };
  return true;
}


// Merge two static levels. The resulting level will be equal to the
// smallest level of both.
inline IR_StaticLevel
merge_level(IR_StaticLevel sl1, IR_StaticLevel sl2);


// ******************************************************************************************
// Name: insert_internal_object_declaration, set of functions
//
// Description: Creates an internal object declaration and inserts it
// into the corresponding declaration list. The new item is inserted
// right before *listp. region points to the declarative region the
// new item belongs to. type_str is the C++ type of the item while
// initial_str stores its initial value. Alternatively, the type may
// be specified by a pIIR_Type pointer and the intial value may be
// defined by a pIIR_Expression.
// 
// Return value: returns pointer to new internal object declaration
//
// ******************************************************************************************

pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, const string &type_str, const string &initial_str, 
				   int flags);
pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, pIIR_Type type, const string &initial_str, 
				   int flags);
pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, const string &type_str, pIIR_Expression initial, 
				   int flags);
pV2CC_InternalObjectDeclaration
insert_internal_object_declaration(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
				   const string &name, pIIR_Type type, pIIR_Expression initial, int flags);


// ******************************************************************************************
// Name: insert_internal_code, function
//
// Description: Creates an internal object which contains code to be
// included into the generated C++ code. hdr contains header code to
// be prepended to the C++ code while impl stores the actual
// implementation code. Usually, this object is used to store the code
// generated from VHDL subporogram declarations. In this case, hdr
// contains the C++ interface definition of the subprogram while impl
// stores its implementation code.
// 
// Return value: returns pointer to new internal object declaration
//
// ******************************************************************************************

pV2CC_InternalCode
insert_internal_code(pIIR_DeclarationList *listp, pIIR_DeclarativeRegion region,
		     const string &name, const string &hdr, const string &impl,
		     int flags);

#endif
