
/* VHDL to C++ translator    

   Copyright (C) 1999, 2000 Edwin Naroska.

   V2CC is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   V2CC is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with V2CC; see the file COPYING.  If not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.

 */

/* This program tanslates the original VHDL source into c++.
 
   Usage:

   v2cc [-v -l lib] file...

       -v             verbose
       -l lib         use lib as the WORK library, default is "."

       file...        the design units to translate

*/

#include <freehdl/vaul.h>
#include "v2cc-chunk.h"
#include <strstream>
#include <algo.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>
#include <unistd.h>

#include "mapping.h"
#include "v2cc-util.h"



// Declare here for now -- TLD
void emit_includes(string &str);
void emit_decls (pIIR_DeclarationList decl_list, string &str, RegionStack &rstack, int l);
void emit_sig_decls (pIIR_InterfaceList interface_list, string &str, RegionStack &rstack, int l);
void emit_handle(pIIR_ArchitectureDeclaration a, string &str);
void emit_sig_interfacecon (pIIR_SignalInterfaceDeclaration s, string &str, RegionStack &rstack, int i);
void emit_constructor(pIIR_ArchitectureDeclaration a, string &str, RegionStack &rstack);
void emit_process_body (pIIR_ProcessStatement p, string &str, RegionStack &rstack, int l);
void emit_decls_init (pIIR_DeclarationList decl_list, string &str, RegionStack &rstack, int l);
string sprint_acl(list<string> &acl_list, const string acl_object);
pIIR_LibraryUnit get_library_unit(pIIR_DeclarativeRegion d);
void emit_impl_decl (pIIR_ConcurrentStatementList s, string &str, int l);

// used to generate error messages
vaul_error_printer codegen_error;


// ******************************************************************************************
// Implementation of member functions declared in v2cc.h
// ******************************************************************************************

// Returns true if range r and the current instance cover the same
// range
bool 
RangeDescriptor::is_equal_to(const RangeDescriptor &r) 
{
  if (range_attribute != NULL)
    internal_error(true); // Currently not supported
  else
    return (left == r.left) && (right = r.right) && (direction == r.direction);
}

// Convert a RangeDescriptor into corresponding integer values and the
// direction. True is returned when the conversion could be done,
// i.e. when the bounds were locally static. Otherwise false is
// returned
bool
RangeDescriptor::rangedes_to_int(int &left_value, IR_Direction &dir_value, int &right_value, 
				 RegionStack &rstack)
{
  if (range_attribute != NULL)
    internal_error(true);

  string str;
  bool ok = true;
  
  // Try to convert the left bounds to integer values
  constant_fold(left, rstack);
  if (!left->is(IR_EXPRESSION) ||
      pIIR_Expression(left)->static_level == IR_LOCALLY_STATIC) {
    left_value = get_folded_value(left).long_value;
  } else
    ok = false; // If the expression is not simple then flag an error

  // Store range direction
  dir_value = direction;

  // Try to convert the right bounds to integer values
  constant_fold(right, rstack);
  if (!right->is(IR_EXPRESSION) ||
      pIIR_Expression(right)->static_level == IR_LOCALLY_STATIC)
    right_value = get_folded_value(right).long_value;
  else
    ok = false; // If the expression is not simple then flag an error

  return ok;
}


// Convert a RangeDescriptor into corresponding integer value strings
// and the direction strings. If a bound is not locally static then
// appropriate code is emitted to extract the corresponding integer
// value from the object.
void
RangeDescriptor::rangedes_to_int_string(string &left_str, string &dir_str, string &right_str, 
					RegionStack &rstack)
{
  if (range_attribute != NULL)
    internal_error(true);

  bool simple;
  
  // Convert the left bounds to integer values
  constant_fold(left, rstack);
  simple = emit_expr (left, left_str, rstack, DEFAULT);

  // Store range direction
  dir_str = direction == IR_DIRECTION_UP? "to" : "downto";
  
  constant_fold(right, rstack);
  simple = emit_expr (right, right_str, rstack, DEFAULT);
}



int
RangeDescriptor::constant_fold_rangedes(RegionStack &rstack)
{
  int error_count = 0;

  if (range_attribute != NULL)
    internal_error(true);

  if (left != NULL && 
      left->is(IR_EXPRESSION))
    error_count += constant_fold(pIIR_Expression(left), rstack);
  else
    error_count += constant_fold(pIIR_Literal(left), rstack);

  if (right != NULL &&
      right->is(IR_EXPRESSION))
    error_count += constant_fold(pIIR_Expression(right), rstack);
  else
    error_count += constant_fold(pIIR_Literal(right), rstack);

  return error_count;
}





// ******************************************************************************************
// Macros to generate standard operations (+,-,...) for the predefined
// VHDL operators
// ******************************************************************************************

// Stores macros to generate code for predefined unary and binary
// operators. The variables are associative arrays indexed by the VHDL
// operator name. Example: "unary_operator_macro[string("\"and\"")]"
// will return the string "(%s&&%s)". The returned string may be used as
// a format string for a "sprintf" function.
map_string2 unary_operator_macro;
map_string2 binary_operator_macro;



// ******************************************************************************************
// Initialize some v2cc internal data structures
// ******************************************************************************************

void
init_operator_macros()
{
  // Add uniary operator macros
  unary_operator_macro["\"not\""] = "(!%s)";
  unary_operator_macro["\"+\""] = "(+%s)";
  unary_operator_macro["\"-\""] = "(-%s)";

  // Add binary operator macros
  binary_operator_macro["\"and\""] = "(%s&&%s)";
  binary_operator_macro["\"or\""] = "(%s||%s)";
  binary_operator_macro["\"xor\""] = "op_xor(%s,%s)";
  binary_operator_macro["\"xnor\""] = "op_xnor(%s,%s)";
  binary_operator_macro["\"nand\""] = "op_nand(%s,%s)";
  binary_operator_macro["\"nor\""] = "op_nor(%s,%s)";
  binary_operator_macro["\"=\""] = "(%s==%s)";
  binary_operator_macro["\">\""] = "(%s>%s)";
  binary_operator_macro["\">=\""] = "(%s>=%s)";
  binary_operator_macro["\"<\""] = "(%s<%s)";
  binary_operator_macro["\"<=\""] = "(%s<=%s)";
  binary_operator_macro["\"/=\""] = "(%s!=%s)";
  binary_operator_macro["\"+\""] = "(%s+%s)";
  binary_operator_macro["\"-\""] = "(%s-%s)";
  binary_operator_macro["\"/\""] = "(%s/%s)";
  binary_operator_macro["\"*\""] = "(%s*%s)";
  binary_operator_macro["\"mod\""] = "mod(%s,%s)";
  binary_operator_macro["\"rem\""] = "rem(%s,%s)";
  binary_operator_macro["\"**\""] = "power(%s,%s)";
  binary_operator_macro["\"&\""] = "concat";
}


/* Generate string for a call to a binary operator */
string
operator_call_string(pIIR_FunctionDeclaration fd, const string arg1, const string arg2)
{
  string &macro = binary_operator_macro[fd->declarator->text.to_chars()];
  int size = macro.length() + arg1.length() + arg2.length() + 1; // Actually "size" is an upper bound
  char *str = (char*)alloca(sizeof(char)*size);
  sprintf(str, macro.c_str(), arg1.c_str(), arg2.c_str());
  return string(str);
}


/* Generate string for a call to a unary operator */
string
operator_call_string(pIIR_FunctionDeclaration fd, const string arg1)
{
  string &macro = unary_operator_macro[string(fd->declarator->text.to_chars())];
  int size = macro.length() + arg1.length() + 1; // Actually "size" is an upper bound
  char *str = (char*)alloca(sizeof(char)*size);
  sprintf(str, macro.c_str(), arg1.c_str());
  return string(str);
}



/* Exit the program with a usage message.  */
void
usage ()
{
  fprintf (stderr, "usage: %s [-v] [-l lib] [-L libdir] file...\n", vaul_application_name);
  exit (1);
}

bool verbose = false;
int parser_flags = 0;

struct mypool : vaul_pool {
  mypool ();
  vaul_design_unit *get (char *library, char *name);
  v2cc_mapper *mapper;
};

bool try_vhdl_source (char *fn)
{
  for (char *cp = fn; *cp; cp++)
    *cp = tolower(*cp);
  return access (fn, R_OK) == 0;
}

char *
find_vhdl_source (char *l, char *n)
{
  char *fn = vaul_aprintf ("../libvaul/vlib/%s/%s.vhd", l, n);
  if (try_vhdl_source (fn))
    return fn;
  free (fn);
  fn = vaul_aprintf ("../libvaul/vlib/%s/%s.vhdl", l, n);
  try_vhdl_source (fn);
  return fn;
}

mypool::mypool () {
  mapper = make_v2cc_mapper ();
}

vaul_design_unit *
mypool::get (char *l, char *n)
{
  vaul_design_unit *du = vaul_pool::get (l, n);
  if (du == NULL)
    {
      char *fn = mapper->find_design_file (l, n);

      if (fn == NULL)
	return NULL;

      fprintf (stderr, "for %s/%s: reading %s\n", l, n, fn);
      vaul_design_file df(fn, NULL, parser_flags);

      begin_session (l);
      while (vaul_design_unit * du = df.read_design_unit (this))
	{
	  insert (du);
	  if (du->is_error ())
	    du->print_err (fn);
	  du->release ();
	}
      if (df.is_error ())
	df.print_err (fn);
      end_session ();

      delete fn;

      du = vaul_pool::get (l, n);
    }

  return du;
}

mypool vaul;

extern int optind, opterr;
extern char *optarg;
extern "C" int getopt (int, char *const *, const char *);

/* Parse FILE and output translation of all contained units to STDOUT.
   In case of failure, prints some messages to STDERR and
   return FALSE, else returns TRUE.
*/
bool emit (vaul_pool *pool, char *file, string &str);

/* Various translators for various node types. The occasional
   INT parameter controls indentation.
*/
void emit_id (pIIR_Declaration, string &);
void emit_id (pIIR_Type, string &);
bool emit_associations (string &str, RegionStack &rstack,
			pIIR_AssociationList assocs,
			pIIR_InterfaceList formals);
void emit_builtin_id (IR_Kind, string &);
void emit_lit (pIIR_Literal lit, string &);

void init_v2cc_chunk ();

bool dry_run = false;

int
main (int argc, char *argv[])
{
  int opt;
  char *libname = "work";
  
  vaul_application_name = "v2cc";

  /* ************************************************************** */
  /* Initialize some internal data structures of the code generator */
  /* ************************************************************** */
  init_operator_macros();
  /* ************************************************************** */
  
  opterr = 0;
  
  while ((opt=getopt (argc, argv, "vnl:L:")) != -1)
    {
      switch (opt)
	{
	case 'v':
	  verbose = true;
	  parser_flags |= VAUL_PARSER_VERBOSE;
	  tree_set_verbose (true);
	  break;
	case 'l':
	  libname =  optarg;
	  break;
	case 'L':
	  vaul.mapper->add_libdir (optarg);
	  break;
	case 'n':
	  dry_run = true;
	  break;
	case '?':
	  usage ();
	  break;
	}
    }
  
  vaul.mapper->add_default_libdirs ();

  if (optind >= argc || argc-1 > optind)
    usage ();
  
  init_v2cc_chunk ();
  
  bool success = true;
  vaul.begin_session (libname);
  string str;
  emit_includes(str);					// #includes for sim
  while (optind < argc)
    {
      success = emit (&vaul, argv[optind], str) && success;
      optind++;
    }
  vaul.end_session ();
  return success? 0 : 1;
}


bool
emit (vaul_pool *pool, char *file, string &str)
{
  vaul_design_file df(file, NULL, parser_flags);

  bool success = true;
  while (vaul_design_unit *du = df.read_design_unit (pool))
    {
      pool->insert (du); 
      if (du->is_error ())
	{
	  du->print_err (file);
	  success = false;
	}
      else if (!dry_run)
	{
	  fprintf (stderr, "emitting %s/%s\n",
		   du->get_library (), du->get_name ());
	  RegionStack rstack;
	  string code;
	  emit (du->get_tree (), code, rstack, 0);
	  str += code;
	  if (codegen_error.n_errors)
	    exit(1);
	}
      du->release();
    }

  cout << str; // just for debugging

  return success;
}


// emit a plain identifier 
void
emit_noqual_id (pIIR_Declaration d, string &str, id_type typ=id_type())
{
  if (d->declarator) 
    {
      if (d->is(IR_ENUMERATION_LITERAL) && verbose) 
	{
	  emit_noqual_id (pIIR_EnumerationLiteral(d)->subtype->declaration, str);
	  str += "_";
	  if (d->declarator->is(IR_CHARACTER_LITERAL)) 
	    {
	      IR_Character ch = d->declarator->text[1];
	      if (isalnum(ch))
		str += ch;
	      else
		str += unsigned(ch);
	    }
	  else
	    str += nid(d, typ);
	}
      else
	{
	  str += nid(d, typ);

	  if (d->is(IR_SUBPROGRAM_DECLARATION) && verbose) 
	    {
	      for(pIIR_InterfaceList il =
		    pIIR_SubprogramDeclaration(d)->interface_declarations;
		  il; il = il->rest)
		{
		  str += "__";
		  str += qid (il->first->subtype->declaration, "_");
		}
	      if(d->is(IR_FUNCTION_DECLARATION))
		{
		  str += "__";
		  str += qid (pIIR_FunctionDeclaration(d)->return_type->declaration, "_");
		}
	    }
	}
    }
  else
    str += "<anonymous>";
}


/* Emit a declaration name */
void
emit_id (pIIR_Declaration d, string &str)
{
  emit_noqual_id (d, str);
}


/* Emit a type name */
void
emit_id (pIIR_Type t, string &str)
{
  if (t->declaration)
    emit_id (t->declaration, str);
  else
    str += "<anonymous" + string(t->kind_name()) + ">";
}


void
emit_decls (pIIR_DeclarationList decl_list, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration type = NULL;
  string type_str = "";
  string old_type_str = "";
  string obj_str = "";

  // Analyze each item of the declaration list
  for (pIIR_DeclarationList dl = decl_list; dl; dl = dl->rest) {
    pIIR_Declaration decl = (pIIR_Declaration)dl->first;

    // Now, generate code depending of the object type
    if (decl->is(IR_SIGNAL_DECLARATION)) { 
      // **************************************************************************
      // signal declaration
      // **************************************************************************
      pIIR_SignalDeclaration sig = (pIIR_SignalDeclaration)dl->first;
      type_str = "sig_info<" + qid(get_declaration(sig->subtype), "_", TYPE) + ">";
      obj_str = "*" + qid(sig);

    } else if (decl->is(IR_CONSTANT_DECLARATION) ||
	       decl->is(IR_VARIABLE_DECLARATION)) {
      // **************************************************************************
      // constant or variable declaration
      // **************************************************************************
      type_str = qid(get_declaration(((pIIR_ObjectDeclaration)decl)->subtype), "_", TYPE);
      obj_str = qid(decl);

    } else if (decl->is(V2CC_INTERNAL_OBJECT_DECLARATION)) {
      // **************************************************************************
      // an internal object required by the generated code
      // **************************************************************************
      pV2CC_InternalObjectDeclaration obj = pV2CC_InternalObjectDeclaration(decl);
      // skip item if it should be declared globally (file scope)
      if (obj->flags & DECLARE_GLOBAL) continue;

      type_str = get_internal_object_type_string(obj, rstack);
      obj_str = obj->declarator->text.to_chars();

    } else if (decl->is(VAUL_PREDEF_OP)) {
      // Predefined VHDL operators are implicitly defined. Hence,
      // there is nothing to do here.
      continue;

    } else if (decl->is(IR_SUBPROGRAM_DECLARATION)) {
      // **************************************************************************
      // emit subprogram
      // **************************************************************************
      string hdr, impl;
      // plot subprogram prototype into string
      emit_hdr(decl, hdr, rstack, 0);
      // plot complete subprogram declaration into string
      emit_impl(decl, impl, rstack, 0);
      // append internal code object to top region. This code is then
      // emitted globally.
      pIIR_DeclarativeRegion top_region = TopDeclarativeRegion(rstack);
      insert_internal_code(NULL, top_region, decl->declarator->text.to_chars(), 
			   hdr, impl, DECLARE_GLOBAL);
      continue;

    } else if (decl->is(IR_TYPE_DECLARATION)) {
      // **************************************************************************
      // a new type has been declared
      // **************************************************************************
      string hdr, impl, info_init;
      pIIR_Type type = pIIR_TypeDeclaration(decl)->type;
      emit_hdr(type, hdr, rstack, 0);
      emit_impl(type, impl, rstack, 0);
      // add internal code object to top region
      pIIR_DeclarativeRegion top_region = TopDeclarativeRegion(rstack);
      insert_internal_code(NULL, top_region, decl->declarator->text.to_chars(), 
			   hdr, impl, DECLARE_GLOBAL);
      // emit initial value
      IR_StaticLevel slevel = emit_info_init(type, info_init, rstack, 0);
      if (slevel == IR_LOCALLY_STATIC) {
	// If the info instance is constant then declare it globally
	insert_internal_object_declaration(NULL, top_region, qid(type->declaration, "_", INFO) + "_INFO", 
					   qid(type->declaration, "_", INFO) + "*" , info_init, DECLARE_GLOBAL);
	continue;
	
      } else {
	codegen_error.error("%:error: sorry, only static declarations are supported yet", decl);
	exit(1);
      }

    } else
      // otherwise, skip item
      continue;

    // several items of the same type will be declared in a single
    // declaration statement
    if (type_str != old_type_str) {
      if (old_type_str != "")
	str += ";\n";
      str += spaces(l) + type_str + " " + obj_str;
      old_type_str = type_str;
    } else 
      str += "," + obj_str;
  }

  if (type_str != "")
    str += ";\n";
}


// create code to initialize VHDL objects
void
emit_decls_init (pIIR_DeclarationList decl_list, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration type = NULL;
  string type_str = "";
  string init_str = "";
  string info_str = "";
  bool complex_info = false;

  //***********************************************************************
  // Analyze each item of the declaration list
  //***********************************************************************
  for (pIIR_DeclarationList dl = decl_list; dl; dl = dl->rest) {
    pIIR_Declaration decl = (pIIR_Declaration)dl->first;

    //***********************************************************************
    // first, check whether a info object has to be created at
    // runtime. This is required e.g. for objects of an implicit
    // created array subtype.
    // (e.g.: "variable var : bit_vector(0 to 3)")
    //***********************************************************************
    if (decl->is(IR_SIGNAL_DECLARATION) ||
	decl->is(IR_VARIABLE_DECLARATION) ||
	decl->is(IR_CONSTANT_DECLARATION)) {
      // currently, only signals, variables and constants are considered.
      // Try to find the corresponding type declaration.
      pIIR_ObjectDeclaration obj = (pIIR_ObjectDeclaration)decl;
      type = obj->subtype->declaration;
      complex_info = false;
      if (is_implicit_array_subtype(pIIR_ObjectDeclaration(decl)->subtype)) {
	// if the type is an array subtype then create code to
	// generate an array_info instance at runtime
	type = ((pIIR_ArraySubtype)(obj->subtype))->immediate_base->declaration;
	info_str = create_array_info_obj((pIIR_ArraySubtype)(obj->subtype), rstack, true);
	complex_info = true; // memorize that this info object will be
	// created at runtime
      } else
	// otherwise, simply plot the name of the array info instance
	info_str = qid(type,"_",INFO) + "_INFO";
      
    } else if (decl->is(V2CC_INTERNAL_OBJECT_DECLARATION)) {

    } else if (decl->is(IR_TYPE_DECLARATION)) {

    } else
      // all other kind of objects are not considered
      continue;


    //***********************************************************************
    // Now, generate code to initialize objects depending on the object type
    //***********************************************************************
    if (decl->is(IR_SIGNAL_DECLARATION)) { 
      // **************************************************************************
      // signal declaration
      // **************************************************************************
      pIIR_SignalDeclaration sig = (pIIR_SignalDeclaration)dl->first;
      str += spaces(l) + qid(sig) + string("= new sig_info<") + qid(type,"_",TYPE) + ">\n";
      str += "      (iname.set(\":" + nid(sig,BARE) + "\")," + info_str + ",vREGISTER,NULL);\n";
      
      // generate code which initializes the object if an initial
      // value was given in the object declaration
      if (sig->initial_value) 
	{
	  str += "      " + qid(sig) + "->init(" ;
	  emit_expr (sig->initial_value, str, rstack, DEFAULT);
	  str += ");\n";
	}

    } else if (decl->is(IR_CONSTANT_DECLARATION) ||
	       decl->is(IR_VARIABLE_DECLARATION)) {
      // **************************************************************************
      // constant or variable declaration
      // **************************************************************************
      if (complex_info || 
	  !is_scalar_type(pIIR_ObjectDeclaration(decl)->subtype)) //
	// if the corresponding info object is created at runtime or
	// the object is of a composite type then assign this info
	// instance to the object
	str += spaces(l) + qid(decl) + ".init(" + info_str + ");\n";

      // skip if there is no initial value
      if (pIIR_ObjectDeclaration(decl)->initial_value != NULL) {
	pIIR_DeclarativeRegion dreg = rstack.back();
	id_type id;
	// processes or subprograms access objects declared within the
	// entity/architecture via pointers
	if (dreg->is(IR_CONCURRENT_STATEMENT) ||
	    dreg->is(IR_SUBPROGRAM_DECLARATION))
	  id = id_type(READER, DEREF | ARCHREF);
	else
	  id = id_type(READER, DEREF);

	str += spaces(l) + qid(decl) + "=" ;
	emit_expr(((pIIR_ObjectDeclaration)decl)->initial_value, str, rstack, id);
	str += ";\n";

      } else if (decl->is(IR_VARIABLE_DECLARATION)) {
	// Next, handle variable declarations without explicit initial
	// value
	pIIR_DeclarativeRegion dreg = rstack.back();
	id_type id;
	// processes or subprograms access objects declared within the
	// entity/architecture via pointers
	if (dreg->is(IR_CONCURRENT_STATEMENT) ||
	    dreg->is(IR_SUBPROGRAM_DECLARATION))
	  id = id_type(READER, DEREF | ARCHREF);
	else
	  id = id_type(READER, DEREF);

	if (is_scalar_type(pIIR_VariableDeclaration(decl)->subtype))
	  str += spaces(l) + qid(decl) + "=" + info_str + "->left();\n";
      }

    } else if (decl->is(V2CC_INTERNAL_OBJECT_DECLARATION)) {
      // ******************************************************************
      // initialize internal variables
      // ******************************************************************
      // Note that the process code must be emitted first because some 
      // internal variables are not created before code emission!       
      pV2CC_InternalObjectDeclaration obj = pV2CC_InternalObjectDeclaration(decl);
      // Skip initialization if no inital value is given or the item
      // should be declared and initialized globally
      if ((obj->cpp_initial_string == "" && obj->initial_value == NULL) ||
	  ((obj->flags & DECLARE_GLOBAL) && 
	   (obj->flags & DECLARE_AND_INIT))) continue;
      init_str = get_internal_object_initial_string(obj, rstack);
      str += spaces(l) + obj->declarator->text.to_chars() + init_str + ";\n";

    } else if (decl->is(IR_TYPE_DECLARATION)) {
      // ******************************************************************
      // initialize type info structures
      // ******************************************************************
      

    }

  }
}


// generate code which creates port signals
void
emit_sig_decls (pIIR_InterfaceList interface_list, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration oldtype = NULL, type = NULL;

  for (pIIR_InterfaceList dl = interface_list; dl; dl = dl->rest)
    if (dl->first->is(IR_SIGNAL_INTERFACE_DECLARATION)) {
      pIIR_SignalInterfaceDeclaration sig = (pIIR_SignalInterfaceDeclaration)dl->first;
      type = get_declaration(sig->subtype);
      if (type != oldtype) {
	if (oldtype)
	  str += ";\n";
	str += spaces(l) + "sig_info<" + qid(type,"_",TYPE) + "> ";
	str += "*" + qid(sig);
	oldtype = type;
      } else 
	str += ",*" + qid(sig);
    }
  
  if (oldtype)
    str += ";\n";
}


/* Emit the definition of type T. If it is a subtype, try to express
   it in terms of its base type. */
void
m_emit_def (pIIR_Subtype t, string &str, RegionStack &rstack)
{
  if (t->immediate_base->declaration)
    emit_id (t->immediate_base->declaration, str);
  else
    emit_def (t->immediate_base, str, rstack);
}

void
m_emit_def (pIIR_IntegerType t, string &str, RegionStack &rstack)
{
  str += "int";
}

void
m_emit_def (pIIR_PhysicalType t, string &str, RegionStack &rstack)
{
  str += "long long int";
}

void
m_emit_def (pIIR_FloatingType t, string &str, RegionStack &rstack)
{
  str += "double";
}

void
m_emit_def (pIIR_EnumerationType t, string &str, RegionStack &rstack)
{
  str += "enum { ";
  for(pIIR_EnumerationLiteralList el = t->enumeration_literals; el; el = el->rest)
    {
      emit_id (el->first, str);
      if (el->rest)
	str += ", ";
    }
  str += " }";
}

void
m_emit_def (pIIR_ArrayType t, string &str, RegionStack &rstack)
{
  emit (t->element_type, str, rstack, 0);
  for(pIIR_TypeList it = t->index_types; it; it = it->rest) 
    {
      str += "[";
      emit (it->first, str, rstack, 0);
      str += "]";
    }
}

void
m_emit_def (pIIR_RecordType t, string &str, RegionStack &rstack)
{
  str += "struct {\n";
  for (pIIR_ElementDeclarationList elts = t->element_declarations; elts;
       elts = elts->rest)
    {
      pIIR_ElementDeclaration re = elts->first;
      str += "  ";
      emit (re->subtype, str, rstack, 0);
      str += " ";
      emit_id (re->declarator, str);
      str += ";\n";
    }
  str += "}";
}

void
m_emit_def (pIIR_AccessType t, string &str, RegionStack &rstack)
{
  emit (t->designated_type, str, rstack, 0);
  str += "*";
}

void
m_emit_def (pIIR_FileType t, string &str, RegionStack &rstack)
{
  str += "file<";
  emit (t->type_mark, str, rstack, 0);
  str += ">";
}

void
m_emit_def (pIIR_Type t, string &str, RegionStack &rstack)
{
  str += "/* emit_def " + string(t->kind_name()) + " */";
}

void
emit_comp_interface (pIIR_Component c, string &str, RegionStack &rstack)
{
  pIIR_InterfaceList gens = get_generics (c);
  pIIR_InterfaceList ports = get_ports (c);
  pIIR_InterfaceList i;

  for (i = gens; i; i = i->rest) 
    {
      emit (i->first, str, rstack, 0);
      if (i->rest || ports)
	str += ", ";
    }
  for (i = ports; i; i = i->rest) 
    {
      emit (i->first, str, rstack, 0);
      if (i->rest)
	str += ", ";
    }
}



char *
constant_fold_string_expr (pIIR_Expression e)
{
  if (e->is(IR_ARRAY_LITERAL_EXPRESSION)) 
    {
      char *val = pIIR_ArrayLiteralExpression(e)->value->text.to_chars();
      val = vaul_xstrdup (val+1);
      val[strlen(val)-1] = '\0';
      return val;
    }

  if (e->is(IR_FUNCTION_CALL))
    {
      pIIR_FunctionCall fc = pIIR_FunctionCall(e);
      if (fc->function->declarator
	  && vaul_name_eq (fc->function->declarator, "\"&\""))
	{
	  char *val = vaul_aprintf ("");
	  for (pIIR_AssociationList al = fc->parameter_association_list;
	       al; al = al->rest)
	    {
	      char *val1 = constant_fold_string_expr (al->first->actual);
	      if (val1 == NULL) 
		{
		  free (val);
		  return NULL;
		}
	      char *val2 = vaul_aprintf("%s%s", val, val1);
	      free (val);
	      free (val1);
	      val = val2;
	    }
	  return val;
	}
      else
	return NULL;
    }

  return NULL;
}

/* Emit a declaration. This can probably be done with less code...
 */

void
emit_prototype (pIIR_SubprogramDeclaration s, string &str, RegionStack &rstack)
{
  if (s->is(IR_FUNCTION_DECLARATION))
    emit (pIIR_FunctionDeclaration(s)->return_type, str, rstack, 0);
  else
    str += "void";
  str += " ";
  emit_id (s, str);
  str += " (";
  
  for(pIIR_InterfaceList il = s->interface_declarations; il; il = il->rest)
    {
      emit (il->first, str, rstack, 0);
      if (il->rest)
	str += ", ";
    }
  str += ")";
}

void 
m_emit (pIIR_Declaration d, string &str, RegionStack &rstack, int l)
{
  emit_decl (d, str, rstack, l);

  for (pIIR_AttributeValueList avl = d->attributes; avl; avl = avl->rest) 
    {
      pIIR_AttributeValue av = avl->first;
      str += spaces(l) + string("sattribute ");
      emit_id (d, str);
      str += "'";
      emit_id (av->attribute, str);
      str += " = ";
      char *s = constant_fold_string_expr (av->value);
      if (s)
	{
	  str += "\"" + string(s) + "\"";
	  free (s);
	}
      else
	emit (av->value, str, rstack, DEFAULT);
      str += ";\n";
    }
}

/*
 * emit IIR Declarations
 */

void
m_emit_decl (pIIR_Declaration d, string &str, RegionStack &rstack, int l)
{
  // catch-all for ignored declarations.
  str += spaces(l) + string("/* ") + d->kind_name();
  emit_id (d, str);
  str += " */\n";
}

void
m_emit_decl (pIIR_SubprogramDeclaration s, string &str, RegionStack &rstack, int l)
{
  emit_prototype (s, str, rstack);
  if (s->subprogram_body == NULL)
    str += ";\n";
  else
    {
      str += "\n{\n";
      emit_decls (s->declarations, str, rstack, 2);
      if (s->declarations)
	str += "\n";
      emit_impl (s->subprogram_body, str, rstack, 2);
      str += "}\n\n";
    }
}

void
m_emit_decl (pIIR_VariableDeclaration v, string &str, RegionStack &rstack, int l)
{
  str += spaces(l);
  emit (v->subtype, str, rstack, 0);
  str += " ";
  emit_id (v, str);
  if (v->initial_value) 				// should be done only in constr.
    {
      str += " = ";
      emit (v->initial_value, str, rstack, DEFAULT);
    }
  str += ";\n";
}

void
m_emit_decl (pIIR_ConstantDeclaration c, string &str, RegionStack &rstack, int l)
{
  str += spaces(l);
  if (!c->initial_value)
    str += "extern ";
  str += "const ";
  emit (c->subtype, str, rstack, 0);
  str += " ";
  emit_id (c, str);
  if (c->initial_value) 
    {
      str += " = ";
      emit (c->initial_value, str, rstack, 0);
    }
  str += ";\n";
}

void
m_emit_decl (pIIR_FileDeclaration f, string &str, RegionStack &rstack, int l)
{
  // A file. The file name is in the INITIAL_VALUE slot.

  str += spaces(l);
  emit (f->subtype, str, rstack, 0);
  str += " ";
  emit_id(f, str);
  if (f->file_logical_name)
    {
      str += " = open (";
      emit (f->file_logical_name, str, rstack, 0);
      if (f->file_open_expression) 
	{
	  str += ", ";
	  emit (f->file_open_expression, str, rstack, 0);
	}
      str += ")";
    }
  str += ";\n";
}

void
m_emit_decl (pIIR_SignalDeclaration s, string &str, RegionStack &rstack, int l)
{
  str += "sig_info<" + qid(s->subtype->declaration,"_",TYPE) 
    + "> *" + qid(s) + ";\n";
}


void
m_emit_decl (pIIR_TypeDeclaration t, string &str, RegionStack &rstack, int l)
{
  // A type. We give types names with typedef and then later refer
  // to them via these names. Anonymous types are not defined.
  str += "typedef " + qid (t->type->declaration) + " " + qid(t) + ";\n";
}

void
m_emit_decl (pIIR_PackageDeclaration p, string &str, RegionStack &rstack, int l)
{
  // A package.  Just emit all declarations in it.
  rstack.push(p);

  // First, explore and check package. Return if checking failed.
  if (explore_and_check(p, rstack)) {
    rstack.pop();
    return;
  } 

  str += "\n/* package " + string(p->declarator->text.to_chars()) +
    "\n*/\n\n"; 
  emit_decls (p->declarations, str, rstack, 0);
  rstack.pop();
}

void
m_emit_decl (pIIR_PackageBodyDeclaration pb, string &str, RegionStack &rstack, int l)
{
  // A package body.  Just emit all declarations in it.
  rstack.push(pb);

  // First, explore and check package body. Return if checking failed.
  if (explore_and_check(pb, rstack)) {
    rstack.pop();
    return;
  } 

  str += "\n/* package body " + string(qid(pb)) + "*/\n";
  emit_decls (pb->declarations, str, rstack, 0);
  rstack.pop();
}

void
m_emit_decl (pIIR_EntityDeclaration e, string &str, RegionStack &rstack, int l)
{
  rstack.push(e);

  // First, explore and check entity. Return if checking failed.
  if (explore_and_check(e, rstack)) {
    rstack.pop();
    return;
  } 

  emit_hdr(e, str, rstack, l);
  emit_impl(e,str, rstack, l);
  str += "\n/* end of " + qid(e,"_") + " Entity */\n";
  rstack.pop();
}

void				// emit Arch decl=hdr+impl+procs
m_emit_decl (pIIR_ArchitectureDeclaration a, string &str, RegionStack &rstack, int l)
{
  rstack.push(a);
  pIIR_EntityDeclaration e = a->entity;

  // First, explore and check architecture. Return if checking failed.
  if (explore_and_check(a, rstack)) {
    rstack.pop();
    return;
  } 

  string impl_str, hdr_str;
  emit_impl (a, impl_str, rstack, l);
  emit_hdr (a, hdr_str, rstack, l);

  // First, emit global prototypes
  for (pIIR_DeclarationList decls = ActiveDeclarativeRegion(rstack)->declarations;
       decls; decls = decls->rest) {
    if (decls->first->is(V2CC_INTERNAL_CODE)) {
      pV2CC_InternalCode ic = pV2CC_InternalCode(decls->first);
      if (ic->flags & DECLARE_GLOBAL)
	str += ic->cpp_header_string;
    }
  }

  // declare all internal global objects which are global before the
  // architecture classes are actually declared
  for (pIIR_DeclarationList decls = ActiveDeclarativeRegion(rstack)->declarations;
       decls; decls = decls->rest) {

    if (decls->first->is(V2CC_INTERNAL_CODE)) {
      // Internal code object
      pV2CC_InternalCode ic = pV2CC_InternalCode(decls->first);
      if (ic->flags & DECLARE_GLOBAL)
	str += ic->cpp_impl_string;

    } else if (decls->first->is(V2CC_INTERNAL_OBJECT_DECLARATION)) {
      // Internal object declaration
      pV2CC_InternalObjectDeclaration obj = pV2CC_InternalObjectDeclaration(decls->first);
      if (obj->flags & DECLARE_GLOBAL) {
	string type_str = get_internal_object_type_string(obj, rstack);
	if (!obj->flags & DECLARE_AND_INIT)
	  str += type_str + " " + obj->declarator->text.to_chars() + ";\n";
	else {
	  string initial_str = get_internal_object_initial_string(obj, rstack);
	  str += type_str + " " + obj->declarator->text.to_chars() + initial_str + ";\n";
	}
      }
    }

  }

  str += hdr_str + impl_str;
  str += "\n/* end of " + qid(a) + " Architecture */\n";
  rstack.pop();
}

void
m_emit_decl (pIIR_ConfigurationDeclaration c, string &str, RegionStack &rstack, int l)
{
  str += "\n/* configuration " +  qid(c) + " */\n";
}

void
m_emit_decl (pIIR_ComponentDeclaration c, string &str, RegionStack &rstack, int l)
{
  // A component. Emit its interface.

  str += "component " + qid(c) + " (\n";
  str += ");\n";
}


/* Refer to a type. If it has a name, use that. Otherwise be more
   verbose.
*/
void
m_emit (pIIR_Type t, string &str, RegionStack &rstack, int l = 0)
{
  if (t->declaration)
    emit_id (t->declaration, str);
  else if (t->is(IR_SUBTYPE)) 
    {
      emit (pIIR_Subtype(t)->immediate_base,  str, rstack, l);
      if (t->is(IR_SCALAR_SUBTYPE))
	{
	  pIIR_ScalarSubtype s = pIIR_ScalarSubtype (t);
	  if (s->range)
	    {
	      str += " ";
	      emit (s->range, str, rstack, l);
	    }
	}
      else if (t->is(IR_ARRAY_SUBTYPE))
	{
	  pIIR_ArraySubtype a = pIIR_ArraySubtype (t);
	  if (a->constraint)
	    {
	      str += " ";
	      emit (a->constraint, str, rstack, l);
	    }
	}
    }
  else
    emit_def (t, str, rstack);
}

void
m_emit (pIIR_TypeList c, string &str, RegionStack &rstack, int l)
{
  while (c == NULL)
    {
      str += "[";
      emit (c->first, str, rstack, l);
      str += "]";
      c = c->rest;
    }
}

// Emit an interface element for use in a parameter list
void
m_emit (pIIR_InterfaceDeclaration i, string &str, RegionStack &rstack, int l)
{
  VAUL_ObjectClass obj_class = get_class (i);

  if(obj_class == VAUL_ObjClass_Signal) 
    {
      str += "sig_info<" + qid(i->subtype->declaration,"_",TYPE) + ">" ;
    } 
  else
    emit (i->subtype, str, rstack, l);

  if (obj_class == VAUL_ObjClass_Variable)
    str += "&";

  if (i->declarator) 
    {
      str += " ";
      emit_id (i, str);
    }
  if (i->initial_value) 
    {
      str += " = ";
      emit (i->initial_value, str, rstack, 0);
    }
}


/* Emit a physical unit. This done by expressing it in terms of
   its base unit.
*/
void
emit (pIIR_PhysicalUnit u, string &str, RegionStack &rstack, int l)
{
  str += qid(u -> type -> declaration, "_", INFO);
  str += "::scale[";
  strstream lstr;
  lstr << u -> unit_pos << (char)'\0';
  str += string(lstr.str()) + "]"; 		
}


// Generated code to concat two arrays
string
concat_operator_call_string(pIIR_FunctionCall fc, string &arg1, string &arg2, RegionStack &rstack)
{
 pIIR_Expression arg_exp1 = fc->parameter_association_list->first->actual;
 pIIR_Expression arg_exp2 = fc->parameter_association_list->rest->first->actual;

 pIIR_ArrayType dest_type = (pIIR_ArrayType)get_base_type(fc->subtype);
 pIIR_Type arg_type1 = get_base_type(arg_exp1->subtype);
 pIIR_Type arg_type2 = get_base_type(arg_exp2->subtype);
 
 // Determine range of the destination array
 vector<RangeDescriptor> range_desc = 
   get_discrete_range (((pIIR_ScalarSubtype)(dest_type->index_types->first))->range, rstack, IR_NOT_STATIC);
 string left_str, right_str, dir_str;
 // Convert to integer strings
 range_desc[0].rangedes_to_int_string(left_str, dir_str, right_str, rstack);

 // First, convert the arguments into appropriate array objects if
 // required
 if (dest_type != arg_type1) {
   // First argument is an element of the resulting array.  Create
   // index range for an array consisting of one single element
   string range = left_str + "," + dir_str + "," + left_str;
   // Create call to array consructor. The second constructor argument
   // is a the element value. The first argument is an array_info
   // instance.
   arg1 = qid(dest_type->declaration,"_",TYPE) + "(new array_info(" + 
     qid(dest_type->declaration,"_",INFO) + "_INFO," + range + ",0)," + arg1 + ")";
 }
 if (dest_type != arg_type2) {
   // Second argument is an element of the resulting array.  Create
   // index range for an array consisting of one single element
   string range = left_str + "," + dir_str + "," + left_str;
   // Create call to array consructor. The second constructor argument
   // is a the element value. The first argument is an array_info
   // instance.
   arg2= qid(dest_type->declaration,"_",TYPE) + "(new array_info(" + 
     qid(dest_type->declaration,"_",INFO) + "_INFO," + range + ",0)," + arg2 + ")";
 }
 
 string str = "concat<" + qid(dest_type->declaration, "_", TYPE) + "," + 
   qid(dest_type->element_type->declaration, "_", TYPE)+ ">(" + arg1 + "," + arg2 + ")";

 return str;
}


bool
m_emit_expr (pIIR_AttrSigFunc af, string &str, RegionStack &rstack, id_type t)
{
  string scalar_function_name, composite_function_name;

  // First, determine attribute kind
  if (af->is(IR_ATTR_EVENT)) {
    // Attribute: EVENT
    scalar_function_name = "attr_scalar_EVENT";
    composite_function_name = "attr_composite_EVENT";

  } else if (af->is(IR_ATTR_ACTIVE)) {
    // Attribute: ACTIVE
    scalar_function_name = "attr_scalar_ACTIVE";
    composite_function_name = "attr_composite_ACTIVE";
    
  } else
    // else bail out!
    codegen_error.error("%:error: sorry, this attribute is currently not supported", af);


  // Next, create attribute code
  if (af->signal->is(IR_SIMPLE_REFERENCE)) {
    // Prefix is a scalar signal
    str += scalar_function_name + "(";
    emit_expr(af->signal, str, rstack, id_type(SIGNAL, DEFAULT));
    str += ")";

  } else {
    // Prefix is a composite signal
    str += composite_function_name + "(";
    // Get object declaration of the prefix signal
    pIIR_ObjectDeclaration object_decl = get_object_declaration(af->signal);
    str += qid(object_decl, "_", SIGNAL);

    // Append acl
    list<string> acl_list;
    get_acl(af->signal, acl_list, rstack, IR_NOT_STATIC, id_type(SIGNAL, ARCHREF), true);
    // Create internal acl
    string internal_acl_name = create_internal_acl(acl_list, rstack, true);
    // Add acl sequence
    str += ",&(" + sprint_acl(acl_list, internal_acl_name + "->clear()") + "))";
  }

  return false;
}


bool
m_emit_expr (pIIR_FunctionCall fc, string &str, RegionStack &rstack, id_type t)
{
  // If the function call has been folded then return the optimized
  // result instead of actually emitting the function call
  if (valid_folded_value(fc)) {
    pIIR_Type base_type = get_base_type(fc->subtype);
    if (base_type->is(IR_ENUMERATION_TYPE))
      str += "enumeration(" + i_to_string(folded_value(fc).long_value) + ")";
    else if (base_type->is(IR_FLOATING_TYPE))
      str += d_to_string(folded_value(fc).double_value);
    else
      str += i_to_string(folded_value(fc).long_value);

    return true; // Ok, were are done!
  }

  // count the arguments
  int n_args = 0;
  for (pIIR_AssociationList al = fc->parameter_association_list; al; al = al->rest)
    n_args++;

  string arg1, arg2;
  bool int_literal;
  
  switch (get_operator_type(fc->function)) { // Analyze function call type
  case STD_OP: // The function call is a standard operator call
    // emit first operand
    int_literal = emit_expr (fc->parameter_association_list->first->actual, arg1, 
			     rstack, id_type(READER, DEREF)); // First argument
    //if (int_literal) 
    //arg1 = cast(fc->parameter_association_list->first->actual, arg1);

    // emit second operand
    if (n_args == 2) { // Binary operator call
      int_literal = emit_expr (fc->parameter_association_list->rest->first->actual, arg2, 
			       rstack, id_type(READER, DEREF)); // Second argument
      //if (int_literal) 
      //arg2 = cast(fc->parameter_association_list->rest->first->actual, arg2);

      if (binary_operator_macro[fc->function->declarator->text.to_chars()] == "concat")
	// Add concat operator call. The concat operator is handled
	// differently from the other operators
	str += concat_operator_call_string(fc, arg1, arg2, rstack);
      else
	// Add operator call to code
	str += operator_call_string(fc->function, arg1, arg2); 

    } else // Unary operator call
      str += operator_call_string(fc->function, arg1); // Add operator call

    break;

  case USER_OP: // The function is an user defined operator call
  case BASIC_OP: // The function is a operator defined in an IEEE
    // library. Currently this is handled like a user defined operator
    // call
  case NO_OP: // A ordinary function call (no operator)
    str += get_access_string(fc->function);
    str += " (";
    emit_associations (str, rstack, fc->parameter_association_list,
		       fc->function->interface_declarations);
    str += ")";
  }

  return false;
}

bool
m_emit_expr (pIIR_SimpleReference sor, string &str, RegionStack &rstack, id_type t)
{
  //emit_noqual_id (sor->object, str, t);
  str += qid (sor->object, "_", t);

  return false;
}


bool
m_emit_expr (pIIR_AbstractLiteralExpression ale, string &str, RegionStack &rstack, id_type t)
{
  if (ale->is(IR_PHYSICAL_LITERAL)) {
    str += "(";
    emit_lit (ale->value, str);
    str += "*";
    emit (pIIR_PhysicalLiteral(ale)->unit, str, rstack, 0);
    str += ")";

  } else if (ale->value->is(IR_INTEGER_LITERAL) ||
	     ale->value->is(IR_FLOATING_POINT_LITERAL)) {
    string value_str;
    emit_lit (ale->value, value_str);
    //str += cast(ale, value_str);
    str += value_str;

  } else
    emit_lit (ale->value, str);

  return true;
}

void
emit_lit (pIIR_Literal l, string &str)
{
  if (l == NULL)
    str += "1";
  else if (l->is(IR_TEXT_LITERAL))
    str += pIIR_TextLiteral(l)->text.to_chars();
  else if (l->is(IR_INTEGER_LITERAL))
    str += pIIR_IntegerLiteral(l)->text.to_chars();
  else if (l->is(IR_FLOATING_POINT_LITERAL))
    str += pIIR_FloatingPointLiteral(l)->text.to_chars();
  else
    str += "<" + string(l->kind_name()) + ">";
}

bool
m_emit_expr (pIIR_TypeConversion tc, string &str, RegionStack &rstack, id_type t)
{
  // A type cast

  str += "((";
  emit (tc->subtype, str, rstack, 0);
  str += ")(";
  emit (tc->expression, str, rstack, 0);
  str += "))";

  return false;
}

bool
m_emit_expr (pIIR_QualifiedExpression qe, string &str, RegionStack &rstack, id_type t)
{
  emit_expr (qe->expression, str, rstack, t);

  return false;
}

bool
m_emit_expr (pIIR_EnumLiteralReference elr, string &str, RegionStack &rstack, id_type t)
{
  emit_id (elr->value, str);

  return true;
}

bool
m_emit_expr (pIIR_ArrayReference aor, string &str, RegionStack &rstack, id_type t)
{
  emit_expr (aor->array, str, rstack, t);
  for (pIIR_ExpressionList il = aor->indices; il; il = il->rest)
    {
      str += "[";
      string index;
      bool simple = emit_expr (il->first, index, rstack, t);
      str += index + "]";
    }

  return false;
}


bool
m_emit_expr (pIIR_ArrayLiteralExpression ale, string &str, RegionStack &rstack, id_type t)
{
  // First, get range of array expression. The range (bounds and
  // directions) is stored into an vector of strings where the first
  // strings is the left bound, the second string denotes the range
  // direction and the third string is the right bound.
  vector<RangeDescriptor> range_desc =
    get_discrete_range ((pIIR_ExplicitRange)((pIIR_ScalarSubtype)((pIIR_ArraySubtype)ale->subtype)->constraint->first)->range,
	  rstack, IR_NOT_STATIC);
  string left_str, dir_str, right_str;
  range_desc[0].rangedes_to_int_string(left_str, dir_str, right_str, rstack);

  // Concat all strings values separated by ","
  string range = left_str + "," + dir_str + "," + right_str;

  // Get type declaration of the array literal 
  pIIR_TypeDeclaration type = get_declaration(ale->subtype);

  // Bail out if the array element is not an enumeration type
  if (!((pIIR_ArrayType)type->type)->element_type->is(IR_ENUMERATION_TYPE))
    codegen_error.error("%:error: array element type %n is not an enumeration type",
			ale, ((pIIR_ArrayType)type->type)->element_type);

  pIIR_EnumerationType etype = (pIIR_EnumerationType)((pIIR_ArrayType)type->type)->element_type;
  pIIR_EnumerationLiteralList enum_items = etype->enumeration_literals; // Get list of enumeration items
  if (!ale->value->is(IR_TEXT_LITERAL))
    internal_error(true);

  // Create name for an internal array constant to store the literal
  // values. This variable will be used to initialize the array instance
  string internal_var_name = get_new_internal_var_prefix() + "_lit";

  // Create call to array constructor. The second constructor argument
  // is a pointer to an array which stores the literal values. The
  // first argument is an array_info instance.
  str += qid(type,"_",TYPE) + "(new array_info(" + qid(type,"_",INFO) + "_INFO," + range + ",0)," +
    "(" + qid(etype->declaration,"_",TYPE) + "*)" + internal_var_name + ")";

  // Now, convert each element of the literal into an corresponding
  // number and declare an internal constant which takes the literal
  // values. Note, this variable is declared globally.
  char *literal = pIIR_TextLiteral(ale->value)->text.to_chars();
  int length = strlen(literal) - 2; // don't count leading and trailing \" 
  string init_str = "[]={";
  char *sep = "";
  for (int i = 1; i < length + 1; i++) {
    string literal_str = string("'") + literal[i] + string("'");
    int enum_pos = literal_to_pos(enum_items, literal_str); // Get pos number of literal item
    if (enum_pos < 0) // Bail out if number is not valid
      codegen_error.error("%:error: character %s does not belong to enumertion type %n",
			  ale, literal_str.c_str(), etype);

    init_str += sep + i_to_string(enum_pos);
    sep = ",";
  }
  init_str += "}";
  
  // Declare the internal variable as a global constant if not an
  // object with the same name is already declared
  pIIR_DeclarativeRegion top_region = TopDeclarativeRegion(rstack);
  if (lookup_internal_object_declaration(top_region, internal_var_name) == NULL)
    insert_internal_object_declaration(NULL, top_region, internal_var_name, 
				       "const char", init_str, DECLARE_GLOBAL | DECLARE_AND_INIT);

  return false;
}


bool
m_emit_expr (pIIR_RecordReference ror, string &str, RegionStack &rstack, id_type t)
{
  emit_expr (ror->record, str, rstack, t);
  str += ".";
  emit_id (ror->element->declarator, str);

  return false;
}

bool
m_emit_expr (pIIR_RecordAggregate ra, string &str, RegionStack &rstack, id_type t)
{
  str += "{ ";
  for (pIIR_ElementAssociationList al = ra->element_association_list;
       al; al = al->rest)
    {
      emit_id (al->first->element->declarator, str);
      str += ": ";
      emit_expr (al->first->value, str, rstack, t);
      if (al->rest)
	str += ", ";
    }
  str += " }";

  return false;
}


bool
m_emit_expr (pIIR_ArrayAggregate aa, string &str, RegionStack &rstack, id_type t)
{
  // Subtype of aggregate expression. At least the range direction will
  // be used...
  string dest_type_str, dest_type_info_str;
  pIIR_Type dest_type = aa->subtype;
  int dim_number = 1;
  if(dest_type->is(VAUL_SUBARRAY_TYPE)) {
    // If the aggreate is an multi-dimensional then a node
    // VAUL_SubarrayType is used to describe the (sub)type of the
    // sub-aggregate
    dest_type = pVAUL_SubarrayType(aa->subtype)->complete_type->declaration->type;
    pIIR_ArrayType at = pVAUL_SubarrayType(aa->subtype)->complete_type;
    // Determine dimension of main array to which the aggregate
    // belongs
    int dim_counter = 0;
    for (pIIR_TypeList tl = at->index_types; tl; tl = tl->rest)
      dim_counter++;
    // Next, determine the index associated with the array
    // aggregate. Note that we can only count how many dimensions are
    // left from where the aggregate starts.
    int current_dim_counter = 0;
    for (pIIR_TypeList tl = pVAUL_SubarrayType(aa->subtype)->index_types; tl; tl = tl->rest)
      current_dim_counter++;
    // Now, determine the index number the aggregate belongs to
    dim_number = dim_counter - current_dim_counter + 1;
    // Setup dest_type_str and dest_type_info_str
    dest_type_str = qid(get_declaration(dest_type),"_",TYPE);
    dest_type_info_str = qid(get_declaration(dest_type),"_",INFO) + "_INFO";
    for (int i = 1; i <= dim_counter - current_dim_counter; i++) {
      dest_type_str += "::E_type";
      dest_type_info_str = "parray_info(" + dest_type_info_str + "->element_type)";
    }

  } else if (dest_type->is(IR_ARRAY_TYPE)) {
    dest_type = aa->subtype->declaration->type;
    dest_type_str = qid(get_declaration(dest_type),"_",TYPE);
    dest_type_info_str = qid(get_declaration(dest_type),"_",INFO) + "_INFO";

  } else {
    dest_type_str = qid(get_declaration(dest_type),"_",TYPE);
    dest_type_info_str = qid(get_declaration(dest_type),"_",INFO) + "_INFO";
  }
  // Get range type of aggregate and determine left bound, right
  // bound, direction and length of aggregat. Note that if the
  // aggregate subtype cannot be determined from the context then
  // dest_range will point to the corresponding index range of the
  // unconstrained array associated with the aggregate.
  vector<RangeDescriptor> range_desc = get_discrete_range(dest_type, rstack, IR_NOT_STATIC);
  // Extract bounds from aggregate range type. Perhaps these
  // bounds will be used to determine the range bounds of the aggregat.
  string range;
  string left_str, dir_str, right_str;
  range_desc[dim_number-1].rangedes_to_int_string(left_str, dir_str, right_str, rstack);
  string dest_direction_str = 
    (dest_direction(aa) == IR_DIRECTION_UP? "to" : "downto"); // Direction of aggregate

  int association_count = 0; // Number of associations
  int max_len = 0; // Max index range length of a single choice
  pIIR_IndexedAssociation max_len_choice = NULL; // choice with longest range

  // First, take a look at all choices to determine the choice which
  // covers the largest range. However, an others choice will always
  // be selected as max_len_choice. Further, the number of
  // associations are counted.
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest) { 
    if (al->first->is(IR_OTHERS_INDEXED_ASSOCIATION)) { // others choice
      max_len = INT_MAX; // set length value so that this entry is not overwritten
      max_len_choice = al->first;
    } else {
      if (max_len < length(al->first)) {
	max_len = length(al->first);
	max_len_choice = al->first;
      }
    }

    association_count++; // Count number of association 
  }
  
  // *****************************************************************************
  // Now, generate the code. First. check some special cases.
  // *****************************************************************************
  string value_str;
  if (has_others(aa)) { // Does the aggregate include an others choice?
    // If the answer is yes, then the subtype of the aggregate must be
    // determinable from the context. Get value expression of others choice.
    emit_expr(pIIR_OthersIndexedAssociation(max_len_choice)->value, value_str, rstack, t);
    // Determine range of the array aggregate.
    range = left_str + "," + dir_str + "," + right_str;

  } else {
    // Ok, aggregate does not have an others choice. Hence, the array
    // bounds are directly determined from the choices. Generate code
    // for the first array instance. The initial array instance is
    // created from those choice which covers the longest range.
    if (association_count == 1) { 
      // Ok, there is a single choice
      if (max_len_choice->is(IR_SINGLE_INDEXED_ASSOCIATION)) {
	// Array contains only a single element
	pIIR_SingleIndexedAssociation saa = pIIR_SingleIndexedAssociation(max_len_choice);
	string index_str;
	if (named_association(aa)) {
	  // In case of named association use index value of choice
	  emit_expr(saa->index, index_str, rstack, DEFAULT);
	} else
	  // In case of positional association use left bound of
	  // aggregate type
	  index_str = left_str;
	
	range = index_str + "," + dest_direction_str + "," + index_str;

      } else if (max_len_choice->is(IR_RANGE_INDEXED_ASSOCIATION)) {
	// Array contains a single range choice. Note that the range
	// does not have to be locally static in this case.
	pIIR_RangeIndexedAssociation raa = pIIR_RangeIndexedAssociation(max_len_choice);
	vector<RangeDescriptor> range_desc = get_discrete_range(raa->index_range, rstack, IR_NOT_STATIC);
	string left_str, dir_str, right_str;
	range_desc[0].rangedes_to_int_string(left_str, dir_str, right_str, rstack);

	if (dest_direction_str == dir_str)
	  range = left_str + "," + dest_direction_str + "," + right_str;
	else
	  range = right_str + "," + dest_direction_str + "," + left_str;

      } else 
	// This should never happen!
	internal_error(true);

    } else
      // Ok, there are more than one choices. Hence, we use min_index
      // and max_index to setup the array range
      if (dest_direction_str == "to")
	range = i_to_string(min_index(aa)) + "," + dest_direction_str + "," + i_to_string(max_index(aa));
      else
	range = i_to_string(max_index(aa)) + "," + dest_direction_str + "," + i_to_string(min_index(aa));

    // Generate value of choice 
    if (max_len_choice->is(IR_SINGLE_INDEXED_ASSOCIATION)) 
      emit_expr(pIIR_SingleIndexedAssociation(max_len_choice)->value, value_str, rstack, t);
    else if (max_len_choice->is(IR_RANGE_INDEXED_ASSOCIATION))
      emit_expr(pIIR_RangeIndexedAssociation(max_len_choice)->value, value_str, rstack, t);
    else
      internal_error(true);
  }

  // Create call to array constructor. The second constructor argument
  // is the element value. The first argument is an array_info
  // instance.
  str += dest_type_str + "(new array_info(" + dest_type_info_str + "," + range + ",0)," + value_str + ")";
  // If only a single choice is given then were done.
  if (association_count == 1) return false;

  // Next, handle the remaining choices
  for (pIIR_IndexedAssociationList al = aa->indexed_association_list; al; al = al->rest) {
    pIIR_IndexedAssociation ia = al->first;
    if (ia == max_len_choice) continue; // Skip choice if it has been already handled
    // Emit choice value 
    value_str = "";
    emit_expr(ia->value, value_str, rstack, t);
    if (ia->is(IR_SINGLE_INDEXED_ASSOCIATION) && !named_association(aa)) {
      // If choice is a single indexed *positional* association then
      // index number of the element is dertermined by adding the
      // choice number (stored in min_index) to the left bound of the
      // aggregate (stored in left_str).
      string index_str = left_str + (dest_direction_str == "to"? "+" : "-") + 
	i_to_string(min_index(ia));
      str += ".aggregate_set(" + i_to_string(min_index(ia)) + ",to," + 
	i_to_string(max_index(ia)) + "," + value_str + ")";      
    } else
      // Add code to set the corresponding array elements 
      str += ".aggregate_set(" + i_to_string(min_index(ia)) + ",to," + 
	i_to_string(max_index(ia)) + "," + value_str + ")";
  }
  
  return false;
}



bool
m_emit_expr (pIIR_SliceReference sr, string &str, RegionStack &rstack, id_type t)
{
  vector<bool> simple_vec;
  vector<string> str_vec;

  // Get slice range
  vector<RangeDescriptor> range_desc = get_discrete_range(sr->range, rstack, IR_NOT_STATIC);
  string left_str, dir_str, right_str;
  range_desc[0].rangedes_to_int_string(left_str, dir_str, right_str, rstack);

  pIIR_TypeDeclaration type_declaration = get_declaration(sr->array->subtype);
  str += qid(type_declaration, "_", id_type(TYPE | ALIAS, DEFAULT));

  str += "(new array_info(";
  str += qid(type_declaration,"_",INFO) + "_INFO," + left_str + "," + dir_str + "," + right_str + ",0),&";

  emit_expr (sr->array, str, rstack, t);
  str += "[" + left_str + "]";

  str += ")";

  return false;
}

bool
m_emit_expr (pIIR_AccessReference aor, string &str, RegionStack &rstack, id_type t)
{
  str += "*";
  emit_expr (aor->access, str, rstack, t);

  return false;
}

bool
m_emit_expr (pIIR_AttrSignalRef asr, string &str, RegionStack &rstack, id_type t)
{
  emit_builtin_id (asr->kind(), str);
  str += "(";
  emit_expr (asr->signal, str, rstack, t);
  str += ")";

  return false;
}

bool
m_emit_expr (pIIR_AttrTypeFunc atf, string &str, RegionStack &rstack, id_type t)
{
  emit_builtin_id (atf->kind(), str);
  str += "(";
  emit (atf->prefix, str, rstack, 0);
  if (atf->argument) 
    {
      str += ", ";
      emit_expr (atf->argument, str, rstack, t);
    }
  str += ")";

  return false;
}

bool
m_emit_expr (pIIR_ArrayAttr aa, string &str, RegionStack &rstack, id_type t)
{
  emit_builtin_id (aa->kind(), str);
  str += "(";
  emit_expr (aa->array, str, rstack, t);
  strstream nstr;
  nstr << aa->index << (char)'\0';
  str += "(" + string(nstr.str()) + ")";

  return false;
}

bool
m_emit_expr (pIIR_Expression e, string &str, RegionStack &rstack, id_type t)
{
  str += "/* IIR_Expression " + string(e->kind_name()) + " */";

  return false;
} 


// Actually, IIR_EnumerationLiteral is not an expression but we need
// this method for internal use
bool
m_emit_expr (pIIR_EnumerationLiteral e, string &str, RegionStack &rstack, id_type t)
{
  str += qid(e->subtype->declaration, "_", TYPE) + "(" + i_to_string(e->enum_pos) + ")";

  return true;
}


// vector<bool>
// m_emit (pIIR_Range r, vector<string> &strv, RegionStack &rstack, int l)
// {
//   string str;
//   vector<bool> simple_vec;
//   bool simple;

//   if (r->is (IR_EXPLICIT_RANGE)) 
//     {
//       pIIR_ExplicitRange er = pIIR_ExplicitRange(r);
//       simple = emit_expr (er->left, str, rstack, DEFAULT);
//       strv.push_back(str);
//       simple_vec.push_back(simple);

//       strv.push_back(string(er->direction == IR_DIRECTION_UP? "to" : "downto"));
//       simple_vec.push_back(true);

//       simple = emit_expr (er->right, str = "", rstack, DEFAULT);
//       strv.push_back(str);
//       simple_vec.push_back(simple);
//     }
//   else if (r->is(IR_ARRAY_RANGE))
//     {
//       str = string(r->is(IR_ARRAY_ATTR_RANGE)? "range" : "reverse_range") + "(";
//       emit_expr (pIIR_ArrayRange(r)->array, str, rstack, DEFAULT);
//       strv.push_back(str);
//       if (pIIR_ArrayRange(r)->index)
// 	emit_expr (pIIR_ArrayRange(r)->index, str = "", rstack, DEFAULT);
//       else
// 	str = "1";
//       strv.push_back(str);
//     } 
//   else
//     strv.push_back("/* emit range " + string(r->kind_name()) + " */");

//   return simple_vec;
// }


bool
emit_associations (string &str, RegionStack &rstack, pIIR_AssociationList assocs, pIIR_InterfaceList formals)
{
  bool need_comma = false;
  for (pIIR_InterfaceList fl = formals; fl; fl = fl->rest)
    {
      pIIR_InterfaceDeclaration f = fl->first; // select formal parameter
      pIIR_AssociationElement a;
      pIIR_AssociationList al;

      /* Select association element from an association list which
       * corresponds with given formal */
      a = find_matching_actual(assocs, f);

      if (a != NULL && !a->actual->is(IR_OPEN_EXPRESSION)) {
	// An actual is associated with the formal
	if (a->formal_conversion)  {
	  str += "/* converted by ";
	  emit_id (a->formal_conversion, str);
	  str += " */";
	}
	// emit actual
	emit_expr (a->actual, str, rstack, DEFAULT);

      } else if (f->initial_value != NULL) {
	// If the parameter was left open then insert the default
	// value if one is defined in the interface declaration
	emit_expr (f->initial_value, str, rstack, DEFAULT);

      } else {
	// If no default value is defined for the formal then create a
	// dummy intance of the appropriate type
	str += create_default_instance(f->subtype, rstack);
      }
      
      need_comma = true;
      if (fl->rest)
	str += ", ";
    }

  return need_comma;
}

void
emit_builtin_id (IR_Kind k, string &str)
{
  str += tree_kind_name(k);
}

/*
 * Headers
 */

void
m_emit_hdr (pIIR_EntityDeclaration e, string &str, RegionStack &rstack, int l)		// Entity header
{
  str += string("/* Entity class declaration */\n") +
       string("class ") + qid(e) + string(" {\n") +
       string("public:\n") +
       string("  ") + qid(e) + string(" (name_stack &iname, map_list *mlist);\n") +
       string("  ~") + qid(e) + string("() {};\n");

  pIIR_InterfaceList gens = get_generics (e);
  pIIR_InterfaceList ports = get_ports (e);
  pIIR_InterfaceList i;

  for (i = gens; i; i = i->rest) 
    {
      emit (i->first, str, rstack, 0);
      if (i->rest || ports)
	str += ", ";
    }

  emit_sig_decls (ports, str, rstack, 2);				// emit signal decls

  str += "};\n\n";
}

void							// Arch header
m_emit_hdr (pIIR_ArchitectureDeclaration a, string &str, RegionStack &rstack, int l)
{
  pIIR_EntityDeclaration e = a->entity;
  str += string("/* Architecture class declaration */\n") + 
       string("class ") + qid(a) + string(" : public ") + qid(e) + string(" {\n") +
       string("public:\n") +
       "  " + qid(a,"_") + string(" (name_stack &iname, map_list *mlist);\n") +	   
       "  " + string("~") + qid(a,"_") + string("();\n");

  emit_decls (a->declarations, str, rstack, 2);				// emit signal decls
  str += "};\n";										
  
  emit_hdr(a->architecture_statement_part, str, rstack, 2);		// emit process hdrs
}

void							// ConcStat headers
m_emit_hdr (pIIR_ConcurrentStatementList csl, string &str, RegionStack &rstack, int l)
{
  str += "\n/* Concurrent statement class declaration(s) */\n\n";
  while (csl)
    {
      emit_hdr (csl->first, str, rstack, l);  
      csl = csl->rest;
    }
}

void												
m_emit_hdr (pIIR_ConcurrentStatement p, string &str, RegionStack &rstack, int l)
{ 
  rstack.push(p);
  str += "/* Concurrent Statement header */\n"; 
  rstack.pop();
}

void
m_emit_hdr (pIIR_BlockStatement bs, string &str, RegionStack &rstack, int l)
{ 
  rstack.push(bs);
  str += "/* Block Statement header */\n"; 
  rstack.pop();
}

void										
m_emit_hdr (pIIR_ConcurrentGenerateStatement gs, string &str, RegionStack &rstack, int l)
{ 
  rstack.push(gs);
  str += "/* Concurent Generate Statement header */\n"; 
  rstack.pop();
}

void								
m_emit_hdr (pIIR_ComponentInstantiationStatement ci, string &str, RegionStack &rstack, int l)
{ 
}

void							//emit hdr for Process
m_emit_hdr (pIIR_ProcessStatement p, string &str, RegionStack &rstack, int l)
{
  rstack.push(p);
  str += "\n/* Class decl. process " + nid(p, BARE) + " */\n" +
       "class " + qid(p) + " : public process_base {\n" +
       "public:\n" +
       " " + qid(p) + "(" + qid(p->declarative_region) +
       " *architecture,\n" + 
       "          name_stack &iname);\n" + 
       " ~" + qid(p) + "() {};\n" +
       "  bool execute();\n";		// execute()
  

  // get context object. The contex stores various informations needed during
  // the code generation process
  ContextInfo &pctxt = *ActiveContext(rstack);


  // *****************************************************************************
  // create the driver pointer for each signal that is written by the
  // process
  // *****************************************************************************
  bool first = true;
  char *sep = " ";
  // first, get a list of driven signals
  access_list driven_sigs = 
    filter_unique(pctxt.accessed_objects, WRITE, 
		  tree_kind_list(IR_SIGNAL_DECLARATION, IR_SIGNAL_INTERFACE_DECLARATION));
  for (access_list::iterator i = driven_sigs.begin(); i != driven_sigs.end(); i++)
    {
      if (first) {
	first = false;
	str += "  driver_info";
      }
      str += string(sep) + "*" + qid((*i).declaration, "_", DRIVER);
      sep = ",";
    }
  if (!first) str += ";\n";


  // *****************************************************************************
  // get reader pointer for all signals that are read within the
  // process
  // *****************************************************************************
  pIIR_TypeDeclaration old_type = NULL;
  first = true;
  sep = " ";
  // first, get a list of read signals
  access_list read_sigs = 
    filter_unique(pctxt.accessed_objects, READ, 
		  tree_kind_list(IR_SIGNAL_DECLARATION, IR_SIGNAL_INTERFACE_DECLARATION));
  for (access_list::iterator i = read_sigs.begin(); i != read_sigs.end(); i++)
    {
      pIIR_TypeDeclaration type = get_declaration((*i).declaration->subtype);
      if (type != old_type) {
	if (!first) str +=";\n";
	str += "  " + qid(type,"_",TYPE);
	first = false;
	sep = " ";
	old_type = type;
      }
      str += string(sep) + "*" + qid((*i).declaration, "_", READER);
      sep = ",";
    }
  if (read_sigs.size()) str += ";\n";


  // *****************************************************************************
  // Create local copy of sig_info pointer for signals that are prefix
  // of an signal function kind attribute (e.g. event, active,
  // last_event, last_active,...)
  // *****************************************************************************
  old_type = NULL;
  first = true;
  sep = " ";
  // first, get a list of signals which are prefix of an attribute
  access_list attr_sigs = 
    filter_unique(pctxt.accessed_objects, ATTRIBUTE, 
		  tree_kind_list(IR_SIGNAL_DECLARATION, IR_SIGNAL_INTERFACE_DECLARATION));
  for (access_list::iterator i = attr_sigs.begin(); i != attr_sigs.end(); i++)
    {
      pIIR_TypeDeclaration type = get_declaration((*i).declaration->subtype);
      if (type != old_type) {
	if (!first) str +=";\n";
	str += "  sig_info<" + qid(type,"_",TYPE) + ">";
	first = false;
	sep = " ";
	old_type = type;
      }
      str += string(sep) + "*" + qid((*i).declaration, "_", DEFAULT);
      sep = ",";
    }
  if (attr_sigs.size()) str += ";\n";


  // *****************************************************************************
  // declare variables and constants
  // *****************************************************************************
  emit_decls(p->declarations, str, rstack, l);


  // *****************************************************************************
  // declare winfo_item array
  // *****************************************************************************
  if (pctxt.wait_statements.size()) {
    strstream wc;
    wc <<  pctxt.wait_statements.size() << (char)'\0';
    str += string("  winfo_item winfo[") + string(wc.str()) + string("];\n");
  }
  

  // *****************************************************************************
  // ptr to parent arch
  // *****************************************************************************
  str += string("  ") + qid(p->declarative_region) + string(" *arch;\n") + string("};\n");

  rstack.pop();
} // emit_hdr for Process


// Print subprogram prototype to str
void							//emit hdr for subprograms
m_emit_hdr (pIIR_SubprogramDeclaration sbp, string &str, RegionStack &rstack, int l)
{
  rstack.push(sbp);

  str += "/* Prototype for subprogram " + get_subprogram_name(sbp) + " */\n";
  string return_type_str;
  char *separator = "";
  // Check whether sbp is a function or a procedure
  bool is_function = sbp->is(IR_FUNCTION_DECLARATION); 

  // Determine return type of generated function
  if (sbp->is(IR_FUNCTION_DECLARATION))
    return_type_str = qid(get_declaration(pIIR_FunctionDeclaration(sbp)->return_type), "_", TYPE);
  else
    return_type_str = "void";

  // Print return type and name 
  str += return_type_str + " " + get_subprogram_name(sbp) + "(";
  
  // Next, analyze each subprogram parameter
  for (pIIR_InterfaceList il = sbp->interface_declarations; il; il = il->rest) {
    pIIR_InterfaceDeclaration par = il->first;
    // The parameter is passed in by reference if it is a non scalar
    // type or if the parameter is of mode OUT, INOUT or BUFFER.
    bool call_by_reference = !is_scalar_type(par->subtype) ||
      par->mode == IR_OUT_MODE || par->mode == IR_INOUT_MODE || 
      par->mode == IR_BUFFER_MODE;
    // Assume that parameter is not defined as const
    bool is_const = false;

    if (par->is(IR_VARIABLE_INTERFACE_DECLARATION)) {
    } else if (par->is(IR_CONSTANT_INTERFACE_DECLARATION)) {
      is_const = true;
    } else if (par->is(IR_FILE_INTERFACE_DECLARATION)) {
      codegen_error.error("%:error: sorry, file parameter are currently not supported", par);
    } else if (par->is(IR_SIGNAL_INTERFACE_DECLARATION)) {
      codegen_error.error("%:error: sorry, signal parameter are currently not supported", par);
    }

    // Append parameter type only as we are printing here the function
    // prototype
    str += separator;
    if (is_const) str += "const ";
    str += qid(get_declaration(par->subtype), "_", TYPE);
    if (call_by_reference) str += "&";
    
    // Set separator for next parameter
    separator = ",";
  }

  str += ");\n";

  rstack.pop();
}


void
m_emit_hdr (pIIR_EnumerationType et, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration decl = et->declaration;
  str += "/* Definitions for enumeration type " + qid(decl, "_", TYPE) + " */\n";
  
  // Emit enum info class definition 
  str += "class " + qid(decl, "_", INFO) + ":public enum_info_base{\n" +
    "   static const char *values[];\n" +
    "public:\n" +
    "   " + qid(decl, "_", INFO) + "():enum_info_base(0,"+ i_to_string(enum_item_number(et) - 1) + ",values) {};\n" + 
    "   static const char **get_values() { return values; }\n" +
    "   static int low() { return 0; }\n" +
    "   static int high() { return " + i_to_string(enum_item_number(et) - 1) + "; }\n" +
    "   static int left() { return 0; }\n" +
    "   static int right() { return " + i_to_string(enum_item_number(et) - 1) + "; }\n" + 
    "};\n";

  str += "extern " + qid(decl, "_", INFO) + " *" + qid(decl, "_", INFO) + "_INFO;\n";
}


void
m_emit_hdr (pIIR_ArrayType at, string &str, RegionStack &rstack, int l)
{
  int counter = 0;

  // Count number of dimensions
  for (pIIR_TypeList tl = at->index_types; tl; tl = tl->rest)
    counter++;
  
  // If the array hase more than a single dimension then first declare
  // the a separate internal array type for each dimension. Note that
  // e.g. a two-dimensional array "type mytype array(integer range <>,
  // positive range <>) of bit" is transformed into two one
  // dimentional arrays similar to: "type internal_array is
  // array(positive range <>) of bit" and "type mytype is
  // array(integer range <>) of internal_array".
  string array_type_str = qid(at->element_type->declaration, "_", TYPE);
  for (int i = counter; i >= 1; i--) {
    pIIR_TypeList tl = at->index_types;
    for (int j = 1; j < i; j++)
      tl = tl->rest;
    array_type_str = "array_type<" + array_type_str + "," + qid(tl->first->declaration, "_", INFO) + ">";
  }

  str += "extern array_info *" + qid(at->declaration, "_", INFO) + "_INFO;\n";
  str += "#define " + qid(at->declaration, "_", TYPE) + " " + array_type_str + "\n";
  str += "typedef array_info " + qid(at->declaration, "_", INFO) + ";\n\n";
}


void
m_emit_hdr (pIIR_ArraySubtype ast, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration decl = ast->declaration;
  // get base type of array subtype
  if (!ast->base->is(IR_ARRAY_TYPE))
    internal_error(true);
  pIIR_ArrayType array_type = pIIR_ArrayType(ast->base);

  int counter = 0;

  // Count number of dimensions
  for (pIIR_TypeList tl = ast->constraint; tl; tl = tl->rest)
    counter++;
  
  // If the array hase more than a single dimension then first declare
  // the a separate internal array type for each dimension. Note that
  // e.g. a two-dimensional array "type mytype array(integer range <>,
  // positive range <>) of bit" is transformed into two one
  // dimentional arrays similar to: "type internal_array is
  // array(positive range <>) of bit" and "type mytype is
  // array(integer range <>) of internal_array".
  string array_type_str = qid(array_type->element_type->declaration, "_", TYPE);
  for (int i = counter; i >= 1; i--) {
    pIIR_TypeList tl = array_type->index_types;
    for (int j = 1; j < i; j++)
      tl = tl->rest;
    array_type_str = "array_type<" + array_type_str + "," + qid(tl->first->declaration, "_", INFO) + ">";
  }

  str += "extern array_info *" + qid(ast->declaration, "_", INFO) + "_INFO;\n";
  str += "#define " + qid(ast->declaration, "_", TYPE) + " " + array_type_str + "\n";
  str += "typedef array_info " + qid(ast->declaration, "_", INFO) + ";\n\n";
}


void
m_emit_hdr (pIIR_ScalarSubtype sst, string &str, RegionStack &rstack, int l)
{
  if (sst->base->is(IR_ENUMERATION_TYPE)) {
    pIIR_TypeDeclaration decl = sst->declaration;
    str += "/* Definitions for enumeration type " + qid(decl, "_", TYPE) + " */\n";
    
    // determine range of the new enumeration subtype
    pIIR_ExplicitRange er = pIIR_ExplicitRange(sst->range);
    int left_pos = pIIR_EnumLiteralReference(er->left)->value->enum_pos;
    int right_pos = pIIR_EnumLiteralReference(er->right)->value->enum_pos;

    // Emit enum info class definition 
    str += "class " + qid(decl, "_", INFO) + ":public enum_info_base{\n" +
      "   static const char *values[];\n" +
      "public:\n" +
      "   " + qid(decl, "_", INFO) + "():enum_info_base(" + i_to_string(left_pos) + ","+ i_to_string(right_pos) + ",values) {};\n" + 
      "   static const char **get_values() { return values; }\n" +
      "   static int low() { return " + i_to_string(left_pos) + "; }\n" +
      "   static int high() { return " + i_to_string(right_pos) + "; }\n" +
      "   static int left() { return " + i_to_string(left_pos) + "; }\n" +
      "   static int right() { return " + i_to_string(right_pos) + "; }\n" + 
      "};\n";
    
    str += "extern " + qid(decl, "_", INFO) + " *" + qid(decl, "_", INFO) + "_INFO;\n" +
      "#define " + qid(decl, "_", TYPE) + " enum_type<" + qid(decl, "_", INFO) + ">\n";

  } else
    codegen_error.error("%:error: sorry, this subtype declaration is currently not supported", sst);
}


/*
 * Simulation Object Implementation
 */

void
m_emit_impl (pIIR_ArrayType at, string &str, RegionStack &rstack, int l)
{
  /* Nothing to do */
}

IR_StaticLevel
m_emit_info_init (pIIR_ArrayType at, string &str, RegionStack &rstack, int l)
{
  // Count number of dimensions
  int counter = 0;
  for (pIIR_TypeList tl = at->index_types; tl; tl = tl->rest)
    counter++;
  
  // If the array hase more than a single dimension then first declare
  // the a separate internal array type for each dimension. Note that
  // e.g. a two-dimensional array "type mytype array(integer range <>,
  // positive range <>) of bit" is transformed into two one
  // dimentional arrays similar to: "type internal_array is
  // array(positive range <>) of bit" and "type mytype is
  // array(integer range <>) of internal_array".
  string array_info_str = qid(at->element_type->declaration, "_", INFO) + "_INFO";
  for (int i = counter; i >= 1; i--) {
    pIIR_TypeList tl = at->index_types;
    for (int j = 1; j < i; j++)
      tl = tl->rest;
    array_info_str = "new array_info(" + array_info_str + "," + qid(tl->first->declaration, "_", INFO) + "_INFO,-1)";
  }

  str += "=" + array_info_str;

  return IR_LOCALLY_STATIC;
}


void
m_emit_impl (pIIR_ArraySubtype ast, string &str, RegionStack &rstack, int l)
{
  /* Nothing to do */
}


IR_StaticLevel
m_emit_info_init (pIIR_ArraySubtype ast, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration decl = ast->declaration;
  // get base type of array subtype
  if (!ast->base->is(IR_ARRAY_TYPE))
    internal_error(true);
  pIIR_ArrayType array_type = pIIR_ArrayType(ast->base);

  // Count number of dimensions
  int counter = 0;
  for (pIIR_TypeList tl = ast->constraint; tl; tl = tl->rest)
    counter++;

  // If the array hase more than a single dimension then first declare
  // the a separate internal array type for each dimension. Note that
  // e.g. a two-dimensional array "type mytype array(integer range <>,
  // positive range <>) of bit" is transformed into two one
  // dimentional arrays similar to: "type internal_array is
  // array(positive range <>) of bit" and "type mytype is
  // array(integer range <>) of internal_array".
  string array_info_str = qid(array_type->element_type->declaration, "_", INFO) + "_INFO";
  IR_StaticLevel slevel = IR_LOCALLY_STATIC;
  for (int i = counter; i >= 1; i--) {
    pIIR_TypeList tl = ast->constraint;
    for (int j = 1; j < i; j++)
      tl = tl->rest;
    string base_info_str = "new array_info(" + array_info_str + "," + 
      qid(get_declaration(tl->first), "_", INFO) + "_INFO,-1)";

    // Get index range
    vector<RangeDescriptor> range_desc = get_discrete_range(tl->first, rstack, IR_NOT_STATIC);
    slevel = merge_level(range_desc[0].static_level, slevel);
    // Bail out if more than a single range descriptor is returned
    if (range_desc.size() > 1) internal_error(true);
    // Next convert range descriptor to int strings
    string left_str, dir_str, right_str;
    range_desc[0].rangedes_to_int_string(left_str, dir_str, right_str, rstack);

    // setup init value for the array info instance
    array_info_str = "new array_info(" + base_info_str + "," + left_str + "," + dir_str + "," + right_str + ",-1)";
  }

  // Finally, prepend "="
  str += "=" + array_info_str;

  return slevel; // return compound static level of ranges
}


void
m_emit_impl (pIIR_ScalarSubtype sst, string &str, RegionStack &rstack, int l)
{
  if (sst->base->is(IR_ENUMERATION_TYPE)) {
    pIIR_TypeDeclaration decl = sst->declaration;

    // determine left pos of new enumeration subtype
    pIIR_ExplicitRange er = pIIR_ExplicitRange(sst->range);
    int left_pos = pIIR_EnumLiteralReference(er->left)->value->enum_pos;

    str += "/* Implementation of enumeration type " + qid(decl, "_", TYPE) + " */\n";
    str += "const char *" + qid(decl, "_", INFO) + "::values[]=" + 
      "&" + qid(sst->base->declaration, "_", INFO) + "_INFO->get_values()[" + i_to_string(left_pos) + "];\n";
    
    str += qid(decl, "_", INFO) + " *" + qid(decl, "_", INFO) + "_INFO=new " + qid(decl, "_", INFO) + ";\n";

  } else
    codegen_error.error("%:error: sorry, this subtype declaration is currently not supported", sst);
}
  


void
m_emit_impl (pIIR_EnumerationType et, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration decl = et->declaration;

  str += "/* Implementation of enumeration type " + qid(decl, "_", TYPE) + " */\n";
  str += "const char *" + qid(decl, "_", INFO) + "::values[" +/* + i_to_string(enum_item_number(et))+*/ "]={";
  char *separator = "";
  for (pIIR_EnumerationLiteralList ell = et->enumeration_literals; ell; ell = ell->rest) {
    str += string(separator) + "\"" + string(ell->first->declarator->text.to_chars()) + "\"";
    separator = ",";
  }
  str += "};\n";
}


IR_StaticLevel
m_emit_info_init (pIIR_EnumerationType et, string &str, RegionStack &rstack, int l)
{
  pIIR_TypeDeclaration decl = et->declaration;
  str += "=new " + qid(decl, "_", INFO) + ";\n";

  return IR_LOCALLY_STATIC;
}


void
m_emit_impl (pIIR_EntityDeclaration e, string &str, RegionStack &rstack, int l)
{
  str += "/* Implementation of entity methods */\n";
  str += qid(e ,"_");
  str += "::\n";
  str += qid(e, "_");
  str += "(name_stack &iname, map_list *mlist) {\n";

  pIIR_InterfaceList ports = get_ports (e);
  pIIR_InterfaceList i;
  int j = 0;

  for (i = ports; i; i = i->rest) 
      if (i->first->is(IR_SIGNAL_INTERFACE_DECLARATION))
	emit_sig_interfacecon((pIIR_SignalInterfaceDeclaration) i->first, str, rstack, j++);

  if (j) 
    str += "    iname.pop();\n";
  str += "};\n\n";
}


void
m_emit_impl (pIIR_ArchitectureDeclaration a, string &str, RegionStack &rstack, int l)
{
  emit_impl(a->architecture_statement_part, str, rstack, 2);		// process impls
  emit_handle (a, str);					// handle for sim
  emit_constructor(a, str, rstack);					// arch constructor
}

void
emit_handle(pIIR_ArchitectureDeclaration a ,string &str)
{
  str += "/* handle for simulator to find architecture */\n";	// handle func
  str += "void\n";
  str += qid (a, "_");
  str += "_handle(name_stack &iname, map_list *mlist) {\n";
  str += " REPORT(cout << \"Starting constructor ";
  str += qid (a, "_");
  str += " ...\" << endl);\n new ";
  str += qid (a, "_");
  str += "(iname, mlist);\n";
  str += " REPORT(cout << \"done.\" << endl);\n};\n";
  str += "int ";					// dummy var
  str += qid (a, "_");
  str += "_dummy =\nadd_handle(\"";
  str += qid (a, "\",\"",BARE);  
  str += "\", &";
  str += qid (a, "_");
  str += "_handle);\n";
}


// emit component instantiation code
void 
emit_component_instantiation(pIIR_ComponentInstantiationStatement cs, string &str, 
			     RegionStack &rstack, int l)
{
  str += spaces(l) + "tmpml.reset();\n";
  // generic_map_aspect port_map_aspect
  for (pIIR_AssociationList al = cs->port_map_aspect; al; al = al->rest) {
    pIIR_AssociationElement ae = al->first;

    str += spaces(l) + "tmpml.signal_map(\":" + ae->formal->declarator->text.to_chars() +
      "\",NULL,";

    // determine actual signal. First, get list of accessed objects
    // then extract signals from the list
    ContextInfo tmpctxt;
    get_context(ae->actual, tmpctxt, rstack, false, 0);
    list<AccessDescriptor> actual_signals = filter(tmpctxt.accessed_objects, ACCESS, 
		  tree_kind_list(IR_SIGNAL_DECLARATION, IR_SIGNAL_INTERFACE_DECLARATION));
    // bail out if not exactly one signal is included in the actual
    // expression
    if (actual_signals.size() != 1)
      codegen_error.error("%:error: sorry, currently only a single signal actual is supported", cs);
    // print actual signal
    str += qid(actual_signals.front().declaration) + ",";
    // append acl for actual
    list<string> acl_list;
    get_acl(ae->actual, acl_list, rstack, IR_GLOBALLY_STATIC, id_type(SIGNAL, ARCHREF), true);
    string actual_acl = sprint_acl(acl_list, "tmpacl2->clear()");
    if (actual_acl == "") 
      actual_acl = "NULL";
    else
      actual_acl = "&(" + actual_acl + ")";
    str += actual_acl + ");\n";
  }

  str += spaces(l) + "kernel.elaborate_component(\"\",";
  if (cs->instantiated_unit->is(IR_ARCHITECTURE_REF)) {
    pIIR_ArchitectureRef aref = (pIIR_ArchitectureRef)cs->instantiated_unit;
    string entity_name = aref->entity->declarator->text.to_chars();
    string architecture_name = 
      aref->architecture_name != NULL? aref->architecture_name->text.to_chars() : "";
    str += "\"" + entity_name + "\",\"" + architecture_name + "\",iname,";
    str += "\":" + string(cs->declarator->text.to_chars()) + "\",&tmpml);\n";
  } else
    internal_error(true);

}


// emit architecture constructor code
void
emit_constructor(pIIR_ArchitectureDeclaration a, string &str, RegionStack &rstack)
{
  rstack.push(a);
  pIIR_EntityDeclaration e = a->entity;

  str += "\n/* Architecture Constructor */\n";	// handle func

  str += qid (a, "_");
  str += "::\n";
  str += qid (a, "_");
  str += "(name_stack &iname, map_list *mlist) :\n  ";
  str += qid (e, "_");
  str += "(iname, mlist) {\n";
  str += "    iname.push(\":";
  emit_noqual_id(a, str, BARE);
  str += "\");\n";
  str += "    iname.push(\"\");\n";
  int i=0;

  if (a->declarations) {
    // emit code to init object declared within the architecture
    emit_decls_init (a->declarations, str, rstack, 4);
  }


  // emit code to create all processes and components directly
  // instantiated within the architecture body
  pIIR_ConcurrentStatementList s=a->architecture_statement_part;
  while (s)
    {
      if (s->first->is(IR_PROCESS_STATEMENT)) {
	// add process to architecture
	pIIR_ProcessStatement p=(pIIR_ProcessStatement) s->first; // !!!
	str += "    kernel.add_process(new " + qid(p) + "(\n" +
	  "                     this, iname.set(\":" + nid(p,BARE) +
	  "\")));\n";
      } else if (s->first->is(IR_COMPONENT_INSTANTIATION_STATEMENT)) {
	// add component instantiation to architecture
	emit_component_instantiation((pIIR_ComponentInstantiationStatement)s->first, str, rstack, 4);
      }

      s = s->rest;
    }
  str += string("    iname.pop(); /* pop last declaration from name stack */\n") +
       string("    iname.pop(); /* pop architecture from name stack */\n") +
       string("};\n");

  rstack.pop();
}
  

// emit port signal constructor part
void
emit_sig_interfacecon (pIIR_SignalInterfaceDeclaration s, string &str, RegionStack &rstack, int i)
{
  pIIR_TypeDeclaration type_declaration = s->subtype->declaration;

  // determine info instance
  string info_str;
  if (is_implicit_array_subtype(s->subtype)) {
    type_declaration = ((pIIR_ArraySubtype)(s->subtype))->immediate_base->declaration;
    info_str = create_array_info_obj(pIIR_ArraySubtype(s->subtype), rstack, true);
  } else
    info_str = qid(type_declaration,"_",INFO) + "_INFO";

  // determine mode of the port signal
  string mode_str;
  if (s->mode == IR_IN_MODE)
    mode_str = "vIN";
  else if (s->mode == IR_OUT_MODE)
    mode_str = "vOUT";
  else if (s->mode == IR_INOUT_MODE)
    mode_str = "vINOUT";
  else if (s->mode == IR_BUFFER_MODE)
    mode_str = "vBUFFER";
 
  str += string("    ") + qid(s) + string("= new sig_info<");
  str += qid(type_declaration,"_",TYPE);
  if (i)						
    str += ">\n      (iname.set(\":";
  else								// first time thru do a push
    str += ">\n      (iname.push(\":";
  str += nid(s,BARE) + "\"),mlist," + info_str + "," + mode_str + ",NULL);\n";

  if (s->initial_value) 
    {
      str += "    " + qid(s) + "->init(" ;
      emit_expr (s->initial_value, str, rstack, DEFAULT);
      str += ");\n";
    }
}


void
m_emit_impl (pIIR_ConcurrentStatementList s, string &str, RegionStack &rstack, int l)
{
  while (s)
    {
      str += spaces(l);
      emit_impl (s->first, str, rstack, l);
      s = s->rest;
    }
}


string
sprint_acl(list<string> &acl_list, const string acl_object)
{
  if (acl_list.size() == 0)
    return string("");

  string result = acl_object;

  for (list<string>::iterator it = acl_list.begin(); it != acl_list.end(); it++)
    result += "<<" + (*it);
  
  return result;
}


void
m_emit_impl (pIIR_ProcessStatement p, string &str, RegionStack &rstack, int l)
{
  rstack.push(p);
  ContextInfo &ctxt = *ActiveContext(rstack);

  // ******************************************************************
  // append prolog
  // ******************************************************************
  str += "\n/* Implementation of process " + qid(p,"_",BARE) + " methods */\n" +
       qid(p) + "::\n" + 
       qid(p) + "(" + qid(p->declarative_region) +
       " *architecture, name_stack &iname) : process_base(iname) {\n" + 
       "    arch=architecture;\n";


  // ******************************************************************
  // map signals which are read within the process
  // ******************************************************************
  set<pIIR_ObjectDeclaration, less<void*> > sig_set;
  for (access_list::iterator i = ctxt.accessed_objects.begin(); 
       i != ctxt.accessed_objects.end(); i++)
    {
      // skip the signals that are not read
      if ((((*i).access_type & READ) == 0) ||
	  !((*i).declaration->is(IR_SIGNAL_DECLARATION) || 
	    (*i).declaration->is(IR_SIGNAL_INTERFACE_DECLARATION))|| 
	  sig_set.find((*i).declaration) != sig_set.end()) continue;
      sig_set.insert((*i).declaration);
      
      str += "    " + qid((*i).declaration, "_", READER) + "=&" + 
	qid((*i).declaration, "_", id_type(SIGNAL, ARCHREF)) + "->reader();\n"; 

      // If an attribute has been applied to the signal then the
      // sig_info pointer must be initialized
      if ((*i).access_type & ATTRIBUTE)
	str += "    " + qid((*i).declaration, "_", DEFAULT) + "=" + 
	  qid((*i).declaration, "_", id_type(SIGNAL, ARCHREF)) + ";\n"; 
    }


  // ******************************************************************
  // create the driver for the signals that are written by the process.
  // ******************************************************************
  // First extract written signals from ctxt.accessed_objects list.
  list<AccessDescriptor> written_signals = 
    filter(ctxt.accessed_objects, WRITE, 
	   tree_kind_list(IR_SIGNAL_DECLARATION, IR_SIGNAL_INTERFACE_DECLARATION));

  while (written_signals.size()) {
    set<string> acls_for_signal;
    // get first signal in written_signal list and remove it from the
    // list
    AccessDescriptor current_signal = written_signals.front();
    written_signals.erase(written_signals.begin());

    while (true) {
      // get acl for the signal target expression and convert it into a
      // string. Append the string to the other acl strings for that
      // signal.
      list<string> acl_list;
      get_acl(current_signal.object_ref, acl_list, rstack, IR_GLOBALLY_STATIC, 
	      id_type(SIGNAL, ARCHREF), true);
      acls_for_signal.insert(sprint_acl(acl_list, "tmpacl->clear()"));
      // search for the next assignment to the same signal
      list<AccessDescriptor>::iterator iter = written_signals.begin();
      for (; iter != written_signals.end(); iter++)
	if ((*iter).declaration == current_signal.declaration) break;
      // exit loop if no more assignment to the same signal were
      // found. Otherwise, copy found signal to current_sig and remove
      // the corresponding entry from written_signals.
      if (iter == written_signals.end()) break;
      current_signal = (*iter);
      written_signals.erase(iter);
    }
    
    // acls_for_signal contains all different acl values which were
    // created from the signal target expressions (wihtin a process)
    // for a specific signal. Now, build the driver creation code.
    string start_str = "    " + qid(current_signal.declaration, "_", DRIVER) + 
      string("=kernel.get_driver(this,") + 
      qid(current_signal.declaration, "_", id_type(SIGNAL, ARCHREF));
    string next_start_str = "    " + string("kernel.get_driver(") + 
	qid(current_signal.declaration, "_", DRIVER) + ",this";

    for (set<string>::iterator aiter = acls_for_signal.begin();
	 aiter != acls_for_signal.end(); aiter++) {
      str += start_str;
      if ((*aiter) != "") // add acl
	str += ",&(" + (*aiter) + ")";
      str += string(");\n");
      start_str = next_start_str;
    }
  }

  // ******************************************************************
  // create the wait_info objects
  // ******************************************************************
  int windex = 0;
  for (list<pIIR_WaitStatement>::iterator i = ctxt.wait_statements.begin(); 
       i != ctxt.wait_statements.end(); i++)
    {
      // collect all signals the wait statement is sensitive on
      ContextInfo wctxt;
      get_context((*i), wctxt, rstack, false, 0);
      if (wctxt.accessed_objects.size() == 0) continue;
      
      // Analyze each signal which was referenced in the corresponding
      // wait expression
      string codestr;
      int length = 0;
      for (access_list::iterator iter = wctxt.accessed_objects.begin(); 
	   iter != wctxt.accessed_objects.end(); iter++) {
	if (((*iter).access_type & SENSITIVE) == 0) continue;
	list<string> acl_list;
	// convert each signal expression into an acl list
	get_acl((*iter).object_ref, acl_list, rstack, IR_GLOBALLY_STATIC, id_type(SIGNAL, ARCHREF), true);
	codestr += string("     sal.add(") + qid((*iter).declaration, "_", id_type(SIGNAL, ARCHREF));
	if (acl_list.size())
	  codestr += ",&(" + sprint_acl(acl_list, "tmpacl->clear()") + ")";
	codestr += string(");\n");

	length++;
      }
      
      // prolog
      strstream lstr;
      lstr << length << (char)'\0';
      str += 
	string("    {\n") +
	string("     sigacl_list sal(") + lstr.str() + string(");\n");
      
      // add code 
      str += codestr;

      // epilog
      strstream istr;
      wait_info_index(*i) = windex++;
      istr << wait_info_index(*i) << (char)'\0';
      str += string("     winfo[") + istr.str() + string("]=kernel.setup_wait_info(sal,this);\n") +
	string("    }\n");
    }


  // ******************************************************************
  // append epilog code
  // ******************************************************************
  string code_str = 
    string("}\n") +
    string("bool\n") +
    qid(p) + string("::execute()\n") +
    string("{\n");

  emit_process_body (p, code_str, rstack, l+2);			// SeqStatList
  code_str += "}\n\n";

  // *****************************************************************************
  // init variables and constants.  Note that the process code is
  // emitted first as due to the operations some internal variables
  // may be generated which must be initialized now!!!!
  // *****************************************************************************

  if (p->declarations)
    // Emit declarations
    emit_decls_init(p->declarations, str, rstack, 4);


  // Now, append process code!
  str += code_str;

  // Remove context from context stack
  rstack.pop();
}



void
emit_process_body (pIIR_ProcessStatement p, string &str, RegionStack &rstack, int l)
{
  pIIR_SequentialStatementList sl = p->process_statement_part;

  ContextInfo &ctxt = *ActiveContext(rstack);

  if (ctxt.wait_statements.size() &&
      ! p->is(IR_SENSITIZED_PROCESS_STATEMENT)) {
    str += "  switch (jmp) {\n";					// create jump tbl
    for(int i=1 ; i <= ctxt.wait_statements.size(); i++)
      str += "      case " + i_to_string(i) + ": goto lab" + i_to_string(i) + ";\n";
    str += "  };\n\n  lab0:\n";
  }

  // emit sequential statements
  emit_impl(sl, str, rstack, l);

  // the last statement of a process loops back to the beginning
  if (ctxt.wait_statements.size() &&
      ! p->is(IR_SENSITIZED_PROCESS_STATEMENT))
    str += "    goto lab0;\n";
}


// Print subprogram implementation
void
m_emit_impl (pIIR_SubprogramDeclaration sbp, string &str, RegionStack &rstack, int l)
{
  rstack.push(sbp);
  ContextInfo &ctxt = *ActiveContext(rstack);

  str += "/* Implementation of subprogram " + get_subprogram_name(sbp) + " */\n";
  string return_type_str;
  string copy_back_code = spaces(4) + "rlabel:\n", parameter_setup_code;
  char *separator = "";

  // Check whether sbp is a function or a procedure
  bool is_function = sbp->is(IR_FUNCTION_DECLARATION); 

  // Determine return type of generated function
  if (sbp->is(IR_FUNCTION_DECLARATION))
    return_type_str = qid(get_declaration(pIIR_FunctionDeclaration(sbp)->return_type), "_", TYPE);
  else
    return_type_str = "void";

  // Print return type and name 
  str += return_type_str + " " + get_subprogram_name(sbp) + "(";
  
  // Next, analyze each subprogram parameter
  for (pIIR_InterfaceList il = sbp->interface_declarations; il; il = il->rest) {
    pIIR_InterfaceDeclaration par = il->first;
    // The parameter is passed in by reference if it is a non scalar
    // type or if the parameter is of mode OUT, INOUT or BUFFER.
    bool call_by_reference = !is_scalar_type(par->subtype) ||
      par->mode == IR_OUT_MODE || par->mode == IR_INOUT_MODE || 
      par->mode == IR_BUFFER_MODE;
    // Assume that parameter is not defined as const
    bool is_const = false;
    // Determine whether the parameter must be copied back
    bool copy_back = is_scalar_type(par->subtype) && 
      (par->mode == IR_OUT_MODE || par->mode == IR_INOUT_MODE || 
       par->mode == IR_BUFFER_MODE);
    // Determine whether the parameter must be copied in
    bool copy_in = copy_back && 
      (par->mode == IR_INOUT_MODE || par->mode == IR_BUFFER_MODE);
    // Determine whether the parameter can be used directly or an
    // temporary must be created. It cannot be used directly if the
    // parameter value must be copied back or the parameter type is a
    // constrained array type.
    bool direct_use = !(copy_back || is_constrained_array_type(par->subtype));

    if (par->is(IR_VARIABLE_INTERFACE_DECLARATION)) {
    } else if (par->is(IR_CONSTANT_INTERFACE_DECLARATION)) {
      is_const = true;
    } else if (par->is(IR_FILE_INTERFACE_DECLARATION)) {
      codegen_error.error("%:error: sorry, file parameter are currently not supported", par);
    } else if (par->is(IR_SIGNAL_INTERFACE_DECLARATION)) {
      // Signal values are NOT copied back! Instead, any operations
      // are directly applied on the corresponding reader/driver...
      copy_back = false;
      codegen_error.error("%:error: sorry, signal parameter are currently not supported", par);
    }

    // Append parameter type 
    str += separator;
    if (is_const) str += "const ";
    str += qid(get_declaration(par->subtype), "_", TYPE) + " ";
    if (call_by_reference) str += "&";

    // Append parameter name
    string par_name = qid(par, "_", DEFAULT);
    if (!direct_use) par_name += "_PAR";
    str += par_name;

    // If the paramter is not used directly create another variable
    // which is actually used within the subprogram body
    if (!direct_use) {
      // If the parameter is passed back by copy then add the
      // corresponding code
      if (copy_back)
	copy_back_code += spaces(4) + par_name + "=" + qid(par, "_", DEFAULT) + ";\n";
      
      // Check type of parameter. If it is an implicit array subtype
      // then append code to setup the array_info instance of the
      // parameter
      if (is_implicit_array_subtype(par->subtype)) {
	string info_str = create_array_info_obj(pIIR_ArraySubtype(par->subtype), rstack, true);
	parameter_setup_code += spaces(4) + 
	  qid(get_declaration(par->subtype), "_", id_type(TYPE | ALIAS, DEFAULT)) + " " + 
	  qid(par, "_", DEFAULT) + "(" + info_str + "," + par_name + ".data)";
      } else
	parameter_setup_code += spaces(4) + 
	  qid(get_declaration(par->subtype), "_", TYPE) + " " + qid(par, "_", DEFAULT);

      // If the parameter is passed in by copy then add the
      // corresponding code
      if (copy_in)
	parameter_setup_code += "=" + par_name + ";\n";
      else
	parameter_setup_code += ";\n";

    }

    // Set separator for next parameter
    separator = ",";
  }

  str += ")\n";
  str += "{\n";
  
  // ********************************************************
  // The subprogram body starts here!
  // ********************************************************

  // Add parameter setup code
  str += parameter_setup_code;
  
  // emit sequential statements
  string body_code;
  emit_impl(sbp->subprogram_body, body_code, rstack, 4);

  // Emit local declarations of the subprogram 
  emit_decls(sbp->declarations, str, rstack, 4);
  // Emit initialization code of local declarations
  emit_decls_init(sbp->declarations, str, rstack, 4);

  // Now, append subprogram body code. Note that the subprogram body
  // is emitted first before the declarations are printed.
  str += body_code;
  
  // If subprogram is not a function then add copy back code and
  // return subprogram. The copy back code is executed when the
  // subprogram returns.
  if (!is_function)
    str += copy_back_code;

  if (!is_function)
    str += spaces(4) + "return;\n";

  str += "}\n";

  rstack.pop();
}


void                                                            // SeqStatList impl
m_emit_impl (pIIR_SequentialStatementList sl, string &str, RegionStack &rstack, int l)	
{
  while (sl)
    {
      pIIR_SequentialStatement s = sl->first;
      
      if (s->label)
	{
	  emit_id (s->label->declarator, str);
	  str += ":\n";
	}
      str += spaces(l);
      emit_impl (s, str, rstack, l);
      sl = sl->rest;
    }
}


void
m_emit_impl (pIIR_NullStatement, string &str, RegionStack &rstack, int l)
{
  str += "/* NullStat impl */;\n";
}

void
m_emit_impl (pIIR_ReturnStatement r, string &str, RegionStack &rstack, int l)
{
  if (r->return_expression)
    {
      str += "return ";
      emit (r->return_expression, str, rstack, DEFAULT);
      str += ";\n";
    }
  else
    str += "return;\n";
}

void
m_emit_impl (pIIR_VariableAssignmentStatement a, string &str, RegionStack &rstack, int l)
{
  emit_expr (a->target, str, rstack, id_type(READER, DEREF));
  str += " = ";
  emit_expr (a->expression, str, rstack, id_type(READER, DEREF));
  str += ";\n";
}

void
m_emit_impl (pIIR_IfStatement is, string &str, RegionStack &rstack, int l)
{
  str += "if(";
  emit_expr (is->condition, str, rstack, DEFAULT);
  str += ") {\n";
  emit_impl (is->then_sequence, str, rstack, l+2);
  if (is->else_sequence)
    {
      str += spaces(l) + "} else {\n";
      emit_impl (is->else_sequence, str, rstack, l+2);
    }
  str += spaces(l) + "}\n";
}

void
m_emit_impl (pIIR_CaseStatement cs, string &str, RegionStack &rstack, int l)
{
  // A case statement. This is probably too complicated. We should
  // somehow unify ranges and subtypes
  
  
  if (is_scalar_type(cs->expression->subtype)) {
    //****************************************************
    // Expression is scalar. 
    //****************************************************
    
    str += "switch(";
    emit_expr (cs->expression, str, rstack, DEFAULT);
    str += ") {\n";

    for (pIIR_CaseStatementAlternativeList al = cs->case_statement_alternatives; al; al = al->rest) {
      pIIR_CaseStatementAlternative a = al->first;
      
      for (pIIR_ChoiceList cl = a->choices; cl; cl = cl->rest)
	
	if (cl->first->is(IR_CHOICE_BY_RANGE)) {
	  //***************************************
	  // Choice by range
	  //***************************************

	  pIIR_ChoiceByRange cbr = pIIR_ChoiceByRange(cl->first);

	  // Get range.
	  vector<RangeDescriptor> range_desc = get_discrete_range(cbr->range, rstack, IR_LOCALLY_STATIC);
	  int left, right;
	  IR_Direction dir;
	  range_desc[0].rangedes_to_int(left, dir, right, rstack);
	  if (left == right)
	    str += spaces(l) + "case " + i_to_string(left) + ":\n";
	  else if (left < right && dir == IR_DIRECTION_UP)
	    str += spaces(l) + "case " + i_to_string(left) + " ... " + i_to_string(right) + ":\n";
	  else if (left > right && dir != IR_DIRECTION_UP)
	    str += spaces(l) + "case " + i_to_string(right) + " ... " + i_to_string(left) + ":\n";

	} else if (cl->first->is(IR_CHOICE_BY_EXPRESSION)) {
	  //***************************************
	  // Choice by expression
	  //***************************************
	  
	  pIIR_ChoiceByExpression cbe = pIIR_ChoiceByExpression(cl->first);
	  int value = int(folded_value(cbe->value).long_value);
	  str += spaces(l) + "case " + i_to_string(value) + ":\n";

	} else if (cl->first->is(IR_CHOICE_BY_OTHERS)) {
	  //***************************************
	  // Choice by others
	  //***************************************
	  str += spaces(l) + "default:\n";
	}

      str += spaces(l + 1) + "{\n";
      // Analyze sequential statements associated with alternative
      emit_impl(a->sequence_of_statements, str, rstack, l + 2);
      str += spaces(l + 1) + "}\n";
    }
    
    str += spaces(l) + "}\n";
  }
}


void
m_emit_impl (pIIR_LoopStatement ls, string &str, RegionStack &rstack, int l)
{
  ContextInfo &ctxt = *ActiveContext(rstack);
  pIIR_DeclarativeRegion active_region = ActiveDeclarativeRegion(rstack);

  // Check whether a name has been associated with the loop statement
  if (ls->declarative_region->declarator == NULL) {
    // If the loop statement has no name then create a default name
    string loop_name = i_to_string(ls->declarative_region->seqno) + loop_default_postfix;
    ls->declarative_region->declarator = new IIR_TextLiteral(0, loop_name.c_str());
  }

  if (ls->is(IR_WHILE_LOOP_STATEMENT))
    {
      str += "while(";
      emit_expr (pIIR_WhileLoopStatement(ls)->condition, str, rstack, DEFAULT);
      str += ")";
    }
  else if (ls->is(IR_FOR_LOOP_STATEMENT))
    {
      pIIR_ForLoopStatement fl = pIIR_ForLoopStatement (ls);
      // generate a internal variable for the loop variable, the
      // start value and the end value
      string loop_var_name = qid(fl->iterator);
      string loop_end_name = loop_var_name + "_le";
      string loop_var_type = qid(get_declaration(fl->iterator->subtype), "_", TYPE);

      insert_internal_object_declaration(NULL, active_region, loop_var_name, loop_var_type, "", 0);
      insert_internal_object_declaration(NULL, active_region, loop_end_name, loop_var_type, "", 0);

      // Get iteration range
      vector<RangeDescriptor> range_desc = 
	get_discrete_range (((pIIR_ScalarSubtype)fl->iterator->subtype)->range, rstack, IR_NOT_STATIC);
      string begin ,direction, end;
      range_desc[0].rangedes_to_int_string(begin, direction, end, rstack);
      
      str += "for (" + loop_var_name + "=" + begin + "," + loop_end_name + "=" + end + ";\n";
      str += spaces(l + 5) + loop_var_name + string(direction=="to"? "<=" : ">=") + loop_end_name + ";\n";
      str += spaces(l + 5) + loop_var_name + string(direction=="to"? "++" : "--") + ")";
    }
  else
    str += "while(1)";

  str += " {\n";
  emit_impl (ls->sequence_of_statements, str, rstack, l+2);
  str += spaces(l) + "}\n";
}

void
m_emit_impl (pIIR_ProcedureCallStatement pcs, string &str, RegionStack &rstack, int l)
{
  // A procedure call.

  pIIR_ProcedureDeclaration p = pcs->procedure;
  //emit_id (p, str);
  str += get_subprogram_name(p);
  str += " (";
  emit_associations (str, rstack, pcs->actual_parameter_part, p->interface_declarations);
  str += ");\n";
}

void
m_emit_impl (pIIR_WaitStatement ws, string &str, RegionStack &rstack, int l)
{
  int windex = wait_info_index(ws);
  str += "wait(winfo[" + i_to_string(windex) + "]);\n";
  str += spaces(l) + "jmp = " + i_to_string(windex + 1) + ";\n";
  str += spaces(l) + "return true;\n";
  str += spaces(l - 2) + "lab" + i_to_string(windex + 1) + ":;\n";
  // Emit code to test condition 
  if (ws->condition_clause != NULL) {
    str += spaces(l);
    if (ws->timeout_clause != NULL) 
      str += "if ((!timeout_jmp)&&(";
    else
      str += "if (!(";
    emit_expr(ws->condition_clause, str, rstack, id_type(READER, DEREF)); 
    str += ")) return false;\n";
  }
}

void
m_emit_impl (pIIR_AssertionStatement as, string &str, RegionStack &rstack, int l)
{
  str += "assert(";
  emit (as->assertion_condition, str, rstack, DEFAULT);
  str += "); /* ";
  emit (as->severity_expression, str, rstack, DEFAULT);
  str += " ";
  emit (as->report_expression, str, rstack, DEFAULT);
  str += " */\n";
}

void
m_emit_impl (pIIR_ReportStatement rs, string &str, RegionStack &rstack, int l)
{
  str += "fputs (";
  emit (rs->report_expression, str, rstack, DEFAULT);
  str += ", stderr); /* ";
  emit (rs->severity_expression, str, rstack, DEFAULT);
  str += " */\n";
}

void
m_emit_impl (pIIR_LoopControlStatement lcs, string &str, RegionStack &rstack, int indent)
{
  // Controlling the loop. For innermost loop, `break' and
  // `continue' are used directly. Outer loops are handled with
  // gotos. The needed labels are not yet emitted, though.

  pIIR_LoopStatement l = lcs->loop;
  if (lcs->condition) 
    {
      str += "if(";
      emit (lcs->condition, str, rstack, DEFAULT);
      str += ") ";
    }

  string stat = lcs->is(IR_EXIT_STATEMENT)? "break" : "continue";
  if (l && l->label)
    {
      str += "goto ";
      emit_id (l->label->declarator, str);
      str += "_" + stat + ";\n";
    }
  else
    str += stat + ";\n";
}


void
m_emit_impl (pIIR_SignalAssignmentStatement sa, string &str, RegionStack &rstack, int l)
{
  // determine the signals that are written!
  ContextInfo local_ctxt;
  ContextInfo &ctxt = *ActiveContext(rstack);

  get_context(sa->target, local_ctxt, rstack, true, 0);
  list<AccessDescriptor> sig_written = 
    filter(local_ctxt.accessed_objects, WRITE, 
	   tree_kind_list(IR_SIGNAL_DECLARATION, IR_SIGNAL_INTERFACE_DECLARATION));

  // bail out if more than one signal is written. Only simple signal
  // assignment targets are supported yet.
  if (sig_written.size() > 1)
    codegen_error.error("%:error: sorry, no resolved signals are supproted yet", sa);
    
  // Get decelration of target signal
  pIIR_SignalDeclaration target_sig_decl = (pIIR_SignalDeclaration)sig_written.front().declaration;

  // Determine the first part of the synthesized signal assignment statement
  string target_str, target_specifier;

  if (is_scalar_type(target_sig_decl->subtype)) {
    //******************************************************************
    // if the target is a scalar signal then simply get the driver for
    // that signal
    //******************************************************************
    target_str = qid(target_sig_decl, "_", id_type(DRIVER));

  } else if (is_scalar_type(sa->waveform->first->value->subtype)) {
    //******************************************************************
    // if the target is a *scalar* element of a composite signal then
    // get the scalar driver for that element
    //******************************************************************
    list<string>acl_list;
    get_acl(sig_written.front().object_ref, acl_list, rstack, IR_NOT_STATIC, id_type(DEFAULT, DEREF), true);
    // generate a name for a new acl object
    string internal_var_name = create_internal_acl(acl_list, rstack, false);
    // Create signal assignment code
    target_str = qid(target_sig_decl, "_", id_type(DRIVER)) + 
      "->scalar_driver(&(" + sprint_acl(acl_list, internal_var_name + "->clear()") + "))";

  } else {
    //******************************************************************
    // Several elements of a composite signal are target of the
    // assignment operation. First, get acl for assignment target
    //******************************************************************
    list<string>acl_list;
    get_acl(sig_written.front().object_ref, acl_list, rstack, IR_NOT_STATIC, id_type(DEFAULT, DEREF), true);

    // Check whether the entrire signal or a only a part of the signal
    // is target of the assignement
    if (acl_list.size()) {
      // Only a part of the signal is targeted. First, generate a name
      // for a new acl object
      string internal_var_name = create_internal_acl(acl_list, rstack, false);
      // Create signal assignment code
      target_str = qid(target_sig_decl, "_", id_type(DRIVER));
      target_specifier = "," + qid(target_sig_decl, "_", id_type(READER)) + 
	"->info->acl_to_index(&(" + sprint_acl(acl_list, internal_var_name + "->clear()") + "))";
      
    } else {
      // the *entire* signal is assignement target
      target_str = qid(target_sig_decl, "_", id_type(DRIVER));
      target_specifier = ",0";
    }
  }

  bool first = true;
  for (pIIR_WaveformList wl = sa->waveform; wl; wl = wl->rest)
    {
      strstream lstr;
      if (!first)
        str += spaces(l+1);
      str += target_str;
      if (sa->delay_mechanism==IR_INERTIAL_DELAY && first)
	str += "->inertial_assign(";
      else
	str += "->transport_assign(";

      pIIR_WaveformElement we = wl->first;
      string value_str;
      emit_expr (we->value, value_str, rstack, id_type(READER, DEREF));
      value_str += target_specifier;
      if (we->time) {
	value_str += ",time(";
	emit_expr (we->time, value_str, rstack, id_type(READER));     
      } else 
	value_str += ",time(0";

      str += value_str + "));\n";
      first = false;
    }
}


void
m_emit_impl (pIIR_SequentialStatement s, string &str, RegionStack &rstack, int l)
{
  str += string("/* emit statement ") + s->kind_name () + string(" */\n");
}


void
m_emit_impl (pIIR_ComponentInstantiationStatement ci, string &str, RegionStack &rstack, int l)
{
  if (ci->configuration_specification) 
    {
      str += ": ";
      emit (ci->configuration_specification, str, rstack, 0);
    }
}

void
m_emit_impl (pIIR_BlockStatement bs, string &str, RegionStack &rstack, int l)
{
  str += "block ";
  str += string("\n") + spaces(l) + string("{\n");
  emit_decls (bs->declarations, str, rstack, l+2);
  if (bs->declarations)
    str += "\n";
  emit (bs->block_statement_part, str, rstack, l+2);
  str += spaces(l) + string("}\n\n");
}

void
m_emit_impl (pIIR_ConcurrentGenerateStatement gs, string &str, RegionStack &rstack, int l)
{
  str += "generate ";
  if (gs->is(IR_CONCURRENT_GENERATE_IF_STATEMENT))
    {
      str += "if ";
      emit (pIIR_ConcurrentGenerateIfStatement(gs)->condition, str, rstack, DEFAULT);
    }
  else if (gs->is(IR_CONCURRENT_GENERATE_FOR_STATEMENT))
    {
      pIIR_ConcurrentGenerateForStatement f =
	pIIR_ConcurrentGenerateForStatement (gs);
      str += "for (";
      emit (f->generate_parameter_specification->subtype, str, rstack, 0);
      str += " ";
      emit_id (f->generate_parameter_specification, str);
      str += ")";
    }
  str += " {\n";
  emit (gs->concurrent_statement_part, str, rstack, l+2);
  str += spaces(l) + "}\n";
}

void
m_emit_impl (pIIR_ConcurrentStatement cs, string &str, RegionStack &rstack, int l)
{
  str += string("/* ") + string(cs->kind_name()) + string(" */\n");
}

/*
 * v2c Leftovers
 */

void
emit (pIIR_ConfigurationSpecification s, string &str, RegionStack &rstack)
{
}

void
emit (pIIR_ConcurrentStatementList s, string &str, RegionStack &rstack, int l)
{
  while (s)
    {
      str += spaces(l);
      emit_impl (s->first, str, rstack, l);
      s = s->rest;
    }
}

/*
 * Utility
 */

void
emit_includes(string &str)
{
  str += string ("#include <freehdl/kernel.h>\n") +
         string ("#include <freehdl/std.h>\n\n\n");
}



// ******************************************************************************************
// Name: get_library_unit
//
// Description: returns library unit a declarative region belongs to
//
// Parameter: pIIR_DeclarativeRegion d = declarative region
// 
// Return value: library unit
//
// ******************************************************************************************

pIIR_LibraryUnit
get_library_unit(pIIR_DeclarativeRegion d)
{
  while (d != NULL && !d->is(IR_LIBRARY_UNIT)) {
    d = d->declarative_region;
  }
    
  return (pIIR_LibraryUnit)d;
}

// returns name of the libraray
char *
get_lib_name(pIIR_DeclarativeRegion r)
{
  pIIR_LibraryUnit lu = get_library_unit(r); 

  return lu->library_name->text.to_chars();
}


// ******************************************************************************************
// Name: m_get_operator_type (generic function)
//
// Description: returns type of an operator. Possible values are:
//   - NO_OP if the function does not denote an operator
//   - STD_OP if the funcion denotes an predefined VHDL operator
//   - BASIC_OP if the function denotes an operator defined in an 
//           IEEE library (e.g. numeric_std)
//   - USER_OP an user defined operator
//
// Parameter: pIIR_FunctionDeclaration f = pointer to function declaration
// 
// Return value: op_type
//
// ******************************************************************************************

op_type
m_get_operator_type(pIIR_FunctionDeclaration fd)
{
  char *function_str = fd->declarator->text.to_chars(); // Get function name

  if (function_str[0] != '"') // Is function not an operator?
    return NO_OP;

  char *libname = get_lib_name(fd->declarative_region);

  if (!strcmp(libname, "std"))
    // Function is defined in library "std"
    return STD_OP;
  else if (!strcmp(libname, "ieee"))
    // Function is defined in library "ieee"
    return BASIC_OP;
  else
    return USER_OP;
}



// ******************************************************************************************
// Name: m_get_access_string ,(generic function)
//
// Description: return the c++ acces string is used in the generated code to access
//   the objects which were generated from the corresponding VHDL source. 
//
// Parameter: Pointer to corresponding declaration.
// 
// Return value: string&
//
// ******************************************************************************************

string&
m_get_access_string(pIIR_FunctionDeclaration fd)
{
  // First test whether access is set. If yes then the access string can be
  // return directly...
  if (access(fd) != "") 
    return access(fd);

  // Otherwise the complete access name must be determined
  string str;
  op_type op = get_operator_type(fd); // Get type of function. Perhaps it's an operator.

  if (op == NO_OP) { // fd is a function call (no operator)
      access(fd) = qid(fd);

  } else { // fd denotes an operator
    // count the arguments
    int n_args = 0;
    for (pIIR_InterfaceList idec = fd->interface_declarations; idec; idec = idec->rest)
      n_args++;

    switch (op) {
    case STD_OP: // fd is a predefined operator
      break;
    case BASIC_OP: // fd is a operator defined in the library IEEE
    case USER_OP: // fd denotes a user defined operator
      access(fd) = qid(fd);
      break;
    }
  }
  
  return access(fd); // Return access string
}




// ******************************************************************************************
// Name: m_get_acl , generic function
//
// Description: Extracts ACL list from an expression. Depending on the
// parameter slevel only the locally static, globally static or all
// parts of the expression are converted and added to the ACL list. t
// controls how VHDL objects are tanslated into C++ objects. start is
// true when the entire expression e shall be converted into an ACL
// object.
//
// Parameter: slevel = static level of the expression which shall be considered.
//   alist = stores the ACL string items
//   t = Controls conversion of VHDL objects into C++ objects
//   start = false if the ENTIRE expression shall be converted into an ACL
// 
// Return value: returns true if e matches the static level slevel
//
// ******************************************************************************************

bool
m_get_acl(pIIR_Expression e, 
	  list<string>& alist, RegionStack &rstack, IR_StaticLevel slevel, id_type t = id_type(), bool start = false)
{
  bool match = level_match(e->static_level, slevel);
  if (start) return match;

  // first check whether slevel matches the static level of the
  // expression e
  if (! match) return false;

  string str;
  emit_expr(e, str, rstack, t);
  alist.push_back(str);
  
  return true; // e matches the static level slevel
}


bool
m_get_acl (pIIR_SimpleReference sor, 
	   list<string>& alist, RegionStack &rstack, IR_StaticLevel slevel, id_type t = id_type(), bool start = false)
{
  return start ? true : level_match(sor->static_level, slevel);
}


bool
m_get_acl (pIIR_ArrayReference aor,
	   list<string>& alist, RegionStack &rstack, IR_StaticLevel slevel, id_type t = id_type(), bool start = false)
{
  if (start) {
    // if start is true then first calculate the ACL for the prefix
    bool match = get_acl(aor->array, alist, rstack, slevel, t, true);

    // Now test whether each index value matches the static level
    // slevel. Further, convert the index values into ACL objects.
    for (pIIR_ExpressionList il = aor->indices; match && il; il = il->rest)
      {
	// Stop creating ACL objects as soon as the first index is
	// found which does not match the static level slevel
	if (! level_match(il->first->static_level, slevel)) {
	  match = false;  
	  break; 
	}
	string str;
	bool simple = emit_expr (il->first, str, rstack, t);

	alist.push_back(str);
      }
    
    return match;
    
  } else {
    // if start is false then convert the entire expression into an
    // ACL object if the slevel matches the static level of the
    // expression aor.
    if (! level_match(aor->static_level, slevel)) return false;
    string str;
    emit_expr (aor, str, rstack, t);
    return true;
  }
}


bool
m_get_acl (pIIR_RecordReference ror, 
	   list<string>& alist, RegionStack &rstack, IR_StaticLevel slevel, id_type t = id_type(), bool start = false)
{
  if (start) {
    // if start is true then calculate the ACL for the prefix
    if (get_acl(ror->record, alist, rstack, slevel, t, start)) {
      // if the prefix is static then determine the decalration number
      // of the record element (the first element is associated with
      // number 0, the next element is associated with 1, ...)
      strstream lstr;
      lstr << ror->element->declaration_pos;
      string str(lstr.str());
      alist.push_back(str);

      return true;
      
    } else
      // if the prefix is not static then return false. Do not append
      // the declaration number!
      return false;

  } else {
    // if start is false then convert the entire expression into an
    // ACL object if the slevel matches the static level of the
    // expression ror
    if (! level_match(ror->static_level, slevel)) return false;
    string str;
    emit_expr(ror, str, rstack, t);
    return true;
  }
}



bool
m_get_acl (pIIR_ExplicitRange er, 
	   list<string>& alist, RegionStack &rstack, IR_StaticLevel slevel, id_type t = id_type(), bool start = false)
{
  // bail out if start is true
  if (start) internal_error(true); 

  // check whether left and right border match the static level
  if (! level_match(er->left->static_level, slevel) ||
      ! level_match(er->right->static_level, slevel))
    return false;

  // append marker to ACL list
  alist.push_back("ACL_RANGE");

  // now add left border, direction and right border
  string left, right, direction;
  bool simple = emit_expr(er->left, left, rstack, t);
  alist.push_back(left);

  direction = er->direction == IR_DIRECTION_UP ? "(int)to" : "(int)downto";
  alist.push_back(direction);

  simple = emit_expr(er->right, right, rstack, t);
  alist.push_back(right);

  return true;
}



bool
m_get_acl (pIIR_SliceReference sr, list<string>& alist, 
	   RegionStack &rstack, IR_StaticLevel slevel, id_type t = id_type(), bool start = false)
{
  if (start) {
    // if start is true then calculate the ACL for the prefix as well
    // as the ACL for the range
    if (get_acl(sr->array, alist, rstack, slevel, t, true) &&
	get_acl(sr->range, alist, rstack, slevel, t, false))
      return true;
    else
      return false;
    
  } else {
    // if start is false then convert the entire expression into an
    // ACL object if the slevel matches the static level of the
    // expression ror
    if (! level_match(sr->static_level, slevel)) return false;
    string str;
    emit_expr(sr, str, rstack, t);
    return true;
  }
}

