
// constants to mark how objects are accessed
#define READ       0x1
#define WRITE      0x2
#define SENSITIVE  0x4
#define ATTRIBUTE  0x8 // Attribute has been applied on object
#define ACCESS     (READ|WRITE|SENSITIVE|ATTRIBUTE)

// Optimization stages
#define CONST_FOLD 0x1


// Max und min integer values
#define INTEGER_MIN (INT_MIN + 1)
#define INTEGER_MAX (INT_MAX)

typedef long long int lint; // To save some typing

inline int 
min(const int a, const int b) { return a < b? a : b; }

inline int 
max(const int a, const int b) { return a > b? a : b; }


// Union which either stores an integer or an double value
union scalar_data {
  long long int long_value;
  double double_value;
};


// a AccessDescriptor instance stored information about the objects
// that are accessed within the VHDL code. 
struct AccessDescriptor {
  pIIR_ObjectDeclaration declaration; // points to the declaration of the object
  pIIR_ObjectReference object_ref; // points to the part of the
  // expression which references the object
  int access_type; // stores whether the object is read, written or sensed
  int level; // level the object is located at. The entire statement
  // is assigned level 0. The level is increased each time an object is
  // included in an argument to a subroutine or an index value of an
  // array reference.

  AccessDescriptor() {
    declaration = NULL;
    object_ref = NULL;
    access_type = 0;
    level = 0;
  }

  AccessDescriptor(pIIR_ObjectDeclaration decl, pIIR_ObjectReference oref, 
		   int at = READ, int l = 0) {
    declaration = decl;
    object_ref = oref;
    access_type = at;
    level = l;
  };

  AccessDescriptor(const AccessDescriptor &sacs) {
    declaration = sacs.declaration;
    object_ref = sacs.object_ref;
    access_type = sacs.access_type;
    level = sacs.level;
  };

  AccessDescriptor &operator=(const AccessDescriptor &sacs) {
    declaration = sacs.declaration;
    object_ref = sacs.object_ref;
    access_type = sacs.access_type;
    level = sacs.level;
    return *this;
  };
};


typedef list<AccessDescriptor> access_list;
typedef AccessDescriptor *pAccessDescriptor;

// Stores a list of tree_kind values
class tree_kind_list : public vector<tree_kind> {
 public:
  tree_kind_list() { };
  tree_kind_list(const tree_kind a) { push_back(a); }
  tree_kind_list(const tree_kind a, const tree_kind b) { push_back(a); push_back(b); }
  tree_kind_list operator<<(const tree_kind a) { push_back(a); }
  bool matches(const pIIR_ObjectDeclaration decl) const { 
    for (const_iterator iter = begin(); iter != end(); iter++) 
      if (decl->is(*iter)) return true;
    return false;
  }
};

// function generates a new list from an access_list which only
// includes items that matche the access_type and the
// object_kind. Example: filter(lst, WRITE, IR_SIGNAL_DECLARATION)
// will create a new list from lst where only written signals are
// inlcuded.
inline list<AccessDescriptor>
filter(list<AccessDescriptor> &lst, int access_type, const tree_kind_list &object_kind)
{
  list<AccessDescriptor> result;

  for (list<AccessDescriptor>::iterator it = lst.begin(); it != lst.end(); it++)
    if (((*it).access_type & access_type) && 
	object_kind.matches((*it).declaration))
      result.push_back(*it);

  return result;
}


// same as filter besides that no items with the same member
// "declaration" will appear more than once on the list
inline list<AccessDescriptor>
filter_unique(list<AccessDescriptor> &lst, int access_type, const tree_kind_list &object_kind)
{
  list<AccessDescriptor> result;
  set<pIIR_ObjectDeclaration, less<void*> > obj_set;

  for (list<AccessDescriptor>::iterator it = lst.begin(); it != lst.end(); it++)
    if (((*it).access_type & access_type) && 
	object_kind.matches((*it).declaration) &&
	obj_set.find((*it).declaration) == obj_set.end()) {
      result.push_back(*it);      
      obj_set.insert((*it).declaration);
    }

  return result;
}



typedef map<string, string, less<string> > map_string2;


// Some constants to control handling of internal variables generated
// by the code generator
#define DECLARE_AND_INIT 1 // The object shall be declared and directly initialized
#define DECLARE_GLOBAL 2 // The items should be declared as a global object


// A "ContextInfo" object is used to store data required by the code
// generator
struct ContextInfo {
  list<AccessDescriptor> accessed_objects; // accessed objects
  list<pIIR_WaitStatement> wait_statements; // wait statements
  list<pIIR_SignalAssignmentStatement> signal_assignment_statements; // signal assignements
  void clear() {
    accessed_objects.clear();
    wait_statements.clear();
    signal_assignment_statements.clear();
  }
};

typedef ContextInfo *pContextInfo;


struct RegionStack : public deque<pIIR_DeclarativeRegion> {
  void push(pIIR_DeclarativeRegion r) { push_back(r); }
  void pop() { pop_back(); }
};


inline char *
to_chars(const AccessDescriptor &a) {
  return a.declaration->declarator->text.to_chars();
}


// class to store range information. Range information is either given
// as an "explicit" range by specifying left border, right border and
// direction or derived from an array instance via the attributes
// RANGE or REVERSE_RANGE.
class RangeDescriptor {
 public:
  // Members used to store "explicit" ranges
  pIIR_Root left, right;
  IR_Direction direction;
  // Member stores an array attribtute range expression. If this
  // pointer is not equal to NULL then an range attribute expression
  // is stored by the current range descriptor.
  pIIR_ArrayRange range_attribute;
  // Static level of the instance
  IR_StaticLevel static_level;

  RangeDescriptor() { 
    left = right = NULL; direction = IR_DIRECTION_UP; 
    range_attribute = NULL; 
    static_level = IR_NOT_STATIC; 
  }
  // Create range descriptor from "explitit" range
  RangeDescriptor(pIIR_Root l, IR_Direction d, pIIR_Root r, IR_StaticLevel slevel) { 
    left = l; right = r; direction = d; static_level = slevel; 
    range_attribute = NULL; // Flag explicit range
  }
  // Create range decriptor from 
  RangeDescriptor(const RangeDescriptor &r) { 
    left = r.left; right = r.right; direction = r.direction; static_level = r.static_level;
    range_attribute = r.range_attribute;
  }
  // Destructor. Nothing to do.
  ~RangeDescriptor() { };
  // Returns true if range r and the current instance cover the same
  // range
  bool is_equal_to(const RangeDescriptor &r);
  // Returns true if range is "explicit"
  bool is_explicit_range() { return left !=  NULL; }
  // Convert a RangeDescriptor into corresponding integer values and the
  // direction. True is returned when the conversion could be done,
  // i.e. when the bounds were locally static. Otherwise false is
  // returned
  bool rangedes_to_int(int &left, IR_Direction &dir, int &right, RegionStack &rstack);
  // Convert a RangeDescriptor into corresponding integer value strings
  // and the direction strings. If a bound is not locally static then
  // appropriate code is emitted to extract the corresponding integer
  // value from the object.
  void rangedes_to_int_string(string &left, string &dir, string &right, RegionStack &rstack);
  // Returns Attribute node
  pIIR_ArrayRange get_attribute_range() { return range_attribute; };
  // Constant fold
  int constant_fold_rangedes(RegionStack &rstack);
};




// id_type is used to control the output of the qid and nid function.
#define DEFAULT 0x1
#define SIGNAL 0x2 
#define READER 0x4
#define DRIVER 0x8
#define TYPE 0x10
#define INFO 0x20
#define BARE 0x40
#define LIBRARY 0x80
#define ALIAS 0x100

#define ARCHREF 0x2  // object access shall look like "arch->..."
#define DEREF 0x4  // object access shall look like "*..."

struct id_type {
  int obj;
  int acc;
  id_type(int o = DEFAULT, int a = DEFAULT) {
    obj = o;
    acc = a;
  }
  int object() const { return obj; }
  int access() const { return acc; }
};



// Size of various VHDL data types in bytes
#define ENUM_SIZE 1


// Some name constants
extern const string internal_prefix_start;
extern const string process_default_postfix;
extern const string loop_default_postfix;


// Some globally defined functions

extern int constant_fold(RangeDescriptor &r, RegionStack &rstack);
