/* blocks, components and bindings

   Copyright (C) 1994-1997 University of Dortmund
   Department of Electrical Engineering, AG SIV

   VAUL is free software; you can redistribute it and/or modify it
   under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   VAUL is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General
   Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with VAUL; see the file COPYING.LIB.  If not, write
   to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307 USA.


*/

#include <freehdl/vaul-parser.h>
#include <freehdl/vaul-chunk.h>
#include <freehdl/vaul-dunit.h>
#include <freehdl/vaul-util.h>
#include "vaulgens-chunk.h"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#define psr vaul_parser

pVAUL_BindingIndic
psr::build_BindingIndic (pVAUL_IncrementalBindingIndic ibi)
{
  return build_BindingIndic (ibi->unit, ibi->generic_assoc, ibi->port_assoc);
}

pVAUL_BindingIndic
psr::build_BindingIndic (pIIR_Component unit,
			 pVAUL_NamedAssocElem gm,
			 pVAUL_NamedAssocElem pm)
{
  if (unit == NULL)
    return NULL;

  pIIR_AssociationList g = associate (gm, get_generics (unit), false);
  pIIR_AssociationList p = associate (pm, get_ports (unit), false);
  return mVAUL_BindingIndic ((IIR_PosInfo *)NULL, unit, g, p);
}

void
add_spec (pIIR_BlockStatement b, pVAUL_ConfigSpec cs)
{
  pVAUL_ConfigSpec *csp;
  for(csp = &config_specs(b); *csp; csp = &(*csp)->next)
    ;
  *csp = cs;
}

#if 0
void
add_disconnect_spec (nBlock b, nDisconnectSpec ds)
{
  nDisconnectSpec *dsp;
  for(dsp = &b->disconnect_specs; *dsp; dsp = &(*dsp)->next)
    ;
  *dsp = ds;
}
#endif

pIIR_ComponentInstantiationStatement
psr::build_CompInst (int lineno, pVAUL_BindingIndic bi)
{
  if (bi == NULL)
    return NULL;

  return mIIR_ComponentInstantiationStatement (lineno,
					       NULL,
					       bi->unit,
					       bi->generic_assoc,
					       bi->port_assoc,
					       NULL);
}

pIIR_ComponentInstantiationStatement
psr::build_CompInst (pIIR_PosInfo pos, pVAUL_BindingIndic bi)
{
  if (bi == NULL)
    return NULL;

  return mIIR_ComponentInstantiationStatement (pos,
					       NULL,
					       bi->unit,
					       bi->generic_assoc,
					       bi->port_assoc,
					       NULL);
}

void
psr::bind_specs (pIIR_BlockStatement b)
{
  for (pVAUL_ConfigSpec cs = config_specs(b); cs; cs = cs->next)
    {
      if (cs->comps == NULL)
	continue;

      pIIR_ConfigurationSpecification cf =
	mIIR_ConfigurationSpecification(cs->pos, NULL,
					cs->binding->unit,
					cs->binding->generic_assoc,
					cs->binding->port_assoc);
      bound_by_name(cf) = cs->comps->ids->is(VAUL_INST_LIST_IDS);

      add_decl_plain (b, cf);

      for (pIIR_ComponentInstantiationList cil =
	     list_comps (cs->comps, b); cil; cil = cil->rest)
	{
	  if (cil->first && cil->first->configuration_specification)
	    {
	      error ("%:%n is already configured by..", cs, cil->first);
	      info ("%:..this configuration specification",
		    cil->first->configuration_specification);
	    }
	  else if (cil->first)
	    cil->first->configuration_specification = cf;
	}
    }
}

static void
add (pIIR_BlockConfiguration b, pIIR_ConfigurationItemList c)
{
  pIIR_ConfigurationItemList cp;
  for (cp = b->configuration_item_list; cp->rest; cp = cp->rest)
    ;
  cp->rest = c;
}

pIIR_ArchitectureRef
psr::get_architecture_ref (int lineno, pVAUL_Name en, pIIR_Identifier arch)
{
  assert(en != NULL);

  pIIR_EntityDeclaration entity =
    pIIR_EntityDeclaration(find_single_decl (en, IR_ENTITY_DECLARATION,
					     "entity"));
  if (entity == NULL)
    return NULL;

  return get_architecture_ref (entity, mVAUL_SimpleName(lineno, arch));
}

pIIR_ArchitectureRef
psr::get_architecture_ref (pIIR_EntityDeclaration entity, pVAUL_SimpleName arch)
{
  return mIIR_ArchitectureRef(arch->pos, NULL, entity, arch->id);
}

pIIR_BlockConfiguration
psr::start_BlockConfig (pVAUL_Name n)
{
  assert (cur_scope);

  pIIR_Component block = NULL;
  
  if (!n->is(VAUL_SIMPLE_NAME))
    error ("%:block specification must be a simple name", n);
  else if (cur_scope->is(IR_CONFIGURATION_DECLARATION))
    block = get_architecture_ref (pIIR_ConfigurationDeclaration(cur_scope)->entity,
				  pVAUL_SimpleName(n));
  else if (cur_scope->is(IR_BLOCK_CONFIGURATION))
    block = pIIR_BlockStatement (find_single_decl (n, IR_BLOCK_STATEMENT,
						   "block statement"));
  else if (cur_scope->is(IR_COMPONENT_CONFIGURATION))
    info ("XXX - no component config");
  
  pIIR_BlockConfiguration bc = mIIR_BlockConfiguration (n->pos, NULL,
							block,
							NULL, NULL);
  bc->continued = block;
  
  if (cur_scope->is(IR_BLOCK_CONFIGURATION))
    add (pIIR_BlockConfiguration(cur_scope),
	 mIIR_ConfigurationItemList (bc->pos, bc, NULL));
  
  add_decl (bc);
  push_scope (bc);
  return bc;
}

static pIIR_ComponentConfiguration
get_CompConfig (pIIR_BlockConfiguration bc,
		pIIR_ComponentInstantiationStatement ci)
{
  for (pIIR_ConfigurationItemList cl = bc->configuration_item_list;
       cl; cl = cl->rest)
    {
      pIIR_ConfigurationItem c = cl->first;
      if (!c->is(IR_COMPONENT_CONFIGURATION))
	continue;
      pIIR_ComponentConfiguration cc = pIIR_ComponentConfiguration(c);
      for (pIIR_ComponentInstantiationList cil = cc->instantiation_list;
	   cil; cil = cil->rest)
	if (cil->first == ci)
	  return cc;
    }

  return NULL;
}

static bool
bound_by_name (pIIR_ComponentInstantiationStatement ci,
	       pIIR_BlockConfiguration bc)
{
  pVAUL_ComponentSpec s = NULL;
  if (bc)
    {
      pIIR_ComponentConfiguration cc = get_CompConfig(bc, ci);
      return cc && bound_by_name (cc);
    }
  else
    return (ci->configuration_specification &&
	    bound_by_name (ci->configuration_specification));
}

pIIR_ComponentInstantiationList
psr::list_comps (pVAUL_ComponentSpec c, pIIR_DeclarativeRegion s)
{
  if (c == NULL || c->ids == NULL || s == NULL)
    return NULL;

  pIIR_ComponentInstantiationList l = NULL;

  if (c->ids->is(VAUL_INST_LIST_IDS))
    {
      for (pVAUL_IdList idl = pVAUL_InstList_Ids(c->ids)->ids; idl; idl = idl->link) 
	{
	  pVAUL_SimpleName sn = mVAUL_SimpleName (idl->pos, idl->id);
	  pIIR_ComponentInstantiationStatement ci = 
	    pIIR_ComponentInstantiationStatement
	    (find_single_decl(sn, IR_COMPONENT_INSTANTIATION_STATEMENT,
			      "component instantiation"));

	  if (ci && ci->instantiated_unit && c->comp &&
	      ci->instantiated_unit != c->comp)
	    {
	      error ("%:configured unit %n does not match"
		     " instantiated unit %n", idl, c->comp,
		     ci->instantiated_unit);
	    }
	  else
	    l = mIIR_ComponentInstantiationList (ci->pos, ci, l);
	}
    }
  else if (c->ids->is(VAUL_INST_LIST_ALL) || c->ids->is(VAUL_INST_LIST_OTHERS))
    {
      pIIR_Component b;
      pIIR_BlockConfiguration bc;
      if (s->is(IR_BLOCK_CONFIGURATION))
	{
	  bc = pIIR_BlockConfiguration(s);
	  b = bc->block_specification;
	  if (b == NULL)
	    return NULL;
	}
      else if (s->is(IR_BLOCK_STATEMENT))
	{
	  bc = NULL;
	  b = pIIR_BlockStatement(s);
	} 
      else
	return NULL;
      
      for (pIIR_ConcurrentStatementList csl = get_stats (b);
	   csl; csl = csl->rest) 
	{
	  pIIR_ConcurrentStatement cs = csl->first;
	  if (!cs->is(IR_COMPONENT_INSTANTIATION_STATEMENT))
	    continue;
	  pIIR_ComponentInstantiationStatement ci =
	    pIIR_ComponentInstantiationStatement(cs);
	  if (ci->instantiated_unit == c->comp
	      && !(c->ids->is(VAUL_INST_LIST_OTHERS) && bound_by_name(ci, bc)))
	    l = mIIR_ComponentInstantiationList (ci->pos, ci, l);
	}
    }
  
  return l;
}

static inline pIIR_Component
get_bound_unit (pIIR_ComponentInstantiationStatement ci)
{
  if (ci && ci->configuration_specification)
    return ci->configuration_specification->unit;
  return NULL;
}

pIIR_ComponentConfiguration
psr::start_CompConfig (int lno,
		       pVAUL_ComponentSpec c,
		       pVAUL_IncrementalBindingIndic ib)
{
    // XXX - do lots of checking

  if (c == NULL)
    return NULL;

  pIIR_ComponentConfiguration cc =
    mIIR_ComponentConfiguration (lno, NULL, NULL, 
				 list_comps (c, cur_scope), NULL, NULL, NULL);
  bound_by_name(cc) = c->ids->is(VAUL_INST_LIST_IDS);

  if (ib && ib->unit == NULL)
    {
      for (pIIR_ComponentInstantiationList cil = cc->instantiation_list;
	   cil; cil = cil->rest)
	{
	  pIIR_Component u = get_bound_unit(cil->first);
	  if (u)
	    {
	      if (ib->unit && ib->unit != u)
		info ("XXX - incrementally bound component instantiations"
		      " must all be bound to the same unit");
	      else
		ib->unit = u;
	    }
	}
    }

  if (ib)
    {
      pVAUL_BindingIndic b = build_BindingIndic (ib);
      assert (b->unit->is(IR_LIBRARY_UNIT));
      cc->entity_aspect = pIIR_LibraryUnit(b->unit);
      cc->generic_map_aspect = b->generic_assoc;
      cc->port_map_aspect = b->port_assoc;
    }

  if (cur_scope->is(IR_BLOCK_CONFIGURATION))
    add (pIIR_BlockConfiguration(cur_scope),
	 mIIR_ConfigurationItemList (cc->pos, cc, NULL));

  add_decl (cc);
  push_scope (cc);
  return cc;
}

void
psr::check_BlockConfig (pIIR_BlockConfiguration bc)
{
  // see if any component instantiation is bound more than once
  //
  for (pIIR_ConfigurationItemList c = bc->configuration_item_list;
       c; c = c->rest)
    {
      if (!c->is(IR_COMPONENT_CONFIGURATION))
	continue;
      pIIR_ComponentConfiguration cc = pIIR_ComponentConfiguration(c);
      for (pIIR_ComponentInstantiationList cil = cc->instantiation_list;
	   cil; cil = cil->rest)
	{
	  pIIR_ComponentConfiguration cc2 = get_CompConfig(bc, cil->first);
	  if (cc2 != cc)
	    {
	      error("%:%n is already configured by..", cc, cil->first);
	      info("%:..this component configuration", cc2);
	    }
	}
    }
}

void
psr::add_disconnect_spec (pIIR_ExpressionList sl, pVAUL_Name mark,
			  pIIR_Expression after)
{
  pIIR_Type t = get_type (mark);
  overload_resolution (after, std->predef_TIME);

  if (t == NULL || after == NULL)
    return;

  pIIR_Type bt = get_base (t);

  while (sl)
    {
      if (sl->first)
	{
	  if (get_base (get_type (sl->first)) != bt)
	    error ("%:%n does not match type %n", sl->first, sl->first, bt);
	  else
	    {
	      pIIR_ObjectReference s = pIIR_ObjectReference(sl->first);
	      assert (s->is(IR_OBJECT_REFERENCE));
	      pIIR_DisconnectSpecification ds =
		mIIR_DisconnectSpecification (sl->pos, NULL, t, after, s);
	      add_decl_plain (cur_scope, ds);
	    }
	}
      sl = sl->rest;
    }
}

pIIR_InterfaceList
m_get_generics (pIIR_ComponentDeclaration d)
{
  return d->local_generic_clause;
}

pIIR_InterfaceList
m_get_generics (pIIR_EntityDeclaration d)
{
  return d->generic_clause;
}

pIIR_InterfaceList
m_get_generics (pIIR_ArchitectureDeclaration d)
{
  return d->entity? d->entity->generic_clause : NULL;
}

pIIR_InterfaceList
m_get_generics (pIIR_ArchitectureRef d)
{
  return d->entity? d->entity->generic_clause : NULL;
}

pIIR_InterfaceList
m_get_generics (pIIR_BlockStatement bs)
{
  return bs->generic_clause;
}

pIIR_InterfaceList
m_get_ports (pIIR_ComponentDeclaration d)
{
  return d->local_port_clause;
}

pIIR_InterfaceList
m_get_ports (pIIR_EntityDeclaration d)
{
  return d->port_clause;
}

pIIR_InterfaceList
m_get_ports (pIIR_ArchitectureDeclaration d)
{
  return d->entity? d->entity->port_clause : NULL;
}

pIIR_InterfaceList
m_get_ports (pIIR_ArchitectureRef d)
{
  return d->entity? d->entity->port_clause : NULL;
}

pIIR_InterfaceList
m_get_ports (pIIR_BlockStatement bs)
{
  return bs->port_clause;
}

void
m_set_generics (pIIR_ComponentDeclaration d, pIIR_InterfaceList generics)
{
  d->local_generic_clause = generics;
}

void
m_set_generics (pIIR_EntityDeclaration d, pIIR_InterfaceList generics)
{
  d->generic_clause = generics;
}

void
m_set_generics (pIIR_BlockStatement bs, pIIR_InterfaceList generics)
{
  bs->generic_clause = generics;
}

void
m_set_ports (pIIR_ComponentDeclaration d, pIIR_InterfaceList ports)
{
  d->local_port_clause = ports;
}

void
m_set_ports (pIIR_EntityDeclaration d, pIIR_InterfaceList ports)
{
  d->port_clause = ports;
}

void
m_set_ports (pIIR_BlockStatement bs, pIIR_InterfaceList ports)
{
  bs->port_clause = ports;
}

pIIR_ConcurrentStatementList
m_get_stats (pIIR_ComponentDeclaration d)
{
  return NULL;
}

pIIR_ConcurrentStatementList
m_get_stats (pIIR_EntityDeclaration d)
{
  return d->entity_statement_part;
}

pIIR_ConcurrentStatementList
m_get_stats (pIIR_ArchitectureDeclaration d)
{
  return d->architecture_statement_part;
}

pIIR_ConcurrentStatementList
m_get_stats (pIIR_BlockStatement bs)
{
  return bs->block_statement_part;
}

pIIR_ConcurrentStatementList
m_get_stats (pIIR_ConfigurationDeclaration d)
{
  return NULL;
}
