/* types and subtypes

   Copyright (C) 1994-1997 University of Dortmund
   Department of Electrical Engineering, AG SIV

   VAUL is free software; you can redistribute it and/or modify it
   under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   VAUL is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General
   Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with VAUL; see the file COPYING.LIB.  If not, write
   to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307 USA.


*/

#include <freehdl/vaul-parser.h>
#include <freehdl/vaul-chunk.h>
#include <freehdl/vaul-dunit.h>
#include <freehdl/vaul-util.h>

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#define psr vaul_parser

struct resolution_filter_closure {
  pIIR_Type base;
};

static int
resolution_filter (pIIR_Declaration d, void *closure)
{
  resolution_filter_closure *rfc = (resolution_filter_closure *)closure;

  if (!d->is(IR_FUNCTION_DECLARATION))
    return -1;

  pIIR_FunctionDeclaration f = pIIR_FunctionDeclaration(d);
  if (!f->pure)
    return -1;
  if (get_base (f->return_type) != rfc->base)
    return -1;
  if (f->interface_declarations == NULL
      || f->interface_declarations->rest != NULL)
    return -1;

  pIIR_InterfaceDeclaration p = f->interface_declarations->first;
  if (!p->is(IR_CONSTANT_INTERFACE_DECLARATION) || p->mode != IR_IN_MODE)
    return -1;
  if (!p->subtype->is(IR_ARRAY_TYPE))
    return -1;

  pIIR_ArrayType pt = pIIR_ArrayType (p->subtype);
  if (pt->index_types == NULL || pt->index_types->rest != NULL)
    return -1;
  if (get_base (pt->element_type) != rfc->base)
    return -1;
  
  return 0;
}

pIIR_FunctionDeclaration
psr::find_resolution_function (pVAUL_Name res_name, pIIR_Type type)
{
  pIIR_FunctionDeclaration res_func = NULL;

  if (res_name)
    {	
      vaul_decl_set ds(this);
      find_decls (ds, res_name);
      resolution_filter_closure rfc = { type->base };
      ds.filter (resolution_filter, &rfc);
      ds.invalidate_pot_invalids ();
      res_func = pIIR_FunctionDeclaration (ds.single_decl (false));
      if (res_func == NULL)
	{
	  error ("%:no match for resolution function %n",
		 res_name, res_name);
	  ds.show(false);
	}
      assert (!res_func || res_func->is(IR_FUNCTION_DECLARATION));
    }

  return res_func;
}

pIIR_Type
psr::build_ArraySubtype (pVAUL_Name res_name,
			 pVAUL_Name type_mark,
			 pIIR_TypeList constraint)
{
  pIIR_Type base = get_type (type_mark);
  if (base == NULL)
    return NULL;
  
  pIIR_FunctionDeclaration res_func =
    find_resolution_function (res_name, base);
  
  if(constraint) 
    constraint = build_IndexConstraint (constraint, base);

  // XXX - do some checks here, like constraints on
  // access/record/file types
  
  if (constraint == NULL && res_func == NULL)
    return base;
  return mIIR_ArraySubtype (type_mark->pos, base->base, base, res_func,
			    constraint);
}

pIIR_Type
psr::build_ScalarSubtype (pVAUL_Name res_name,
			  pVAUL_Name type_mark,
			  pIIR_Range range)
{
  pIIR_Type base = get_type (type_mark);
  if (base == NULL)
    return NULL;

  pIIR_FunctionDeclaration res_func =
    find_resolution_function (res_name, base);

  if (range) 
    {
      if (range->is(IR_EXPLICIT_RANGE))
	{
	  pIIR_ExplicitRange r = pIIR_ExplicitRange(range);
	  overload_resolution(r->left, base);
	  overload_resolution(r->right, base);
	}
      else if (range->is(IR_ARRAY_RANGE))
	{
	  pIIR_Type t = pIIR_ArrayRange(range)->type;
	  if (t->base != base)
	    error ("%:%n is not a base type of %n", range, t, base);
	}
      else
	assert (false);
    }

  // XXX - do some checks here, like constraints on
  // access/record/file types

  if (range == NULL && res_func == NULL)
    return base;
  return mIIR_ScalarSubtype (type_mark->pos, base->base, base,
			     res_func, range);
}

pIIR_TypeList
psr::build_IndexConstraint (pIIR_TypeList pre,
			    pIIR_Type immediate_base)
{
  if (immediate_base->is(IR_ARRAY_SUBTYPE))
    {
      error ("%:a constrained array can't be further constrained", pre);
      return NULL;
    }
  else if (!immediate_base->is(IR_ARRAY_TYPE))
    {
      error ("%:only array types can have index constraints", pre);
      return NULL;
    }

  pIIR_TypeList itypes = pIIR_ArrayType(immediate_base)->index_types;
  pIIR_TypeList pcons = pre;
  pIIR_TypeList cons = NULL, *ctail = &cons;

  while (pcons && itypes)
    {
      assert (pcons->first->is(VAUL_PRE_INDEX_CONSTRAINT));
      pVAUL_PreIndexConstraint pic = pVAUL_PreIndexConstraint(pcons->first);
      if (itypes->first == NULL)
	return NULL;

      pIIR_Type type = NULL;

      if (pic->is(VAUL_PRE_INDEX_RANGE_CONSTRAINT))
	{
	  if (pIIR_Range r = pVAUL_PreIndexRangeConstraint(pic)->range) 
	    {
	      if (r->is(IR_EXPLICIT_RANGE))
		{
		  pIIR_ExplicitRange er = pIIR_ExplicitRange(r);
		  if (pIIR_Type itype = find_index_range_type (er)) 
		    {
		      overload_resolution(er->left, itype);
		      overload_resolution(er->right, itype);
		    }
		}
	      type = mIIR_ScalarSubtype (pic->pos, 
					 itypes->first->base,
					 itypes->first,
					 NULL,
					 r);
	    }
	}
      else if (pic->is(VAUL_PRE_INDEX_SUBTYPE_CONSTRAINT))
	type = pVAUL_PreIndexSubtypeConstraint(pic)->type;
      else
	vaul_fatal ("build_IndexConstraint confused.\n");

      if (type && itypes->first && type->base != itypes->first->base)
	error ("%:constraint type (%n) does not match index type (%n)",
	       pre, type->base, itypes->first->base);

      *ctail = mIIR_TypeList (pcons->pos, type, *ctail);
      ctail = &(*ctail)->rest;
      itypes = itypes->rest;
      pcons = pcons->rest;
    }

  if (pcons)
    error ("%:too many index constraints for %n", pre, immediate_base);
  else if (itypes)
    error ("%:too few index constraints for %n", pre, immediate_base);
  return cons;
}

pIIR_Type
psr::find_index_range_type (pIIR_ExplicitRange r)
{
  pIIR_Type_vector &left_types = *ambg_expr_types(r->left);
  pIIR_Type_vector &right_types = *ambg_expr_types(r->right);

  if (left_types.size() == 0 || right_types.size() == 0)
    return NULL;

  pIIR_Type_vector types;
  
  for (int i = 0; i < left_types.size(); i++) 
    {
      assert(left_types[i]);
      pIIR_Type t = left_types[i];
      if (!is_discrete_type(t->base))
	{
	  info ("%:%n is not discrete", t->base, t->base);
	  continue;
	}
      for (int j = 0; j < right_types.size(); j++) 
	{
	  assert (right_types[j]);
	  pIIR_Type tt = right_types[j];
	  if(!is_discrete_type(get_base(tt)))
	    {
	      info ("%:%n is not discrete", t->base, t->base);
	      continue;
	    }
	  if(t == std->universal_integer)
	    t = tt;
	  else if(tt == std->universal_integer)
	    tt = t;
	  if(get_base(t) == get_base(tt)) {
	    if(t == std->universal_integer)
	      t = std->predef_INTEGER;
	    if(try_overload_resolution(r->left, t, IR_INVALID)
	       && try_overload_resolution(r->right, t, IR_INVALID)) {
	      bool already_inserted = false;
	      for(int k = 0; k < types.size(); k++)
		if(get_base(types[k]) == get_base(t)) {
		  // info("+++ - found %n twice", t);
		  already_inserted = true;
		  break;
		}
	      if(!already_inserted)
		types.add(t);
	    }
	    else
	      info ("%:can't coerce %n,%n to %n", t, r->left, r->right, t);
	  }
	}
    }

  if(types.size() == 0) {
    error("%:index bounds must be discrete and of the same type", r);
    if(left_types.size() > 0) {
      info("left bound could be:"); 
      for(int i = 0; i < left_types.size(); i++)
	info("%:   %n", left_types[i], left_types[i]);
    } else
      info("no left types");
    if(right_types.size() > 0) {
      info("right bound could be:"); 
      for(int i = 0; i < right_types.size(); i++)
	info("%:   %n", right_types[i], right_types[i]);
    } else
      info("no right types");
  } else if(types.size() != 1) {
    error("%:type of index bounds is ambigous, it could be:", r);
    for(int i = 0; i < types.size(); i++)
      info("%:   %n (%s)", types[i], types[i], types[i]->kind_name());
  }
  
  delete &left_types;
  delete &right_types;

  return types.size() == 1? types[0] : NULL;
}


pIIR_ScalarSubtype
psr::build_SubType_def (int pos, pIIR_Range r, pIIR_Type base)
{
  if (r == NULL)
    return NULL;

  if (base == NULL) 
    {
      if (r->is(IR_EXPLICIT_RANGE)) 
	{
	  pIIR_ExplicitRange er = pIIR_ExplicitRange(r);
	  if (try_overload_resolution (er->left, NULL,
				       IR_INTEGER_TYPE)
	      && try_overload_resolution (er->right, NULL,
					  IR_INTEGER_TYPE))
	    {
	      base = mIIR_IntegerType (pos);
	    }
	  else if (try_overload_resolution (er->left, NULL,
					    IR_FLOATING_TYPE)
		   && try_overload_resolution (er->right, NULL,
					       IR_FLOATING_TYPE))
	    {
	      base = mIIR_FloatingType (pos);
	    }
	  else 
	    {
	      error ("%!range bounds must be both either integer"
		     " or real values", lex, pos);
#if 0
	      info("%!they can be:", lex, pos);
	      pIIR_Type_vector &types = *ambg_expr_types(er->first);
	      for (int i = 0; i < types.size(); i++)
		info ("%:  %n", types[i], types[i]);
	      pIIR_Type_vector &types2 = *ambg_expr_types(er->last);
	      info ("and");
	      for (int i = 0; i < types2.size(); i++)
		info ("%:  %n", types2[i], types2[i]);
#endif
	      return NULL;
	    }
	} 
      else if (r->is(IR_ARRAY_RANGE))
	{
	  info ("XXX - no array ranges in type definition");
	  return NULL;
	}
      else
	assert (false);
    }

  IR_Kind base_k = base->kind();
  if (base_k == IR_PHYSICAL_TYPE)
    base_k = IR_INTEGER_TYPE;
  assert (r->is(IR_EXPLICIT_RANGE)); // for now...
  overload_resolution (pIIR_ExplicitRange(r)->left, base_k);
  overload_resolution (pIIR_ExplicitRange(r)->right, base_k);

  return mIIR_ScalarSubtype (pos, base->base, base, NULL, r);
}

pIIR_Type
psr::get_type (pVAUL_Name mark)
{
  pIIR_TypeDeclaration d = 
    pIIR_TypeDeclaration(find_single_decl (mark, IR_TYPE_DECLARATION, "type"));
  if (d && d->type)
    {
      if (d->type->is(VAUL_INCOMPLETE_TYPE))
	error ("%:type %n is incomplete", mark, mark);
      else
	return d->type;
    }
  return NULL;
}

void 
psr::add_PredefOp (pIIR_PosInfo pos, pIIR_Type ret, 
		   pIIR_TextLiteral sym, pIIR_Type left,
		   pIIR_Type right)
{
  pIIR_InterfaceList interf = 
    mIIR_InterfaceList (pos,
			mIIR_ConstantInterfaceDeclaration (pos, NULL,
							   left, NULL,
							   IR_IN_MODE,
							   false),
			NULL);
  if (right)
    interf->rest =
      mIIR_InterfaceList (pos,
			  mIIR_ConstantInterfaceDeclaration (pos, NULL,
							     right, NULL,
							     IR_IN_MODE,
							     false),
			  NULL);
    
  add_decl (mVAUL_PredefOp (pos, sym, interf, true, ret));
}

pIIR_Type
psr::is_one_dim_array (pIIR_Type t)
{
  if (!t->is(IR_ARRAY_TYPE))
    return NULL;
  pIIR_ArrayType at = pIIR_ArrayType(t);
  if (!at->index_types || at->index_types->rest)
    return NULL;
  return at->element_type;
}

bool
psr::is_one_dim_logical_array (pIIR_Type t)
{
  pIIR_Type et = is_one_dim_array (t);
  return et && et == std->predef_BIT || et == std->predef_BOOLEAN;
}

bool
psr::is_one_dim_discrete_array (pIIR_Type t)
{
    pIIR_Type et = is_one_dim_array (t);
    return is_discrete_type (et);
}

bool
psr::is_discrete_type (pIIR_Type t)
{
  if (t == NULL)
    return false;
  t = t->base;
  return t && (t->is(IR_INTEGER_TYPE)
	       || t->is(IR_ENUMERATION_TYPE));
}

void
psr::add_predefined_ops (pIIR_Type t)
{
  if (t == NULL)
    return;

  pIIR_Type bt = t;

  if (t->is(IR_SUBTYPE)
      && t->declaration == pIIR_Subtype(t)->immediate_base->declaration)
    bt = pIIR_Subtype(t)->immediate_base;
  if (bt->is(VAUL_INCOMPLETE_TYPE) || bt->is(IR_SUBTYPE))
    return;


# define add(r, op, t1, t2)  add_PredefOp(t->pos, r, make_strlit(#op), \
                                            t1, t2)

  // logical operators
  //
  if ((t == std->predef_BIT || t == std->predef_BOOLEAN)
      || is_one_dim_logical_array(bt)) 
    {
      add (t, "and", t, t);
      add (t, "or", t, t);
      add (t, "nand", t, t);
      add (t, "nor", t, t);
      add (t, "xor", t, t);
      add (t, "xnor", t, t);
      add (t, "not", t, NULL);
      if (bt == std->predef_BOOLEAN)
	{
	  pIIR_IntegerType ui = mIIR_IntegerType (t->pos);
	  std->universal_integer = ui;
	  add_predefined_ops(ui);
	  
	  pIIR_FloatingType ur = mIIR_FloatingType (t->pos);
	  std->universal_real = ur;
	  add_predefined_ops(ur);
	}
    }

  // relational operators
  //
  pIIR_Type b = std->predef_BOOLEAN;
  if (b == NULL) 
    {
      info ("%:can't predefine relational operators for %n", t, t);
      info ("%:since type BOOLEAN is undefined", t);
    } 
  else
    {
      add (b, "=", t, t);
      add (b, "/=", t, t);
      if (bt->is(IR_SCALAR_TYPE) || is_one_dim_discrete_array(bt))
	{
	  add (b, "<", t, t);
	  add (b, ">", t, t);
	  add (b, "<=", t, t);
	  add (b, ">=", t, t);
	}
    }

  // shift operators
  //
  if (is_one_dim_logical_array(bt)) 
    {
      if (pIIR_Type r = std->predef_INTEGER) 
	{
	  add (t, "sll", t, r);
	  add (t, "srl", t, r);
	  add (t, "sla", t, r);
	  add (t, "sra", t, r);
	  add (t, "rol", t, r);
	  add (t, "ror", t, r);
	} 
      else
	{
	  info ("%:can't predefine shift operators for %n", t, t);
	  info ("%:since type INTEGER is undefined", t);
	}
    }

  // Adding, Sign and Miscellanous operators
  //
  if (bt->is(IR_INTEGER_TYPE)
      || bt->is(IR_FLOATING_TYPE)
      || bt->is(IR_PHYSICAL_TYPE))
    {
      add (t, "+", t, t);
      add (t, "-", t, t);
      add (t, "abs", t, NULL);
      add (t, "+", t, NULL);
      add (t, "-", t, NULL);
    }

  // concatenation operator
  //
  if(pIIR_Type et = is_one_dim_array(bt)) 
    {
      add (t, "&", t, t);
      add (t, "&", t, et);
      add (t, "&", et, t);
      add (t, "&", et, et);
    }

  // Multiplying operators
  //
  if (bt->is(IR_INTEGER_TYPE)
      || bt->is(IR_FLOATING_TYPE))
    {
      add (t, "*", t, t);
      add (t, "/", t, t);
      if (bt->is(IR_INTEGER_TYPE))
	{
	  add (t, "mod", t, t);
	  add (t, "rem", t, t);
	}
      if (std->predef_INTEGER)
	add (t, "**", t, std->predef_INTEGER);
      else if (bt != std->universal_integer && bt != std->universal_real) 
	{
	  info ("%:can't predefine \"**\" operator for %n", t, t);
	  info ("%:since type INTEGER is undefined", t);
	}
    }
  
  if (bt->is(IR_PHYSICAL_TYPE))
    {
      if (std->predef_INTEGER && std->predef_REAL) 
	{
	  add (t, "*", t, std->predef_INTEGER);
	  add (t, "*", std->predef_INTEGER, t);
	  add (t, "/", t, std->predef_INTEGER);
	  add (t, "*", t, std->predef_REAL);
	  add (t, "*", std->predef_REAL, t);
	  add (t, "/", t, std->predef_REAL);
	}
      else 
	{
	  info ("%:can't predefine multiplying operators for %n", t, t);
	  info ("%:since types INTEGER and REAL are undefined", t);
	}
      if (std->universal_integer)
	add (std->universal_integer, "/", t, t);
    }

  if (t == std->predef_INTEGER) 
    {
      add (std->universal_integer, "**", std->universal_integer, t);
      add (std->universal_real, "**", std->universal_real, t);
    }

  if (t->is(IR_ACCESS_TYPE)) 
    {
      pIIR_InterfaceList parm =
	mIIR_InterfaceList (t->pos, 
			    mIIR_VariableInterfaceDeclaration (t->pos, NULL,
							       t, NULL,
							       IR_INOUT_MODE,
							       false),
			    NULL);
      pIIR_ProcedureDeclaration dealloc =
	mIIR_ProcedureDeclaration (t->pos, make_id ("deallocate"), parm);
      add_decl(dealloc);
    }
    
  if (t->is(IR_FILE_TYPE))
    {
      pIIR_FileType ft = pIIR_FileType(t);
      
      pIIR_InterfaceList parm =
	mIIR_InterfaceList (t->pos,
			    mIIR_FileInterfaceDeclaration (t->pos, NULL,
							   ft, NULL,
							   IR_UNKNOWN_MODE,
							   false),
			    NULL);
      pIIR_FunctionDeclaration endfile =
	mIIR_FunctionDeclaration (t->pos, make_id ("endfile"), parm,
				  false, std->predef_BOOLEAN);
      add_decl(endfile);
      
      parm =
	mIIR_InterfaceList (t->pos,
			    mIIR_VariableInterfaceDeclaration (t->pos, NULL,
							       ft->type_mark,
							       NULL,
							       IR_OUT_MODE,
							       false),
			    NULL);
      parm =
	mIIR_InterfaceList (t->pos,
			    mIIR_FileInterfaceDeclaration (t->pos, NULL,
							   ft, NULL,
							   IR_UNKNOWN_MODE,
							   false),
			    parm);
      
      pIIR_ProcedureDeclaration read =
	mIIR_ProcedureDeclaration (t->pos, make_id ("read"), parm);
      add_decl(read);
      
      parm =
	mIIR_InterfaceList (t->pos,
			    mIIR_VariableInterfaceDeclaration (t->pos, NULL,
							       ft->type_mark,
							       NULL,
							       IR_IN_MODE,
							       false),
			    NULL);
      parm =
	mIIR_InterfaceList (t->pos,
			    mIIR_FileInterfaceDeclaration (t->pos, NULL,
							   ft, NULL,
							   IR_UNKNOWN_MODE,
							   false),
			    parm);
      pIIR_ProcedureDeclaration write =
	mIIR_ProcedureDeclaration (t->pos, make_id ("write"), parm);
      add_decl(write);
    }
  
#   undef add
}

void
add_use( pVAUL_IncompleteType it, pIIR_Type &ref)
{
  assert (ref == it);
  vaul_incomplete_type_use *u = new vaul_incomplete_type_use;
  u->next = it->uses;
  it->uses = u;
  u->ref = &ref;
}

void
complete (pVAUL_IncompleteType it, pIIR_Type t)
{
  for (vaul_incomplete_type_use *u = it->uses; u; u = u->next) 
    {
      assert(*u->ref == it);
      *u->ref = t;
    }
}

pIIR_TypeList
psr::build_PreIndexConstraint (pVAUL_GenAssocElem a)
{
    pIIR_TypeList ic = NULL, *ict = &ic;

    while (a)
      {
	pIIR_Type type = NULL;

	if (a->is(VAUL_NAMED_ASSOC_ELEM))
	  {
	    pVAUL_NamedAssocElem nae = pVAUL_NamedAssocElem(a);
	    if (nae->formal)
	      error ("%:index constraints can't use named association", nae);
	    if (nae->actual && nae->actual->is(VAUL_UNRESOLVED_NAME))
	      {
		pVAUL_Name n = pVAUL_UnresolvedName(nae->actual)->name;
		pIIR_Type st = get_type (n);
		if (is_discrete_type(st))
		  type = mVAUL_PreIndexSubtypeConstraint(a->pos, st);
		else if (st)
		  error ("%: %n is not a discrete type", n, st);
	      }
	  }
	else if (a->is(VAUL_RANGE_ASSOC_ELEM))
	  {
	    type = mVAUL_PreIndexRangeConstraint(a->pos,
					    pVAUL_RangeAssocElem(a)->range);
	  }
	else if (a->is(VAUL_SUBTYPE_ASSOC_ELEM)) 
	  {
	    type = mVAUL_PreIndexSubtypeConstraint(a->pos,
					      pVAUL_SubtypeAssocElem(a)->type);
	  }

	if (type)
	  {
	    *ict = mIIR_TypeList (a->pos, type, *ict);
	    ict = &(*ict)->rest;
	  }

	a = a->next;
      }

    return ic;
}

pIIR_Type
psr::adapt_object_type (VAUL_ObjectClass c,
			pIIR_Type t,
			pIIR_Expression init)
{
  if (t == NULL)
    return NULL;
  
  if (c == VAUL_ObjClass_Variable || c == VAUL_ObjClass_Signal) 
    {
      if (t->is(IR_ARRAY_TYPE))
	error("array objects must have a constrained type");
      return t;
    }

  if (c == VAUL_ObjClass_Constant) 
    {
      if (init == NULL || !t->is(IR_ARRAY_TYPE))
	return t;

      info ("xxx - dreaming up index bounds of array constant");
      pIIR_ArrayType at = pIIR_ArrayType(t);
      pIIR_TypeList cons = NULL, *ctail = &cons;
      for (pIIR_TypeList it = at->index_types; it; it = it->rest)
	{
	  if (it->first == NULL)
	    return NULL;

	  pIIR_PosInfo p = init->pos;
	  pIIR_IntegerLiteral low =
	    mIIR_IntegerLiteral (p, (IR_Character *)"0", 1);
	  pIIR_IntegerLiteral high =
	    mIIR_IntegerLiteral (p, (IR_Character *)"10", 2);
	  
	  pIIR_ExplicitRange er =
	    mIIR_ExplicitRange (p, 
				build_LiteralExpression (p, low),
				build_LiteralExpression (p, high),
				IR_DIRECTION_UP);
	  pIIR_ScalarSubtype st =
	    mIIR_ScalarSubtype (p,
				it->first->base,
				it->first, NULL, er);
	  *ctail = mIIR_TypeList (p, st, NULL);
	  ctail = &(*ctail)->rest;
	}
      return mIIR_ArraySubtype (init->pos,
				t->base,
				t, NULL, cons);
    }

  info ("xxx - unchecked object type");
  return t;
}

pIIR_ArraySubtype
psr::build_constrained_array_type (pIIR_TypeList pre,
				   pIIR_Type elt)
{
  pIIR_TypeList itypes = NULL, *itail = &itypes;
  
  for (pIIR_TypeList p = pre; p; p = p->rest)
    {
      pIIR_Type it, pt = p->first;
      
      if (pt->is(VAUL_PRE_INDEX_SUBTYPE_CONSTRAINT))
	it = pVAUL_PreIndexSubtypeConstraint(pt)->type;
      else if (pt->is(VAUL_PRE_INDEX_RANGE_CONSTRAINT)) 
	{
	  pIIR_Range r = pVAUL_PreIndexRangeConstraint(pt)->range;
	  if (r == NULL)
	    return NULL;
	  if (r->is(IR_EXPLICIT_RANGE))
	    it = find_index_range_type (pIIR_ExplicitRange(r));
	  else if (r->is(IR_ARRAY_RANGE))
	    it = pIIR_ArrayRange(r)->type;
	  else
	    assert (false);
	}
      else
	assert(false);

      *itail = mIIR_TypeList(p->pos, it, NULL);
      itail = &(*itail)->rest;
    }

  pIIR_ArrayType base = mIIR_ArrayType(pre? pre->pos:NULL, itypes, elt);
  return mIIR_ArraySubtype (base->pos, base,
			    base, NULL, build_IndexConstraint (pre, base));
}

pIIR_FileDeclaration
psr::add_File (pIIR_Identifier id,
	       pIIR_Type file_type,
	       pIIR_Expression mode,
	       pIIR_Expression name)
{
  if (!file_type->is(IR_FILE_TYPE))
    {
      error ("%:%n is not a file type", id, file_type);
      return NULL;
    }
  return pIIR_FileDeclaration (
     add_decl(mIIR_FileDeclaration (id->pos, id, file_type,
				    NULL, mode, name)));
}

pIIR_Type
m_get_base (pIIR_Type t)
{
  return t->base;
}
